function completed = example_linearSysDT_observe_04_pedestrian
% example_linearSysDT_observe_04_pedestrian - example for guaranteed state
%    estimation of linear discrete-time systems from a unit test; provides
%    the solution of the linearSysDT class for a pedestrian example
%
% Syntax:
%    completed = example_linearSysDT_observe_04_pedestrian
%
% Inputs:
%    -
%
% Outputs:
%    completed - true/false

% Authors:       Matthias Althoff
% Written:       03-March-2021
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------


% load pedestrian model
% load pedestrianModel pedestrian params options simRes
[pedestrian,params,options,simRes] = aux_load_model();

% Set of evaluated estimators
Estimator = {
    'VolMin-B' 
    'FRad-A' 
    'FRad-B' 
    'PRad-A' 
    'PRad-B' 
    'PRad-C' 
    'PRad-D' 
    'ESO-A'
    'ESO-B'
    'FRad-C' 
    'PRad-E' 
    'Nom-G' 
    'ESO-D'
    'Hinf-G' 
    };

% set solver
options.solver = 'sdpt3';
% ensure that solver is on path
if ~isSolverInstalled('sdpt3')
    throw(CORAerror('CORA:noSuitableSolver','sdpt3'));
end


%% perform evaluation
% loop over estimators
for iEst = 1:length(Estimator)

    % set algorithm
    estName = Estimator{iEst};
    options.alg = estName;
    disp("Running observer '" + estName + "'");
    
    % if constrained zonotopes or ellipsoids are used
    paramsNew = params;
    if any(strcmp(estName,{'ESO-A','ESO-B','ESO-C','ESO-D','ESO-E'}))
        paramsNew.R0 = ellipsoid(params.R0);
        paramsNew.W = ellipsoid(params.W);
        paramsNew.V = ellipsoid(params.V);
    elseif any(strcmp(estName,{'CZN-A','CZN-B'}))
        paramsNew.R0 = conZonotope(params.R0);
    end
    
    % evaluate observer
    estSet = observe(pedestrian,paramsNew,options);

    % plot results
    if ~isempty(estSet)
        for iDim = 1:4
            figure; hold on;
            % plot time elapse
            plotOverTime(estSet,iDim);
            % plot simulation
            plotOverTime(simRes,iDim);

            % label plot
            xlabel('t');
            ylabel(['x_{',num2str(iDim),'}']);
        end
    end
end

% example completed
completed = true;

end


% Auxiliary functions -----------------------------------------------------

function [pedestrian,params,options,simRes] = aux_load_model()

% generated semi-automatically using printSystem, printSet, etc. 
% from saved .mat file

% build system
name = 'pedestrian';
A = [ 1.0000000000000000 0.0000000000000000 0.0100000000000000 0.0000000000000000 ; 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0100000000000000 ; 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 ];
B = [ 0.0000500000000000 0.0000000000000000 ; 0.0000000000000000 0.0000500000000000 ; 0.0100000000000000 0.0000000000000000 ; 0.0000000000000000 0.0100000000000000 ];
c = [ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ];
C = [ 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 ];
D = [ 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 ];
k = [ 0.0000000000000000 ; 0.0000000000000000 ];
E = [ 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 ];
F = [ 1.0000000000000000 0.0000000000000000 ; 0.0000000000000000 1.0000000000000000 ];
dt = 0.0100000000000000;
pedestrian = linearSysDT(name,A,B,c,C,D,k,E,F,dt);

% build params
params = struct( ...
    'tStart', 0, ...
    'tFinal', 2.3999999999999999, ...
    'R0', zonotope([ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ], [ 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 ]), ...
    'V', zonotope([ 0.0000000000000000 ; 0.0000000000000000 ], [ 0.1000000000000000 0.0000000000000000 ; 0.0000000000000000 0.1000000000000000 ]), ...
    'W', zonotope([ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ], [ 0.0000077254248594 0.0000047745751406 0.0000000000000000 -0.0000047745751406 -0.0000077254248594 -0.0000077254248594 -0.0000047745751406 -0.0000000000000000 0.0000047745751406 0.0000077254248594 ; 0.0000025101426986 0.0000065716389015 0.0000081229924058 0.0000065716389015 0.0000025101426986 -0.0000025101426986 -0.0000065716389015 -0.0000081229924058 -0.0000065716389015 -0.0000025101426986 ; 0.0015450849718747 0.0009549150281253 0.0000000000000000 -0.0009549150281253 -0.0015450849718747 -0.0015450849718747 -0.0009549150281253 -0.0000000000000000 0.0009549150281253 0.0015450849718747 ; 0.0005020285397156 0.0013143277802978 0.0016245984811645 0.0013143277802978 0.0005020285397156 -0.0005020285397156 -0.0013143277802978 -0.0016245984811645 -0.0013143277802978 -0.0005020285397156 ]), ...
    'y', [ ...
 0.2980692070302663, 0.3300786539770822, 0.3852415991930514, 0.3468532331979846, 0.4124961023538538, 0.3608290059438841, 0.4122803406904172, 0.3369973566834393, 0.3849216172308330, 0.3106346704787282, 0.4129591977589715, 0.3946541167702686, 0.3194988557264359, 0.3214377427831501, 0.3555335658813643, 0.3557082977365341, 0.3873035901386656, 0.4122241139149905, 0.3504321402159952, 0.3835658868378040, 0.3953625855748790, 0.3429760372056850, 0.3903573275538083, 0.4243130316842262, 0.3366030933549326, 0.3621680813858036, 0.3575713322893078, 0.3998277038762789, 0.3941650704136003, 0.3524146926163776, 0.3873245223335970, 0.4141381942044291, 0.3611124306494522, 0.4284947945146983, 0.3846339157870800, 0.4301324957090156, 0.3900354510134117, 0.4439653755650746, 0.3836923184179283, 0.3805984301402471, 0.4795458424113690, 0.4080277588314123, 0.3682437366633650, 0.3938116819988856, 0.4613766437299991, 0.4902474476011394, 0.4073293493294644, 0.4052326614310435, 0.5392049714842890, 0.4445344352904539, 0.4634431513382498, 0.4651316662308653, 0.4148199542393058, 0.4238336994863398, 0.5005878519581729, 0.4610925084784989, 0.4596658715356970, 0.5364965095353584, 0.4719676965800700, 0.4198799317955708, 0.4306331216795034, 0.4790960429330037, 0.5114114455225973, 0.5277833546773981, 0.4994144295062131, 0.5331262232223616, 0.5386520096589222, 0.4803083832864816, 0.5244739197524737, 0.4766862259627517, 0.5327780931481038, 0.4641054546830850, 0.4934828545109127, 0.5733789708682276, 0.5432600748407204, 0.5542415321831414, 0.5182435122098329, 0.5627272990344849, 0.5875239211348792, 0.5757909808437883, 0.5717958049432063, 0.6560402499247381, 0.5943085993503119, 0.5863009555319614, 0.5619377846568219, 0.6466190630318142, 0.5998581885730172, 0.6478572590111156, 0.5770118370277825, 0.6015058241035617, 0.5959449267864613, 0.5982748731555726, 0.5512252068807332, 0.5885431373687601, 0.6049015340326043, 0.6648013969071906, 0.6959435524376093, 0.6187093863546892, 0.7154277856240231, 0.7178594426536583, 0.6922595265537183, 0.7106787200276647, 0.6583398206581440, 0.6830202365641699, 0.6728157057482194, 0.7286727183882604, 0.6609985733979106, 0.7482367699299222, 0.6959371774670257, 0.7387302534095823, 0.6784947761250433, 0.7238355476667976, 0.7022156749792690, 0.6717850394324705, 0.7379531176501324, 0.7417162044992114, 0.7339915778269920, 0.7669646848181354, 0.7495146008969249, 0.7748342591422027, 0.7007444847692558, 0.7376832287684785, 0.7872714597090090, 0.7140947823068472, 0.8318859603886829, 0.7826887194776205, 0.8162131366353726, 0.7230814738620773, 0.7467668693065678, 0.7936020888660293, 0.7836022764040147, 0.7994040770361363, 0.7886931003925416, 0.7427522069827877, 0.8070332309217572, 0.8135899918789979, 0.8206182696296445, 0.8128036455059640, 0.7762680627812879, 0.8617821758066224, 0.8928859177493559, 0.8729543637195062, 0.8544519749023575, 0.8888567902808368, 0.9170205609566533, 0.8820895998420801, 0.8724692349012581, 0.8215500693486950, 0.8391046445667777, 0.9063407886942545, 0.9054242562498060, 0.8798673867823070, 0.8690703967592737, 0.8743837787470610, 0.9816866224347279, 0.8790826944975496, 0.9180404603794063, 0.9199461814382158, 0.8697784200030251, 0.8587230529036759, 0.9792256735213942, 0.8882572874227957, 0.9715213500916106, 0.9651712298421316, 0.9478055489877331, 0.9097171553115748, 0.8936533638752947, 0.9666854716531021, 0.9330002526745145, 0.9884626788331938, 1.0228894820046586, 0.9485476781942068, 1.0302011973794047, 1.0296936956371703, 0.9657698198359044, 1.0378137257101037, 1.0471090555795159, 1.0088719357466398, 1.0414526516301230, 1.0493438400918420, 0.9642872462013310, 1.0256876097160688, 1.0028354252967639, 1.0217464919312913, 1.0028640699206273, 1.0610706696731935, 1.0321742124400786, 1.0175010376938103, 0.9965305425705364, 1.0083505171767158, 1.0215212893502410, 1.0822360797488948, 1.1268662149203648, 1.0158263908508776, 1.0822084608529141, 1.1068258882676216, 1.0821445403418841, 1.0528932816693357, 1.0016399971234609, 1.0677764494841926, 1.0452451078330611, 1.1235799778177511, 1.0631345294041181, 1.0799685768109872, 1.0793676901529781, 1.1082814764589595, 1.1552806117895225, 1.1827383559729403, 1.0722267153643852, 1.1016595150890274, 1.1041343418460763, 1.1045977628009438, 1.1530370679758200, 1.1102973528918147, 1.1596012822469557, 1.2183048335161051, 1.1016190486102420, 1.1557053298229774, 1.1091038625866578, 1.1484481822484809, 1.1145531983165973, 1.1569555850529603, 1.1754490575646332, 1.1560405280002490, 1.1857172468993400, 1.1617206141642908, 1.1800500765668391, 1.1466206422844896, 1.1384071020000075, 1.2239903481675469, 1.1347734531678468, 1.2365434528498838, 1.1846889672330589, 1.1819892139425627, 1.1670100746820387, 1.1380574571879967, 1.1254942482197037, 1.2099888384548525, 1.2122776841422731, 1.1995099943798784 ; ...
 0.2910422612439254, 0.3126155502134567, 0.2665465407549422, 0.3645742995409444, 0.4286993897145474, 0.3284002254021727, 0.3372118331369387, 0.3503106537123249, 0.3977028839355408, 0.3631408974757307, 0.4017305340942321, 0.4227194691302033, 0.4084039973502614, 0.3666832960259307, 0.4066299608704425, 0.3830465122660464, 0.3687871199720305, 0.3662369608657081, 0.3841159011156055, 0.3841781659128511, 0.3940609180882924, 0.4489729217740106, 0.4632496980159652, 0.4118209574786592, 0.4886796706272569, 0.4677974956503658, 0.4954069361596016, 0.4633746335235782, 0.4287591379756085, 0.4883903588720786, 0.4821910884743705, 0.5031349782895103, 0.4830275443980326, 0.5247258362049261, 0.5154335297357997, 0.5088734814939709, 0.5687054982572376, 0.5027303926396703, 0.5228619666726789, 0.5196125891379202, 0.4881353892078869, 0.5259995084506477, 0.5927872074728661, 0.5386476966486194, 0.5240349513798099, 0.4908610726952088, 0.5446247061426656, 0.5301002659190028, 0.6021684266690825, 0.5084417759492105, 0.6261444115898417, 0.6119897522924341, 0.6666781168239601, 0.5857524904208519, 0.6273121950694129, 0.6254368465924810, 0.7008563598326834, 0.6377604803341818, 0.6414865121031635, 0.5594389464774855, 0.6519501406783152, 0.6299822645003761, 0.6981762503655045, 0.6127363786602982, 0.6687148797161742, 0.6964402398582910, 0.6743572569807759, 0.7137874329255188, 0.7656552880424612, 0.6683451219112231, 0.7620637652353005, 0.6370826485685590, 0.6839000143154140, 0.7838304608357208, 0.7989806895179622, 0.7715541988718611, 0.7633787273029214, 0.7878015488443733, 0.7527735709485971, 0.8368718372091413, 0.8051589944962358, 0.8351379785980781, 0.7888940983173754, 0.8077265093340955, 0.8596206568919372, 0.8259910252779317, 0.8515361180591093, 0.9052026297434939, 0.9131895266468295, 0.8869148719626387, 0.8558560433032034, 0.9334861965678372, 0.8961776004979952, 0.9089042724235646, 0.8983731651865257, 0.9372041558082422, 0.9947125361312151, 0.9923095798534547, 0.9621682650525050, 0.9427472789535203, 0.9811731815078646, 0.9798641614557161, 1.0131277481496246, 1.0282503053901006, 1.0119186986713686, 1.0247857759910934, 1.0803473246787036, 1.0584758235792420, 1.0521005663180671, 1.1454642960881514, 1.1254189121121911, 1.0733735270113220, 1.0904497124225183, 1.1133933587142704, 1.1703204429923080, 1.1164251393962628, 1.1242766469064389, 1.1368581394274500, 1.0983585651162220, 1.2156911395838477, 1.1352532564489264, 1.1051091397728532, 1.1504159837399173, 1.1946113261834437, 1.2355461871512972, 1.1611912452649022, 1.2260996896323453, 1.2135545510320578, 1.3261421522177388, 1.2756054623428033, 1.2769424619131438, 1.3278360877042119, 1.3228296051357606, 1.2988476143081600, 1.2245780746105681, 1.3418518834220559, 1.2659837219436840, 1.3405288210789097, 1.3436396932640309, 1.3015390307780852, 1.3964301032946587, 1.4011340726748838, 1.3344014452669226, 1.4200168338794963, 1.3587977660114099, 1.3755867596932840, 1.3817004713845329, 1.4022539375185423, 1.4792272262783708, 1.4734888716549952, 1.4846479525967100, 1.4610943442098834, 1.5126275336917232, 1.5168629840647652, 1.4452189121422447, 1.5240059120676581, 1.4891879259419987, 1.4909888131328710, 1.4925505663574121, 1.5697224239476217, 1.4682404638101878, 1.5927666074517310, 1.5810874975573341, 1.5887607106933481, 1.5490528151346270, 1.5732983691547862, 1.6412046009015993, 1.6047489177760503, 1.5915912721729852, 1.6232520155509511, 1.6877239670829813, 1.6645099555747076, 1.6338566650742516, 1.6736571247803871, 1.7222606663197795, 1.6430145874205377, 1.7303757188309059, 1.7402097018998988, 1.7189247038410065, 1.6953318861662614, 1.6960051376474181, 1.7216551338030621, 1.7241940289850750, 1.7815967113446389, 1.7107004032265920, 1.7849537605157695, 1.7514804091086371, 1.8444345893811886, 1.8176677088833968, 1.8989160737627029, 1.7791616962044046, 1.7941958597521670, 1.8791745000412050, 1.8420368883628933, 1.8160676466814669, 1.9140136338582312, 1.8982108209047204, 1.9075025202042992, 1.8941336501062771, 1.9498900227530542, 1.8829643903503852, 1.9499218823027349, 1.9570634333517689, 1.9368903043940309, 1.9139966205472865, 2.0216068487950292, 1.9356516888869888, 1.9436289225067338, 2.0128214476234993, 2.0187318401261560, 1.9972441492510429, 2.0416765194986275, 2.0319379285753421, 2.0807914479167069, 2.0927761729565133, 2.0642792500688270, 2.0322974944191614, 2.1689492580130616, 1.9977914632462066, 2.1107025144565310, 2.1129947798294166, 2.1723016017081656, 2.1253986744491038, 2.0948269355610125, 2.0856580437335852, 2.1796664556805934, 2.1157563062784508, 2.1862110124028389, 2.1867608666747667, 2.1961910082923497, 2.1544179573486533, 2.2220527896272499, 2.1793492035601596, 2.1749051303114064, 2.2706324708061305, 2.2384828709408735, 2.2546937442487152, 2.2568697840281833, 2.2325959541353875, 2.2667925597538794 ; ...
 ], ...
    'u', [ ...
 0.0000000000000000, 0.0333333333333333, 0.0666666666666667, 0.1000000000000000, 0.1333333333333333, 0.1666666666666667, 0.2000000000000000, 0.2333333333333333, 0.2666666666666667, 0.3000000000000000, 0.3333333333333333, 0.3666666666666666, 0.4000000000000000, 0.4333333333333333, 0.4666666666666666, 0.4999999999999999, 0.5333333333333333, 0.5666666666666667, 0.6000000000000000, 0.6333333333333333, 0.6666666666666666, 0.7000000000000000, 0.7333333333333333, 0.7666666666666666, 0.7999999999999999, 0.8333333333333333, 0.8666666666666666, 0.8999999999999999, 0.9333333333333332, 0.9666666666666666, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9999999999999999, 0.9666666666666666, 0.9333333333333332, 0.8999999999999999, 0.8666666666666666, 0.8333333333333333, 0.7999999999999999, 0.7666666666666666, 0.7333333333333333, 0.7000000000000000, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6333333333333333, 0.6000000000000000, 0.5666666666666667, 0.5333333333333333, 0.5000000000000000, 0.4666666666666667, 0.4333333333333333, 0.4000000000000000, 0.3666666666666667, 0.3333333333333334, 0.3000000000000000, 0.2666666666666667, 0.2333333333333334, 0.2000000000000001, 0.1666666666666667, 0.1333333333333334, 0.1000000000000001, 0.0666666666666668, 0.0333333333333334, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, -0.0333333333333332, -0.0666666666666666, -0.0999999999999999, -0.1333333333333332, -0.1666666666666665, -0.1999999999999999, -0.2333333333333332, -0.2666666666666666, -0.2999999999999999, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3666666666666665, -0.3999999999999999, -0.4333333333333332, -0.4666666666666665, -0.4999999999999998, -0.5333333333333332, -0.5666666666666665, -0.5999999999999999, -0.6333333333333332, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6999999999999998, -0.7333333333333332, -0.7666666666666665, -0.7999999999999998, -0.8333333333333331, -0.8666666666666665, -0.8999999999999998, -0.9333333333333331, -0.9666666666666665, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998, -0.9999999999999998 ; ...
 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0333333333333333, 0.0666666666666667, 0.1000000000000000, 0.1333333333333333, 0.1666666666666667, 0.2000000000000000, 0.2333333333333333, 0.2666666666666667, 0.3000000000000000, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3333333333333333, 0.3666666666666666, 0.4000000000000000, 0.4333333333333333, 0.4666666666666666, 0.4999999999999999, 0.5333333333333333, 0.5666666666666667, 0.6000000000000000, 0.6333333333333333, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6666666666666666, 0.6333333333333333, 0.6000000000000000, 0.5666666666666667, 0.5333333333333333, 0.5000000000000000, 0.4666666666666667, 0.4333333333333333, 0.4000000000000000, 0.3666666666666667, 0.3333333333333334, 0.3000000000000000, 0.2666666666666667, 0.2333333333333334, 0.2000000000000001, 0.1666666666666667, 0.1333333333333334, 0.1000000000000001, 0.0666666666666668, 0.0333333333333334, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, 0.0000000000000001, -0.0333333333333332, -0.0666666666666666, -0.0999999999999999, -0.1333333333333332, -0.1666666666666665, -0.1999999999999999, -0.2333333333333332, -0.2666666666666666, -0.2999999999999999, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3333333333333332, -0.3666666666666665, -0.3999999999999999, -0.4333333333333332, -0.4666666666666665, -0.4999999999999998, -0.5333333333333332, -0.5666666666666665, -0.5999999999999999, -0.6333333333333332, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6666666666666665, -0.6999999999999998, -0.7333333333333332, -0.7666666666666665, -0.7999999999999998, -0.8333333333333331, -0.8666666666666665, -0.8999999999999998, -0.9333333333333331, -0.9666666666666665 ; ...
 ] ...
);

% build options
options = struct( ...
    'timeStep', 0.0100000000000000, ...
    'zonotopeOrder', 100, ...
    'reductionTechnique', 'pca', ...
    'points', 1, ...
    'p_conf', 0.9990000000000000 ...
);

% build simResult
simRes = simResult( ...
{ [ 0.382 0.308 0.001 0.558 ; 0.382 0.314 0.003 0.556 ; 0.382 0.319 0.010 0.551 ; 0.382 0.325 0.014 0.550 ; 0.382 0.330 0.010 0.548 ; 0.382 0.336 0.012 0.547 ; 0.382 0.341 0.018 0.550 ; 0.382 0.347 0.013 0.550 ; 0.383 0.352 0.021 0.551 ; 0.383 0.358 0.024 0.553 ; 0.383 0.363 0.030 0.549 ; 0.383 0.369 0.025 0.544 ; 0.384 0.374 0.024 0.545 ; 0.384 0.380 0.028 0.540 ; 0.384 0.385 0.033 0.545 ; 0.385 0.391 0.038 0.545 ; 0.385 0.396 0.041 0.549 ; 0.385 0.402 0.047 0.551 ; 0.386 0.407 0.048 0.549 ; 0.386 0.413 0.055 0.549 ; 0.387 0.418 0.060 0.550 ; 0.388 0.424 0.069 0.550 ; 0.388 0.429 0.078 0.553 ; 0.389 0.435 0.088 0.554 ; 0.390 0.440 0.099 0.557 ; 0.391 0.446 0.109 0.553 ; 0.392 0.451 0.113 0.555 ; 0.393 0.457 0.122 0.553 ; 0.395 0.462 0.133 0.551 ; 0.396 0.468 0.140 0.548 ; 0.397 0.473 0.151 0.551 ; 0.399 0.479 0.162 0.547 ; 0.401 0.484 0.172 0.548 ; 0.403 0.490 0.184 0.555 ; 0.404 0.495 0.193 0.553 ; 0.406 0.501 0.206 0.556 ; 0.408 0.506 0.206 0.559 ; 0.411 0.512 0.219 0.555 ; 0.413 0.518 0.231 0.557 ; 0.415 0.523 0.242 0.560 ; 0.418 0.529 0.253 0.565 ; 0.420 0.534 0.263 0.574 ; 0.423 0.540 0.274 0.578 ; 0.426 0.546 0.282 0.580 ; 0.429 0.552 0.287 0.588 ; 0.431 0.558 0.294 0.598 ; 0.434 0.564 0.297 0.603 ; 0.437 0.570 0.305 0.612 ; 0.441 0.576 0.313 0.621 ; 0.444 0.582 0.324 0.623 ; 0.447 0.588 0.332 0.624 ; 0.450 0.595 0.336 0.626 ; 0.454 0.601 0.335 0.629 ; 0.457 0.607 0.343 0.636 ; 0.461 0.614 0.353 0.640 ; 0.464 0.620 0.359 0.649 ; 0.468 0.627 0.374 0.655 ; 0.472 0.633 0.380 0.662 ; 0.475 0.640 0.385 0.674 ; 0.479 0.647 0.390 0.686 ; 0.483 0.654 0.392 0.692 ; 0.487 0.661 0.396 0.699 ; 0.491 0.668 0.403 0.707 ; 0.495 0.675 0.409 0.713 ; 0.499 0.682 0.418 0.721 ; 0.503 0.689 0.424 0.731 ; 0.508 0.696 0.428 0.728 ; 0.512 0.704 0.431 0.739 ; 0.516 0.711 0.430 0.742 ; 0.521 0.719 0.437 0.749 ; 0.525 0.726 0.446 0.757 ; 0.530 0.734 0.451 0.769 ; 0.534 0.742 0.456 0.773 ; 0.539 0.749 0.455 0.785 ; 0.543 0.757 0.456 0.794 ; 0.548 0.765 0.457 0.799 ; 0.552 0.773 0.460 0.809 ; 0.557 0.781 0.457 0.814 ; 0.562 0.789 0.454 0.816 ; 0.566 0.798 0.455 0.822 ; 0.571 0.806 0.458 0.828 ; 0.575 0.814 0.456 0.836 ; 0.580 0.823 0.459 0.845 ; 0.584 0.831 0.465 0.851 ; 0.589 0.840 0.463 0.855 ; 0.594 0.848 0.466 0.861 ; 0.598 0.857 0.470 0.869 ; 0.603 0.866 0.473 0.873 ; 0.608 0.874 0.473 0.878 ; 0.613 0.883 0.471 0.887 ; 0.617 0.892 0.474 0.896 ; 0.622 0.901 0.471 0.904 ; 0.627 0.910 0.468 0.902 ; 0.631 0.919 0.467 0.905 ; 0.636 0.928 0.464 0.907 ; 0.641 0.937 0.463 0.909 ; 0.645 0.946 0.462 0.908 ; 0.650 0.955 0.463 0.912 ; 0.655 0.965 0.459 0.913 ; 0.659 0.974 0.458 0.915 ; 0.664 0.983 0.457 0.913 ; 0.668 0.992 0.459 0.916 ; 0.673 1.001 0.460 0.917 ; 0.677 1.010 0.455 0.914 ; 0.682 1.019 0.456 0.912 ; 0.687 1.029 0.459 0.906 ; 0.691 1.038 0.460 0.907 ; 0.696 1.047 0.461 0.911 ; 0.700 1.056 0.458 0.911 ; 0.705 1.065 0.453 0.913 ; 0.709 1.074 0.456 0.910 ; 0.714 1.083 0.456 0.909 ; 0.719 1.092 0.455 0.911 ; 0.723 1.101 0.450 0.912 ; 0.728 1.110 0.453 0.916 ; 0.732 1.120 0.452 0.917 ; 0.737 1.129 0.451 0.922 ; 0.741 1.138 0.445 0.924 ; 0.746 1.147 0.439 0.919 ; 0.750 1.156 0.437 0.918 ; 0.754 1.166 0.433 0.914 ; 0.759 1.175 0.433 0.921 ; 0.763 1.184 0.434 0.925 ; 0.767 1.193 0.433 0.927 ; 0.772 1.203 0.436 0.925 ; 0.776 1.212 0.436 0.928 ; 0.780 1.221 0.431 0.923 ; 0.785 1.230 0.432 0.923 ; 0.789 1.239 0.434 0.921 ; 0.793 1.249 0.432 0.916 ; 0.798 1.258 0.431 0.920 ; 0.802 1.267 0.426 0.919 ; 0.806 1.276 0.428 0.918 ; 0.810 1.285 0.425 0.918 ; 0.815 1.295 0.429 0.923 ; 0.819 1.304 0.426 0.929 ; 0.823 1.313 0.428 0.932 ; 0.828 1.322 0.425 0.929 ; 0.832 1.332 0.418 0.931 ; 0.836 1.341 0.418 0.929 ; 0.840 1.350 0.419 0.930 ; 0.844 1.360 0.418 0.929 ; 0.848 1.369 0.425 0.932 ; 0.853 1.378 0.425 0.932 ; 0.857 1.388 0.421 0.929 ; 0.861 1.397 0.421 0.931 ; 0.865 1.406 0.418 0.932 ; 0.870 1.416 0.419 0.930 ; 0.874 1.425 0.418 0.930 ; 0.878 1.434 0.414 0.928 ; 0.882 1.443 0.411 0.927 ; 0.886 1.453 0.405 0.930 ; 0.890 1.462 0.402 0.926 ; 0.894 1.471 0.400 0.925 ; 0.898 1.480 0.400 0.927 ; 0.902 1.490 0.403 0.931 ; 0.906 1.499 0.404 0.931 ; 0.910 1.508 0.410 0.930 ; 0.914 1.518 0.408 0.933 ; 0.918 1.527 0.413 0.929 ; 0.923 1.536 0.412 0.927 ; 0.927 1.546 0.415 0.929 ; 0.931 1.555 0.413 0.931 ; 0.935 1.564 0.418 0.929 ; 0.939 1.573 0.418 0.929 ; 0.943 1.583 0.415 0.927 ; 0.948 1.592 0.414 0.930 ; 0.952 1.601 0.417 0.927 ; 0.956 1.611 0.414 0.926 ; 0.960 1.620 0.415 0.930 ; 0.964 1.629 0.412 0.926 ; 0.968 1.638 0.416 0.926 ; 0.972 1.648 0.418 0.924 ; 0.977 1.657 0.418 0.924 ; 0.981 1.666 0.419 0.932 ; 0.985 1.676 0.419 0.932 ; 0.989 1.685 0.418 0.936 ; 0.993 1.694 0.413 0.936 ; 0.997 1.704 0.413 0.940 ; 1.002 1.713 0.417 0.940 ; 1.006 1.722 0.416 0.938 ; 1.010 1.732 0.414 0.938 ; 1.014 1.741 0.411 0.936 ; 1.018 1.751 0.406 0.939 ; 1.022 1.760 0.408 0.938 ; 1.026 1.769 0.408 0.939 ; 1.030 1.779 0.416 0.943 ; 1.034 1.788 0.406 0.943 ; 1.039 1.798 0.408 0.943 ; 1.043 1.807 0.402 0.947 ; 1.047 1.816 0.399 0.949 ; 1.051 1.826 0.394 0.954 ; 1.054 1.836 0.391 0.955 ; 1.058 1.845 0.389 0.954 ; 1.062 1.855 0.382 0.955 ; 1.066 1.864 0.381 0.949 ; 1.070 1.874 0.377 0.945 ; 1.074 1.883 0.375 0.937 ; 1.077 1.892 0.371 0.939 ; 1.081 1.902 0.368 0.936 ; 1.085 1.911 0.369 0.932 ; 1.088 1.920 0.366 0.930 ; 1.092 1.930 0.362 0.921 ; 1.096 1.939 0.360 0.921 ; 1.099 1.948 0.351 0.921 ; 1.103 1.957 0.352 0.919 ; 1.106 1.967 0.349 0.913 ; 1.110 1.976 0.344 0.910 ; 1.113 1.985 0.336 0.910 ; 1.116 1.994 0.328 0.906 ; 1.120 2.003 0.322 0.902 ; 1.123 2.012 0.317 0.906 ; 1.126 2.021 0.312 0.904 ; 1.129 2.030 0.306 0.905 ; 1.132 2.039 0.302 0.897 ; 1.135 2.048 0.292 0.896 ; 1.138 2.057 0.284 0.893 ; 1.141 2.066 0.272 0.887 ; 1.143 2.075 0.264 0.880 ; 1.146 2.083 0.253 0.867 ; 1.149 2.092 0.253 0.868 ; 1.151 2.101 0.244 0.863 ; 1.153 2.109 0.234 0.858 ; 1.156 2.118 0.232 0.847 ; 1.158 2.126 0.218 0.842 ; 1.160 2.135 0.204 0.838 ; 1.162 2.143 0.194 0.829 ; 1.164 2.151 0.181 0.826 ; 1.166 2.160 0.174 0.822 ; 1.167 2.168 0.167 0.816 ; 1.169 2.176 0.157 0.810 ; 1.171 2.184 0.136 0.802 ; 1.172 2.192 0.127 0.792 ; 1.173 2.200 0.121 0.789 ; 1.174 2.208 0.110 0.777 ; 1.175 2.215 0.094 0.772 ; 1.176 2.223 0.081 0.765 ; 1.177 2.231 0.075 0.758 ; 1.178 2.238 0.064 0.749 ; 1.178 2.246 0.059 0.740 ] }, ...
{ [ 0.010 ; 0.020 ; 0.030 ; 0.040 ; 0.050 ; 0.060 ; 0.070 ; 0.080 ; 0.090 ; 0.100 ; 0.110 ; 0.120 ; 0.130 ; 0.140 ; 0.150 ; 0.160 ; 0.170 ; 0.180 ; 0.190 ; 0.200 ; 0.210 ; 0.220 ; 0.230 ; 0.240 ; 0.250 ; 0.260 ; 0.270 ; 0.280 ; 0.290 ; 0.300 ; 0.310 ; 0.320 ; 0.330 ; 0.340 ; 0.350 ; 0.360 ; 0.370 ; 0.380 ; 0.390 ; 0.400 ; 0.410 ; 0.420 ; 0.430 ; 0.440 ; 0.450 ; 0.460 ; 0.470 ; 0.480 ; 0.490 ; 0.500 ; 0.510 ; 0.520 ; 0.530 ; 0.540 ; 0.550 ; 0.560 ; 0.570 ; 0.580 ; 0.590 ; 0.600 ; 0.610 ; 0.620 ; 0.630 ; 0.640 ; 0.650 ; 0.660 ; 0.670 ; 0.680 ; 0.690 ; 0.700 ; 0.710 ; 0.720 ; 0.730 ; 0.740 ; 0.750 ; 0.760 ; 0.770 ; 0.780 ; 0.790 ; 0.800 ; 0.810 ; 0.820 ; 0.830 ; 0.840 ; 0.850 ; 0.860 ; 0.870 ; 0.880 ; 0.890 ; 0.900 ; 0.910 ; 0.920 ; 0.930 ; 0.940 ; 0.950 ; 0.960 ; 0.970 ; 0.980 ; 0.990 ; 1.000 ; 1.010 ; 1.020 ; 1.030 ; 1.040 ; 1.050 ; 1.060 ; 1.070 ; 1.080 ; 1.090 ; 1.100 ; 1.110 ; 1.120 ; 1.130 ; 1.140 ; 1.150 ; 1.160 ; 1.170 ; 1.180 ; 1.190 ; 1.200 ; 1.210 ; 1.220 ; 1.230 ; 1.240 ; 1.250 ; 1.260 ; 1.270 ; 1.280 ; 1.290 ; 1.300 ; 1.310 ; 1.320 ; 1.330 ; 1.340 ; 1.350 ; 1.360 ; 1.370 ; 1.380 ; 1.390 ; 1.400 ; 1.410 ; 1.420 ; 1.430 ; 1.440 ; 1.450 ; 1.460 ; 1.470 ; 1.480 ; 1.490 ; 1.500 ; 1.510 ; 1.520 ; 1.530 ; 1.540 ; 1.550 ; 1.560 ; 1.570 ; 1.580 ; 1.590 ; 1.600 ; 1.610 ; 1.620 ; 1.630 ; 1.640 ; 1.650 ; 1.660 ; 1.670 ; 1.680 ; 1.690 ; 1.700 ; 1.710 ; 1.720 ; 1.730 ; 1.740 ; 1.750 ; 1.760 ; 1.770 ; 1.780 ; 1.790 ; 1.800 ; 1.810 ; 1.820 ; 1.830 ; 1.840 ; 1.850 ; 1.860 ; 1.870 ; 1.880 ; 1.890 ; 1.900 ; 1.910 ; 1.920 ; 1.930 ; 1.940 ; 1.950 ; 1.960 ; 1.970 ; 1.980 ; 1.990 ; 2.000 ; 2.010 ; 2.020 ; 2.030 ; 2.040 ; 2.050 ; 2.060 ; 2.070 ; 2.080 ; 2.090 ; 2.100 ; 2.110 ; 2.120 ; 2.130 ; 2.140 ; 2.150 ; 2.160 ; 2.170 ; 2.180 ; 2.190 ; 2.200 ; 2.210 ; 2.220 ; 2.230 ; 2.240 ; 2.250 ; 2.260 ; 2.270 ; 2.280 ; 2.290 ; 2.300 ; 2.310 ; 2.320 ; 2.330 ; 2.340 ; 2.350 ; 2.360 ; 2.370 ; 2.380 ; 2.390 ; 2.400 ] }, ...
[], ...
{}, ...
{} ...
);

end

% ------------------------------ END OF CODE ------------------------------
