function completed = example_linearSysDT_observe_01_vehicle()
% example_linearSysDT_observe_01_vehicle - example for guaranteed state
%    estimation of linear discrete-time systems; a corresponding unit test
%    function exists; shows the solution of the linearSysDT class for a
%    vehicle example.
%
% Syntax:
%    completed = example_linearSysDT_observe_01_vehicle()
%
% Inputs:
%    -
%
% Outputs:
%    completed - true/false

% Authors:       Matthias Althoff
% Written:       30-April-2021
% Last update:   13-July-2021
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

%% Load vehicle model
% load vehicleModel_dim6 vehicle params options simRes
[vehicle,params,options,simRes] = aux_load_model();

% save original parameters
paramsOrig = params;

% Set of evaluated estimators
Estimator = {
    'FRad-A' 
    'FRad-B' 
    'ESO-A'
    'FRad-C'  
    'Nom-G' 
    'ESO-D'
    'Hinf-G' 
    };

% set solver
options.solver = 'sdpt3';
% ensure that solver is on path
if ~isSolverInstalled('sdpt3')
    throw(CORAerror('CORA:noSuitableSolver','sdpt3'));
end

%% perform evaluation
% loop over estimators
for iEst = 1:length(Estimator)

    % set algorithm
    estName = Estimator{iEst};
    options.alg = estName;
    
    % reset parameters
    params = paramsOrig;

    %% Initial sets, disturbance sets, and noise sets
    % if ellipsoids are required
    if any(strcmp(estName,{'ESO-A','ESO-B','ESO-C','ESO-D','ESO-E'}))
        params.R0 = ellipsoid(params.R0,'i:norm'); % inscribe ellipsoid
        params.W = ellipsoid(params.W,'i:norm'); % inscribe ellipsoid
        params.V = ellipsoid(params.V,'i:norm'); % inscribe ellipsoid
    % if constrained zonotopes are required
    elseif any(strcmp(estName,{'CZN-A','CZN-B'}))
        params.R0 = conZonotope(params.R0);
    end

    % perform set-based estimation 
    estSet = observe(vehicle,params,options);
    %estSetAll = evaluateSingleObserver(vehicle,params,options); % <-- uncomment for detailed evaluation
    %estSet = estSetAll.EstStates;
    
    % plot results
    for iDim = 1:2
        figure; hold on;
        % plot time elapse
        plotOverTime(estSet,iDim);
        % plot simulation
        plotOverTime(simRes,iDim);

        % label plot
        xlabel('t');
        ylabel(['x_{',num2str(iDim),'}']);
    end
end

% example completed
completed = true;

end


% Auxiliary functions -----------------------------------------------------

function [vehicle,params,options,simRes] = aux_load_model()

% generated semi-automatically using printSystem, printSet, etc. 
% from saved .mat file

% build system
name = 'vehicle';
A = [ 0.9946935969982688 -0.0078691991668298 0.0000000000000000 0.0000000000000000 0.0209662384218574 0.0000692473390723 ; 0.3148478589494614 0.9313367791758462 0.0000000000000000 0.0000000000000000 0.2273008479956925 0.0007451292454428 ; 0.0015342266053718 0.0096512982169148 1.0000000000000000 0.0000000000000000 0.0012038414767041 0.0000028669514454 ; 0.1725622327472465 0.1441668418196496 0.0000000000000000 1.0000000000000000 0.0197395176677208 0.0000470241806250 ; 0.2649205462634165 0.0142908209764458 0.0000000000000000 0.0000000000000000 0.7400344979955725 0.0043087584628022 ; 39.0352940956934731 1.9602566641561181 0.0000000000000000 0.0000000000000000 -37.7608306393335837 0.0506331439472220 ];
B = [ 0.0002215914850313 ; 0.0023844135854171 ; 0.0000091742446252 ; 0.0001504773779999 ; 0.0137880270809670 ; 0.1620260606311103 ];
c = [ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ];
C = [ 0.0000000000000000 15.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 ];
D = [];
k = [];
E = [];
F = [];
dt = 0.0100000000000000;
vehicle = linearSysDT(name,A,B,c,C,D,k,E,F,dt);

% build params
params = struct( ...
    'tFinal', 3.5000000000000000, ...
    'R0', zonotope([ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ], [ 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 1.0000000000000000 ]), ...
    'V', zonotope([ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ], [ 0.1000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.1000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.1000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.1000000000000000 ]), ...
    'W', zonotope([ 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ; 0.0000000000000000 ], [ 0.0050000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0550000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0050000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0400000000000000 0.0000000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0200000000000000 0.0000000000000000 ; 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.0000000000000000 0.2000000000000000 ]), ...
    'u', [ 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, -0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.1000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000, 0.0000000000000000 ], ...
    'y', [ ...
 -5.2780333827667656, -0.4850485282073204, 4.9400789801957981, 9.9392809939114599, 15.6099190711340086, 20.3654568094596620, 25.4297561952685136, 30.1835146005310904, 33.8388209098265591, 37.9970316547130409, 41.2686381054855573, 44.6818527080966703, 48.4538299141379341, 51.3897168586351043, 54.3530219962615959, 57.3624968783100471, 59.1944924405402162, 61.1200172364978229, 63.5100391676178262, 64.9768939815994173, 66.6810218274736570, 68.0069101217344638, 69.3179522615082107, 70.0582190053761167, 71.0833117296599966, 71.5196414472885209, 71.7566575425059483, 72.4519659478677625, 71.4553390522270178, 70.3493853770373221, 69.9288125547442974, 68.8020820038028944, 68.2786978977910763, 67.2003292258413865, 66.1710078621524076, 64.7076800663446079, 63.8142566447653081, 62.3341747414627747, 61.2068549717580055, 59.0408079127157066, 57.0113420610496320, 55.3769901791804955, 53.2270394079661671, 51.3319639827304073, 49.2594967548852907, 47.1602163818822504, 44.2416743827346934, 41.7188099213618315, 40.0139339077322589, 38.4612039866607915, 36.1470978767179147, 33.0860135541571836, 30.9229776270936085, 28.6385726153050832, 26.8859416749797582, 24.6714510222954253, 21.8399729569516410, 20.1905853500569741, 18.0848734122432546, 15.9790617696367683, 13.6700219358415325, 11.5175082203198738, 9.3313464893470517, 7.6557637244244248, 5.8151056407057693, 4.2913705896202643, 2.3342431233398289, 0.2690128923252835, -1.1975634443012297, -3.3282601491592629, -4.9970904043996347, -6.2843477532095777, -7.8545488102513250, -9.0815933255813643, -10.2627020842643120, -12.0188207115971615, -13.2656881487994305, -14.7485334279768932, -15.5232901414567319, -16.5203705769887392, -17.8022570470794967, -18.0059255953791606, -18.9925188449730769, -19.6982813797438503, -20.1893580396005170, -19.8629014405377227, -19.8108859335090912, -19.6872575252353492, -20.0085509452323649, -19.8958332558336117, -19.9119708294056892, -19.8821897829880037, -19.5983351537061061, -20.0476398082671921, -19.5888702188525201, -19.4349556308319080, -19.0223916221891010, -19.6283676754169498, -19.1285889197563677, -19.6898094353078754, -19.9003058408017459, -20.1488748222947400, -19.4376764102879847, -19.0109927978140938, -18.3732124293985528, -17.4739951780450049, -16.4919025154184240, -15.5723282678951342, -15.0915599649223271, -14.7904436705400162, -14.4031821969653571, -14.3531126749735130, -14.4826480598193204, -14.4998733171594818, -13.7795167223636810, -12.9950252971197528, -12.8263725670675406, -12.1160418120241324, -11.4964157691821995, -11.2622104063307997, -10.4769519106073581, -9.3153738097965562, -8.5904288308394232, -8.3633485645942045, -7.3728694389189924, -7.1303564160274204, -6.2191693429699715, -5.1861525723713946, -5.0124156779096936, -4.2168534186875055, -3.6788260325095732, -3.7776200416486163, -2.9649912882140166, -2.3864653382234522, -1.8664789588834374, -0.6230145071359003, -0.1117703338200136, 0.5776622867811677, 0.8427203664297694, 1.1981454449737281, 1.1107427362779370, 1.7324667734112835, 2.1128320871880559, 2.1800528198630289, 2.6604752414996664, 2.8258792483124018, 2.7594814211061389, 2.4799132425095918, 3.1741498047068943, 3.9614090883043813, 4.1331486410405374, 4.6265214297970401, 4.6805142477498656, 4.9112515769930782, 5.3790414049351902, 6.0775999907708478, 6.4269216526817923, 6.8998037334950402, 7.3155179597536852, 7.0572798147156321, 6.9515793851622449, 6.9305208690934981, 7.8217930210088182, 7.8703765345620695, 8.3528223895924771, 8.0777455643903107, 8.0496429880364708, 7.5470681503060302, 6.7944555659450288, 5.9060672961714129, 5.5953827471419455, 5.5101253316273917, 5.6170891285918589, 6.1552460433629577, 6.1825916883177614, 6.1809805529419508, 5.8891252859242940, 5.7442403397587576, 5.4974869087072147, 4.5736577005642616, 3.6836921683762518, 3.0111968727790912, 2.9455142136191670, 2.4721884719998704, 2.2732478456989686, 2.5726172939258962, 3.2137091518038945, 2.7227590185614283, 2.1756073673772054, 2.1887648247115430, 1.3857813336767189, 1.9640416154728613, 1.9301354031803062, 1.4809690409598395, 1.5404260841052968, 1.0302397498454068, 0.8963018462855441, 0.8781656117327794, 0.7436099065009576, 0.3207971673263032, 0.0608147099354043, -0.1959479188076262, -0.0602365398840804, 0.7294816942310174, 0.5684389298375159, 0.1188328139761811, 0.3331050422841721, 0.0794689425767491, 0.6982650542302067, 1.3878092462508671, 0.8751876139128008, 1.6045782043856414, 1.4796947574445292, 1.2548770558709985, 1.1175419269534741, 1.5585111506941567, 1.1702178046115119, 0.9043207319724658, 1.2598536955681165, 1.3250188730727244, 1.2083553441411039, 1.9963850253162212, 2.0407430938429751, 1.6587428560373703, 1.6562712874037184, 1.0435634018313731, 1.3182968416965799, 1.2841247953278776, 1.2965624572625771, 1.6147482021905195, 1.4606660796846340, 1.3211529551741346, 1.2362181911249932, 1.7024620372214363, 2.0388036542251293, 1.5077260319037633, 1.5353920402096961, 1.5731353577897680, 1.2651496402871971, 1.6258340979179275, 2.2082407921427367, 2.5553651486207034, 2.2221229005251990, 1.8867213101482392, 2.2504963913503535, 2.3563953675505953, 2.8773958796449244, 2.6386189032294749, 3.0786442333526343, 2.7535297079062477, 2.9671407794742395, 3.0712245711544996, 2.7366743508709725, 2.7357665976467365, 2.1983362020043655, 2.3600019030018777, 2.5519693682978204, 2.3259388125268843, 2.0472062776442770, 2.1769413641291209, 2.3099839874896886, 2.3245604911940472, 1.8327495405237977, 0.8734410906714640, 0.9941394673005137, 0.4151224027067809, 0.6718957304396420, 0.4176260922201341, -0.4659947903829683, -0.1214977595528897, -0.4617249697484664, -0.9947418323213012, -0.6566398070591905, -0.1290398635043740, -0.0501657090665430, 0.5695554192651542, -0.2069469855949261, -0.5303168908100631, -0.1993158688842447, 0.1640032997297409, 0.6727302913382613, 0.9741888466323750, 1.4191606451339265, 1.4911230887027169, 1.6307330343306217, 1.2774464455237451, 0.5987163078441999, 0.2455656898621337, 0.4682279593913654, 0.5565175071517307, 0.3116389446972102, -0.2896618253612200, -0.1542417823503945, 0.0193021222502279, 0.1025025588115017, 0.7209362261986225, -0.2875290807523564, -1.0113909013185993, -1.2113958317739837, -1.2492970304736912, -1.1461693163373732, -0.7234226711550811, -0.1383829071943681, -0.6615319840160527, -0.8900199180502496, -1.4535758849655711, -2.2011045954988955, -1.8906213525463484, -2.4366434650568078, -2.8333674443875942, -2.6956647619375289, -2.1542179488999751, -2.0178083634913428, -2.5936664435948429, -3.0738904221617083, -2.2658290482076784, -2.0721050489535058, -2.3831578439678367, -1.6970782670888058, -1.3702633478205324, -1.3465573907590083, -1.3625428019896864, -1.1684512391264152, -1.1490724931212959, -1.2937869639936688, -0.7825403891736235, -1.4256668814019624, -1.7613948966817281, -1.7993391694470420, -1.9780703355276175, -2.0762783232314419, -1.7291689447493450, -1.7742708901277457, -1.5569378251958708, -1.6611826519802746, -1.7367008848086527, -1.5536853358623908, -1.9588458582947255, -1.6316317942934981, -1.6645611143721000, -1.0436018147295947, -1.0770577599638858, -1.0445051014341775, -0.9932759000190433, -0.6162808838699482, -1.1927254833954997, -0.3717930895861014, 0.0367440112955872, -0.4264848812362566, -0.7516776768238089 ; ...
 0.1832976301476903, 0.1970502621343604, 0.2619650934826397, 0.2244185011573657, 0.2423303647672004, 0.2102898812444104, 0.2619558266360018, 0.2499949500861377, 0.2433327424737657, 0.2617536172401908, 0.2786604786685340, 0.3892571744383651, 0.3704145120084437, 0.4062954255416703, 0.4596802208693836, 0.4728412266563655, 0.5196485477388365, 0.5710601718705782, 0.6539854116523786, 0.7679130332413970, 0.7369871313485092, 0.6911620519277393, 0.8879987165568464, 0.8652590934109082, 1.0013062311383669, 0.9319505334798461, 1.0193630967145149, 1.0951251452696757, 1.0711169078712519, 1.1906145316735848, 1.1426126461704518, 1.2449902127497872, 1.2790045942165165, 1.2871782435648396, 1.3515640353459002, 1.3803247597981501, 1.4460361346805206, 1.5505285497873524, 1.5755695322756020, 1.5423131355777251, 1.6394721304622508, 1.6590648309272924, 1.7417906775865026, 1.7431676491411532, 1.8021616407667942, 1.7452882862163608, 1.8468260272982862, 1.8903520986568145, 1.8666676177808870, 1.9832645484896603, 1.9624418122279805, 1.9856231015487023, 1.9428635210867542, 2.0599516605944306, 2.0197060304646515, 2.0976189923985622, 2.0630976317172576, 2.0577667586090915, 2.0813560916130869, 2.0873866811007402, 2.0525582839233318, 2.0671521888228730, 2.1142375056877798, 2.1854273929676151, 2.1793821831213132, 2.0784511313480460, 2.1259903346307394, 2.1205439850027705, 2.0947456825503070, 2.1129662159178868, 2.0351048947549186, 2.1735435828616620, 2.1774838385815092, 2.1657725745194947, 2.0773049312027871, 2.0682752716293860, 2.1341075827288565, 2.0353822452810353, 2.0248121838545212, 2.1107257278239122, 2.0610214555996866, 1.9524609317019759, 1.9816177692646195, 2.0672971965114479, 2.0272993606708121, 2.0625262233896042, 1.9917701760832691, 1.9219142707045742, 1.8975432100475456, 1.9232981972358485, 1.8623871140984982, 1.8698881511051209, 1.8720377927372109, 1.9066233554140557, 1.8746523378010107, 1.8539273506493130, 1.9046344640830175, 1.8174054783456348, 1.8899007462165451, 1.8242738723001963, 1.7649049703468718, 1.7031381579969891, 1.7600789390857119, 1.7059458980114421, 1.7495032527743304, 1.7975627282435582, 1.6921145534386284, 1.6847125142084596, 1.7047608937491485, 1.7274882630448272, 1.6628419219598642, 1.7408576932405586, 1.6335459257719451, 1.6755242347403778, 1.6231054132444360, 1.5892301586559630, 1.6040954534755090, 1.5384042177934663, 1.6380379226362185, 1.6429044369204797, 1.6146761050915266, 1.5971102588901336, 1.5228774159311376, 1.5730782172701800, 1.5185792435397680, 1.5301961775732222, 1.5285712438297157, 1.5018629543112274, 1.5218871945941130, 1.5023817956057983, 1.4854270100498392, 1.5353625150771586, 1.5565191064031298, 1.5125472523020789, 1.5387710335469049, 1.4967602631569594, 1.4817823482019352, 1.4663673938717181, 1.5160843559368873, 1.5724805231870167, 1.4778081158426841, 1.5037510192766961, 1.5311121891023145, 1.4775567929326157, 1.4997115159340357, 1.5587241100960416, 1.5536289364515337, 1.5441783475589388, 1.4684887721170987, 1.4988096511680959, 1.4833929984370056, 1.5073649007993857, 1.4923126992599629, 1.5848941502692104, 1.5576722500256048, 1.6505084336327376, 1.6021377088323132, 1.5629854834437569, 1.5727017582848866, 1.5997661205396085, 1.6141731376652841, 1.6819709617096981, 1.6630526369534899, 1.6951834907962937, 1.6173276454343923, 1.6220439307065764, 1.6671527646827387, 1.6299602222796583, 1.6073321885865370, 1.6701049705007118, 1.5693351999254017, 1.6176081246674126, 1.6794822132832055, 1.7076466859599591, 1.6900897858079040, 1.7004538619194687, 1.7287866934370291, 1.7228997602514051, 1.6870025564802111, 1.7248923328429047, 1.6097884209922493, 1.6599681903894667, 1.7657821045466500, 1.7201793578504445, 1.7328788435889009, 1.6987210124107364, 1.7544042953302985, 1.6790132368062489, 1.7311640107933342, 1.7403439744227973, 1.6704553119472474, 1.7253982743548655, 1.7636669187903269, 1.6858656642533749, 1.6864745221726376, 1.6851858983394272, 1.7532575763425298, 1.6540905720596326, 1.6807608291616321, 1.7063313547461898, 1.6836676470932093, 1.6163074228540826, 1.7369702014591601, 1.6527078035508835, 1.7325526391815087, 1.6329973459712801, 1.7086609655467004, 1.7293704650559731, 1.6442729124288986, 1.7192074253906326, 1.6871568755900537, 1.7022720168598315, 1.6906535818968556, 1.6920330111691093, 1.7153036235219576, 1.7271605861351844, 1.7347590259941161, 1.7157774960074823, 1.7821143465423157, 1.7451647052027170, 1.7190762109458766, 1.7389270698141164, 1.7333882257207911, 1.7008751474574235, 1.6965620769632928, 1.7327077101980202, 1.6915402961202208, 1.6759611005422519, 1.7441787617703040, 1.6715623663019572, 1.7332443226533840, 1.7103714083110069, 1.7650978522923841, 1.7555202707092714, 1.6964280359729265, 1.6837839240124708, 1.6941507495579429, 1.7959129578026813, 1.7735784097932088, 1.8157229049366719, 1.8049601682117007, 1.7045398174429256, 1.7084968986173255, 1.6787642570611570, 1.7352158604864785, 1.7232985638194172, 1.7631691441682933, 1.6837583348012666, 1.7376859230014849, 1.7081078432750907, 1.8353714751018035, 1.7875941413256642, 1.8157547302377610, 1.8063711843889998, 1.7413016294141384, 1.7033182540825023, 1.7182748236473209, 1.8063120567140181, 1.8222384343850733, 1.7144876500738238, 1.6648626372986612, 1.7794324906186734, 1.7864123416551463, 1.7720097032548239, 1.8001224838232364, 1.7202905259121239, 1.8393741492243751, 1.6894287838142132, 1.8746501922141883, 1.8141668853887483, 1.8036297964258343, 1.8392557224650810, 1.7607898716254826, 1.6799622454634571, 1.7632700792829805, 1.7889610549651898, 1.7433957188473430, 1.7024823342851905, 1.8149532169893277, 1.7731510699946351, 1.7191954571237065, 1.8159016936161068, 1.7492044997356291, 1.8018230557054147, 1.7143509605732399, 1.7280412420810012, 1.7760347691508762, 1.7464480327271374, 1.7728580393475759, 1.7538929465032631, 1.8111874537735406, 1.7619390743688945, 1.7412763458956833, 1.7467499241618749, 1.7427948676643514, 1.8096691447791964, 1.7581874905358184, 1.7120454466707320, 1.7592554015677606, 1.7496168312595592, 1.7093740405908877, 1.7682354076524509, 1.7111393181769798, 1.7819834164544093, 1.7662093351332597, 1.7578187532746097, 1.8059229600352200, 1.7942808976429234, 1.7774492573512009, 1.7550655972343989, 1.7751393830372781, 1.7188622132503668, 1.7123516971243715, 1.6846229701128974, 1.7450465923468486, 1.7112420093051071, 1.7500431088832857, 1.7579190712641763, 1.7570960491066310, 1.7884594145330746, 1.7462325419132358, 1.7459132185430661, 1.8155407128744145, 1.7495290056029713, 1.7439039052199985, 1.8001432890855640, 1.7160049073542012, 1.7568868997158018, 1.7241461002443206, 1.7655743205076013, 1.8028513538770288, 1.7743464661206512, 1.6535236542641463, 1.7178852902669557, 1.7295225328357253, 1.7286528830682604, 1.7403619303362234, 1.7740992999763814, 1.6968496598114362, 1.7110558973770098, 1.7185019417303882, 1.7161816650240944, 1.7179701255660924, 1.6785008893033604, 1.7136977277409351, 1.7163344018181315, 1.7013639611058953, 1.6695491147961476, 1.6948172384932103, 1.7718313267400854 ; ...
 0.7135466891301367, 0.8706134412780386, 0.9133850132919805, 1.0872446835722718, 1.3435438080717685, 1.6031832653245786, 2.0522970219375285, 2.4553858562111599, 2.8640026446960061, 3.2585001180809305, 3.7542445088274334, 4.3763560149535037, 4.8555869292694283, 5.5210546680277544, 6.1101860572625748, 6.8524180823003968, 7.4879325863922439, 8.1912679556141015, 8.8226141111006928, 9.6714224099188275, 10.4738704605361264, 11.2097469922359068, 11.9616756022991488, 12.7437522254804012, 13.5080655549843129, 14.2945030173633931, 15.0762323793516675, 15.8947474539173221, 16.6801463850043383, 17.4121498222827640, 18.2191966698816934, 19.0340253763147764, 19.7411225495819806, 20.3938612570181377, 21.1495548809959892, 21.9292166100106165, 22.5573830046967814, 23.1578874688037573, 23.7859996266102449, 24.4884515041233684, 25.0441858941257749, 25.6330570450045343, 26.0992571730247604, 26.7245585725636907, 27.2136910671889396, 27.7973461816130616, 28.1420456154600167, 28.5354662097903571, 28.9446984444402524, 29.3047395966851880, 29.6849534050848192, 29.9913946864620335, 30.2818192796853403, 30.6426981237885734, 30.8714538250584560, 31.0821367918298037, 31.3724975285718806, 31.6144453562417631, 31.7389887466810663, 31.9252870036244119, 32.0462033440549590, 32.1609782316397812, 32.2293552896675237, 32.2994268908352211, 32.2550845682143077, 32.3021190602619228, 32.2476351536988233, 32.2552162312475161, 32.1765791353279837, 32.1838356605189162, 32.0771236623615650, 32.0549128962752832, 31.8378496674067684, 31.7383740966706895, 31.7092554096056247, 31.5163222252277677, 31.3387568936752174, 31.1661172140608471, 31.0340703333969294, 30.8241145688773095, 30.6289511237169130, 30.4411238200938499, 30.1045684195149157, 29.9632654415000381, 29.6805498271159891, 29.4894724678669320, 29.2784266334110796, 29.0530094521940327, 28.7596661792412007, 28.4991902123336587, 28.3650913174371198, 28.0606757645223155, 27.9399138231943276, 27.6397751444362854, 27.4337351775097673, 27.1965710650921579, 26.9269280685731083, 26.8536070522380825, 26.4918930933858370, 26.3777006456164145, 26.0685147454468265, 25.9081958635557648, 25.6992562146556800, 25.5952858306863789, 25.2609609116645615, 25.1649018491134697, 24.9215332633330249, 24.7560629941918329, 24.5963298464319706, 24.4721054786908212, 24.2499677975241106, 24.1427648088898117, 24.0257580725656084, 23.7984656369940453, 23.5655578240403720, 23.4886571754993909, 23.3251459824084435, 23.2766671490861583, 23.0264410657498928, 22.9896738704310906, 22.8302898990322518, 22.7016261240176895, 22.7090260963541972, 22.6181459230114221, 22.4795140227161632, 22.5011632555770404, 22.4752992012679407, 22.3359117201550141, 22.3064170738045107, 22.3627376015218573, 22.3159444439487942, 22.2848556623249330, 22.2016480534010654, 22.2048165674476827, 22.2379485106719876, 22.2542203992701779, 22.1714467563068105, 22.2785519590191861, 22.3482558829368578, 22.2790144786499020, 22.3008900410951298, 22.3695670464213237, 22.3174401430691667, 22.4454745059084324, 22.3979765075948620, 22.4430671610651551, 22.4754180149891987, 22.5371445377396427, 22.6005775068340249, 22.6905787537641146, 22.7358058940035299, 22.6947934303515346, 22.8583377184016356, 22.9184154817541064, 22.8628784869368218, 22.9331618410952522, 23.0397236527736240, 23.1320513372099725, 23.1678189042263369, 23.2012682391693232, 23.3313719687548264, 23.3843920841566160, 23.5073210370416312, 23.6021467822634854, 23.7392811769957000, 23.6587759328678224, 23.8463219269294768, 23.9541144534879855, 23.9338852603529979, 24.1004073322403833, 24.1171869329405482, 24.2610474100728446, 24.3558319911723231, 24.3515408477808393, 24.4067016932192224, 24.4261017266807450, 24.4665846071733561, 24.5939844622349746, 24.6636009410128700, 24.7152292992950748, 24.7643919066544385, 24.8166060680849192, 24.9581514964378890, 24.9419751102132103, 24.9576807804472978, 24.9001688129786096, 25.0430168057126785, 25.0598635655875057, 25.0503312830392488, 25.1129346600618995, 25.0561719799972948, 25.0695576208692614, 25.1646809291495046, 25.1643263472307162, 25.2424203478167577, 25.2011726161748371, 25.3329467408887687, 25.3223471857763052, 25.3387071180905004, 25.4221294975393448, 25.4346307016454531, 25.4263404829827309, 25.4142152432701529, 25.3701150568716507, 25.4406335154854695, 25.4162502753940558, 25.4657475013091386, 25.5108380881555519, 25.4204161855910016, 25.5155705692868153, 25.5910287901354145, 25.5243259401701827, 25.5702725561885273, 25.5493664948325332, 25.5336140878595899, 25.6781140837116624, 25.7007494937559713, 25.6276241691648217, 25.6385980808793263, 25.7370179756806401, 25.6744091280652746, 25.7215485217568123, 25.7373182232914139, 25.7983468190830365, 25.7720490661125794, 25.8105034937359399, 25.8277051055033589, 25.8853265689632615, 25.9177365955020740, 25.9589434428510586, 25.9125321271511169, 25.9841653051320023, 25.8833278252655141, 26.0217851395025974, 26.0736661920242945, 26.0095932131514331, 25.9729587477319122, 26.0911430092129919, 26.0563806074210653, 26.1079734411679034, 26.1719959501981165, 26.1592818637744919, 26.1732403072635691, 26.1620168294269249, 26.2401424499597482, 26.2306344578130215, 26.3812428495331659, 26.2645315457430222, 26.3087291351088695, 26.3989858478327442, 26.5038432337969709, 26.4664064957596104, 26.4854473738736864, 26.5031219523159791, 26.5068431030844671, 26.5758360747894535, 26.5744628660088331, 26.7044164643679025, 26.6421255066537661, 26.6722042599419495, 26.7092116346376223, 26.6829146986214987, 26.7512745675714285, 26.7366015993011636, 26.8350098366637795, 26.8302294285417346, 26.8095781391237047, 26.8555631607968621, 26.8348466751096915, 26.7666891580050716, 26.8655911281730617, 26.8129383467623654, 26.7653498392293052, 26.7180188567852461, 26.7649151328700015, 26.6949454708954050, 26.7632753507481809, 26.7299084934861426, 26.6596094117450164, 26.7839255825766394, 26.7358044039521481, 26.7491888690459341, 26.6850034972293884, 26.7921978586434548, 26.7799697533779621, 26.7563205207073409, 26.7924790440833718, 26.8561482909427873, 26.8137902440987190, 26.8932522107686864, 26.9012102881370474, 26.8172817210085981, 26.8813438892230181, 26.8666205466907648, 26.8338591719785029, 26.8267133228793320, 26.8470568839773307, 26.8146998004460215, 26.8262998128932644, 26.8639754919138838, 26.7822268329937749, 26.7213044222826781, 26.7574927822126369, 26.7102329640660159, 26.6362917635465628, 26.6771491245563830, 26.6412092982737470, 26.6451280998856674, 26.5330613730429263, 26.5577948760388196, 26.5086084861038458, 26.4881228608881898, 26.5245187674932374, 26.4936578318722979, 26.4330175883795562, 26.4664786850334828, 26.4313325804174077, 26.3587968505601644, 26.3970416946038355, 26.3203959075360636, 26.2367474519083963, 26.3920233715695112, 26.3627462759916860, 26.2753802963201117, 26.3793168685268853, 26.3872540154177244, 26.3183996628615979, 26.3264505307903605, 26.3080718341870785, 26.2726830403463580, 26.2760108777334302, 26.2598810394633695, 26.1926995614961626, 26.1914735574253967, 26.1870908588489506, 26.1313575756051932, 26.0277490698921596, 26.0730036637539655, 26.0420333906397623, 26.1524690107332347, 26.0372159907067875, 26.1079476792808194, 26.0793666313363701, 26.0663788495916187, 26.1311317795505396, 26.0537475197540687, 26.0195410428221372, 26.0359007633249107, 26.0614835727014089, 26.0010930521117025 ; ...
 0.3195386029000569, 0.4106930053965520, 0.6511750109974280, 0.6407713440402973, 0.7591663851397169, 0.7695835135795640, 0.8054218895053462, 0.8428813893122596, 0.8557877502987761, 0.9232567066047437, 0.8744150498075399, 0.8138116294927860, 0.8901257558406491, 0.8770012588828571, 0.9166014529102043, 0.9253685693313043, 0.9418147030657994, 0.9239140269237796, 0.8828536025382072, 0.8685202830891113, 0.9292503437820608, 0.9520204616563579, 0.9389678412546481, 0.8556825010302982, 0.8585401309514848, 0.7901271310585570, 0.7377006960759585, 0.8252742093955060, 0.7430887512466966, 0.7393119708284127, 0.7430951776766139, 0.6416886295275483, 0.6377540071404270, 0.5923877924366656, 0.6609637008738198, 0.5740479441457313, 0.4418883111832773, 0.5816689370486130, 0.5044860833331892, 0.3990641241784197, 0.3917796926255788, 0.4480831563719777, 0.3910344895612626, 0.3205736493309329, 0.3018459543336241, 0.2055683794021342, 0.2553804544664144, 0.2560044456809896, 0.1780778392746540, 0.1315021626848194, 0.0753245268521047, 0.1146728573617649, 0.0832577831147829, 0.0805244785709107, 0.0822480707932463, -0.0507272123612396, 0.0156700230477967, -0.0620476616900418, -0.1418695317729621, -0.1097628539719863, -0.0981242016685147, -0.1439233348247369, -0.0956818219763045, -0.1293767989845605, -0.2115257598827584, -0.1725302314920673, -0.2534683795178495, -0.3006419689080939, -0.1797725904341255, -0.2080380232948531, -0.2119027091210102, -0.3004391582708062, -0.2396725239768158, -0.1701876371949502, -0.3462549210517069, -0.2806639632775560, -0.2293084301367424, -0.2802744018905655, -0.2800418764333577, -0.3013766733393757, -0.2544426009737444, -0.3415672339461802, -0.2652562980332392, -0.2770159321686049, -0.2259333170844328, -0.2476070197548717, -0.1794067699445115, -0.3020444744786999, -0.2620023701650229, -0.2931249958258864, -0.3111357587705587, -0.2197428320718750, -0.2382994947842753, -0.2925124406373232, -0.2121729117689892, -0.1498027204197590, -0.1807533550233312, -0.1414558597624982, -0.1616171042325926, -0.2532124794175598, -0.1637932838633692, -0.1546412668384678, -0.2253828616421451, -0.1370943612413782, -0.1541138185733426, -0.1755426901982881, -0.2159950283689961, -0.0933563426564730, -0.1800718600714598, -0.1091754358467481, -0.1573014250229560, -0.0865987999251416, -0.1530044921938745, -0.0998808784541370, -0.0615820354728180, -0.0511531599314262, -0.1650337393195355, -0.0312226338753258, -0.0401117297682695, -0.0325585045862631, -0.0606385787321658, 0.0243626128894889, 0.0471736641296064, -0.0141021283192803, 0.0411317914053615, 0.0567241619584589, 0.0296909357808904, -0.1059158587860005, -0.0077577874352313, -0.0192436814316975, 0.0291461658467047, -0.0067358174023871, 0.0041300753711846, 0.0945891704921380, -0.0219289521494158, 0.0013367818307111, 0.1132593619461422, -0.0475963956714243, 0.0356021292606619, 0.1020282023671460, 0.1094201633122149, 0.0384265463369637, 0.0707027365442741, 0.0497965982768527, 0.0948433875645634, -0.0203162305365825, 0.0914894345975680, 0.1694615346695761, 0.1056475749686608, 0.0721410795177729, 0.1147450060161687, 0.0510723603934046, 0.0623989192041871, 0.0459564738239538, 0.0958756706625339, 0.1319042674071772, 0.1189994924363057, 0.0240330810722695, 0.1203502795854512, 0.1313239170854812, 0.1213950024271246, 0.0983477778273470, 0.0859113152324007, 0.0155427960237558, 0.1229481320693107, 0.1326953099541306, 0.0623322696891079, 0.1026477804858698, 0.1020701462003494, 0.0635021809615729, 0.0269668309985206, 0.0540637767250435, 0.0150673045437245, 0.1141262821229972, 0.0090396938307346, 0.0296739293631369, 0.0246692106467894, -0.0091393064086915, 0.0864874095634061, 0.0434515667846995, 0.0899492544226530, -0.0360209234061131, 0.0413846243869304, 0.0313591021745301, -0.0173019339085613, 0.0818223142630367, 0.0370265890645923, 0.0156713225765238, 0.0399267743297478, 0.0301775208019846, 0.0378815006599806, -0.0710890972040088, 0.0361380282657164, 0.0470450810009661, 0.0078072306707080, -0.0298165402934674, 0.0473626154540287, -0.0035012360995183, -0.0722292121499462, -0.0059657060960141, 0.0274278548431049, 0.0067222952541167, 0.0230510914494334, -0.0046544748712667, -0.0052421329548953, 0.1115600455654931, -0.0361741716513659, 0.0352611300070896, -0.0548021690149031, 0.0902870775694047, 0.0743885195546765, -0.0198998228168628, 0.0144583808225148, 0.1057228420450798, -0.0079662986663498, -0.0227621904959335, -0.0265108765904481, -0.0381521464975379, -0.0067372773894259, 0.0513750037229102, 0.0330476637607757, -0.0221191582312228, 0.0471287243292463, 0.0591872572823414, 0.0437346298968808, 0.0224047225530744, 0.0462344333527613, 0.0206710469578221, -0.0099222497457305, 0.0107837343129300, 0.0121712458972536, 0.0546285687039297, 0.0685405281435280, 0.0247209302670661, 0.0143039047361638, -0.0032738116537155, -0.0388445223251034, -0.0136995499258607, 0.0229747861548623, 0.0649750284601735, 0.0154459529317010, 0.0611294512929562, 0.0584122824680200, 0.0475133591761257, 0.0771208455672555, -0.0154577683808438, 0.0566545913359779, 0.0344607087906255, 0.0205497906327831, -0.0240239674915702, 0.0048230262390680, 0.0235703917415565, 0.0731090612841759, 0.0067361892941974, -0.0134133305405933, 0.0653540212662178, -0.0039962822012185, -0.0105484365191023, 0.0396906913197878, 0.0308363151446371, 0.0273892531099981, 0.0237950849080575, -0.0230434696761872, 0.0392354593529690, -0.0499250153826203, -0.0074760940258185, -0.0619405800092153, -0.0108169520435610, -0.0203455581902993, 0.0835468120301850, 0.0338837673289661, -0.0890846724042398, 0.0069056579893980, -0.0366741617191628, -0.0125013286207657, -0.0029656774646153, 0.0449083859157676, 0.0066344616085555, 0.0098591633377646, 0.0147747676282143, 0.0181827001444767, -0.0847814007887437, -0.0919829997864284, -0.0612087881877691, 0.0479039423364686, 0.0519987483145691, -0.0770316408621311, -0.0519310112376971, -0.0494200354094157, -0.0849592010147546, -0.0440279032946774, 0.0639745732840953, -0.0272402440377112, -0.0422976375958498, -0.0468503830860903, -0.0346606411508293, -0.0418380266283203, 0.0193556200851970, -0.0165805774848411, -0.0655992369889948, -0.0235924666074494, -0.0537112304003890, -0.0220527230029969, -0.0643593792116137, 0.0109284699635976, -0.0225979265511189, -0.0315497642838984, -0.1115513601457686, -0.0212303599741569, -0.0791439987270159, -0.0381059864486612, -0.0667283595730656, 0.0584905714952287, -0.0629460794181675, 0.0017757606559389, -0.1123080657077096, -0.0395283465873081, -0.0925161125440918, -0.0367298313856249, 0.0135944008802034, 0.0387215258498533, 0.0476201699133804, -0.0553360053239966, 0.0235393522942191, -0.0576366729998334, -0.0438940224656566, -0.0067616155991831, 0.0140057993108468, -0.0056281387489291, -0.0286363791272066, -0.0285686260472704, -0.1138545104533852, 0.0105546501330442, 0.0487319336807150, -0.0999550499953609, -0.0067854665311862, 0.0607844762038435, -0.0273329963274562, -0.0211379818588367, 0.0201845932205036, -0.0460676861006933, -0.0102651463906230, 0.0842967618265357, 0.0243016696288142, 0.0399366069719547, 0.0553943813889605, -0.0108153187058394, 0.0300700350003689, -0.0346948212194584, -0.0226474520785257 ; ...
 ] ...
);

% build options
options = struct( ...
    'timeStep', 0.0100000000000000, ...
    'zonotopeOrder', 100, ...
    'reductionTechnique', 'pca', ...
    'points', 1, ...
    'p_conf', 0.9990000000000000 ...
);

% build simResult
simRes = simResult( ...
{ [ 0.717 -0.347 0.213 0.729 0.293 0.098 ; 0.721 -0.035 0.212 0.818 0.403 16.320 ; 0.730 0.332 0.211 0.967 0.556 13.643 ; 0.737 0.666 0.218 1.165 0.685 8.950 ; 0.745 1.042 0.226 1.373 0.758 4.634 ; 0.747 1.357 0.237 1.661 0.792 2.811 ; 0.746 1.695 0.250 2.018 0.802 2.124 ; 0.742 2.008 0.267 2.415 0.821 2.194 ; 0.744 2.263 0.291 2.836 0.836 2.141 ; 0.745 2.537 0.317 3.270 0.861 2.003 ; 0.738 2.754 0.348 3.775 0.877 1.623 ; 0.731 2.975 0.381 4.333 0.880 1.250 ; 0.724 3.229 0.409 4.885 0.903 1.277 ; 0.715 3.422 0.438 5.493 0.900 0.684 ; 0.704 3.623 0.473 6.115 0.920 0.783 ; 0.693 3.828 0.510 6.797 0.923 -0.120 ; 0.683 3.948 0.549 7.477 0.933 -0.261 ; 0.665 4.071 0.593 8.174 0.931 -0.906 ; 0.645 4.235 0.634 8.912 0.924 -1.256 ; 0.625 4.331 0.677 9.651 0.906 -1.603 ; 0.610 4.448 0.717 10.405 0.897 -1.538 ; 0.589 4.539 0.763 11.162 0.881 -1.334 ; 0.571 4.620 0.813 11.950 0.867 -1.588 ; 0.548 4.675 0.858 12.742 0.851 -1.453 ; 0.524 4.740 0.902 13.540 0.848 -1.569 ; 0.502 4.766 0.951 14.299 0.828 -2.206 ; 0.477 4.784 0.998 15.084 0.800 -2.528 ; 0.456 4.830 1.042 15.886 0.757 -2.261 ; 0.431 4.761 1.093 16.684 0.738 -1.442 ; 0.408 4.696 1.139 17.455 0.730 -1.880 ; 0.384 4.658 1.184 18.203 0.718 -2.487 ; 0.357 4.589 1.234 18.973 0.686 -3.080 ; 0.332 4.551 1.280 19.718 0.636 -3.142 ; 0.306 4.480 1.328 20.426 0.621 -2.436 ; 0.284 4.415 1.369 21.148 0.600 -2.802 ; 0.256 4.316 1.414 21.865 0.556 -3.065 ; 0.234 4.252 1.458 22.542 0.515 -2.753 ; 0.208 4.158 1.500 23.183 0.489 -2.224 ; 0.183 4.077 1.538 23.826 0.458 -2.303 ; 0.158 3.938 1.582 24.475 0.444 -2.217 ; 0.132 3.804 1.623 25.058 0.417 -2.950 ; 0.106 3.695 1.661 25.605 0.387 -3.395 ; 0.086 3.546 1.700 26.146 0.360 -3.315 ; 0.066 3.422 1.736 26.718 0.325 -3.435 ; 0.045 3.281 1.766 27.220 0.284 -3.111 ; 0.023 3.145 1.799 27.699 0.269 -2.621 ; 0.007 2.949 1.829 28.173 0.240 -3.199 ; -0.011 2.777 1.857 28.596 0.219 -3.098 ; -0.029 2.669 1.886 28.970 0.183 -3.524 ; -0.047 2.562 1.915 29.355 0.143 -3.195 ; -0.062 2.412 1.940 29.702 0.121 -2.354 ; -0.074 2.201 1.967 30.051 0.103 -2.321 ; -0.089 2.061 1.987 30.380 0.070 -2.479 ; -0.105 1.907 2.004 30.657 0.064 -2.115 ; -0.118 1.794 2.022 30.926 0.056 -2.876 ; -0.129 1.646 2.040 31.149 0.034 -3.292 ; -0.143 1.461 2.056 31.392 0.006 -3.385 ; -0.152 1.345 2.067 31.612 -0.023 -3.233 ; -0.161 1.207 2.077 31.789 -0.046 -2.675 ; -0.168 1.060 2.091 31.907 -0.063 -2.324 ; -0.180 0.916 2.096 32.029 -0.081 -2.158 ; -0.184 0.770 2.108 32.135 -0.112 -2.093 ; -0.195 0.622 2.115 32.224 -0.128 -1.526 ; -0.202 0.510 2.117 32.258 -0.142 -1.495 ; -0.208 0.389 2.119 32.296 -0.166 -1.473 ; -0.215 0.290 2.124 32.317 -0.179 -1.252 ; -0.218 0.155 2.126 32.322 -0.200 -0.981 ; -0.227 0.016 2.129 32.284 -0.202 -0.750 ; -0.234 -0.083 2.128 32.225 -0.213 -1.268 ; -0.235 -0.221 2.129 32.179 -0.229 -1.161 ; -0.233 -0.338 2.124 32.110 -0.243 -1.096 ; -0.237 -0.420 2.122 32.005 -0.242 -0.692 ; -0.236 -0.522 2.118 31.898 -0.248 -1.023 ; -0.236 -0.605 2.112 31.803 -0.248 -0.810 ; -0.235 -0.685 2.105 31.680 -0.274 -1.121 ; -0.230 -0.799 2.095 31.531 -0.287 -0.228 ; -0.230 -0.882 2.090 31.357 -0.276 0.316 ; -0.225 -0.982 2.083 31.209 -0.284 -0.218 ; -0.221 -1.035 2.070 31.036 -0.297 0.192 ; -0.219 -1.097 2.060 30.811 -0.303 0.474 ; -0.214 -1.186 2.045 30.630 -0.292 0.655 ; -0.207 -1.200 2.031 30.391 -0.302 0.281 ; -0.201 -1.268 2.024 30.169 -0.283 1.026 ; -0.193 -1.312 2.008 29.932 -0.275 0.556 ; -0.186 -1.345 1.994 29.698 -0.260 0.242 ; -0.180 -1.325 1.980 29.473 -0.257 -0.215 ; -0.172 -1.316 1.967 29.258 -0.240 0.060 ; -0.165 -1.314 1.953 29.009 -0.244 -0.124 ; -0.163 -1.335 1.940 28.777 -0.254 0.225 ; -0.159 -1.326 1.927 28.521 -0.250 0.548 ; -0.157 -1.330 1.910 28.301 -0.249 0.804 ; -0.153 -1.327 1.897 28.075 -0.239 0.746 ; -0.147 -1.306 1.882 27.841 -0.251 0.503 ; -0.138 -1.335 1.865 27.655 -0.247 1.161 ; -0.134 -1.309 1.856 27.443 -0.247 1.331 ; -0.129 -1.293 1.840 27.192 -0.220 1.497 ; -0.121 -1.266 1.826 27.008 -0.201 0.926 ; -0.117 -1.311 1.818 26.787 -0.201 0.305 ; -0.113 -1.273 1.806 26.550 -0.192 0.309 ; -0.106 -1.312 1.789 26.338 -0.196 0.380 ; -0.097 -1.324 1.777 26.138 -0.198 0.680 ; -0.092 -1.342 1.767 25.920 -0.187 1.170 ; -0.085 -1.294 1.756 25.701 -0.169 1.060 ; -0.080 -1.264 1.744 25.522 -0.170 0.454 ; -0.075 -1.224 1.732 25.301 -0.165 0.747 ; -0.072 -1.168 1.723 25.130 -0.154 0.954 ; -0.064 -1.103 1.714 24.946 -0.150 0.765 ; -0.060 -1.039 1.706 24.758 -0.136 0.935 ; -0.053 -1.009 1.696 24.596 -0.137 0.795 ; -0.050 -0.992 1.688 24.407 -0.136 1.336 ; -0.045 -0.959 1.678 24.283 -0.122 1.347 ; -0.039 -0.962 1.670 24.102 -0.124 0.901 ; -0.032 -0.966 1.658 23.981 -0.102 1.467 ; -0.031 -0.967 1.648 23.809 -0.093 0.718 ; -0.022 -0.915 1.635 23.650 -0.092 0.409 ; -0.017 -0.869 1.622 23.482 -0.077 0.842 ; -0.014 -0.855 1.609 23.327 -0.082 0.653 ; -0.010 -0.810 1.603 23.203 -0.073 0.824 ; -0.002 -0.766 1.592 23.090 -0.060 0.805 ; 0.002 -0.751 1.586 22.988 -0.067 0.922 ; 0.009 -0.699 1.582 22.866 -0.047 1.076 ; 0.017 -0.622 1.576 22.749 -0.034 0.870 ; 0.022 -0.572 1.569 22.663 -0.021 0.664 ; 0.029 -0.559 1.564 22.583 -0.019 0.588 ; 0.033 -0.498 1.559 22.498 -0.004 0.779 ; 0.035 -0.476 1.557 22.440 -0.002 0.360 ; 0.034 -0.413 1.549 22.388 0.008 0.566 ; 0.036 -0.344 1.541 22.348 -0.008 0.060 ; 0.035 -0.338 1.538 22.341 0.003 0.935 ; 0.042 -0.281 1.537 22.319 -0.003 0.705 ; 0.043 -0.245 1.536 22.283 0.018 1.258 ; 0.049 -0.246 1.530 22.265 0.035 0.591 ; 0.050 -0.195 1.528 22.226 0.051 -0.003 ; 0.052 -0.155 1.526 22.237 0.050 -0.364 ; 0.056 -0.122 1.521 22.210 0.059 -0.185 ; 0.061 -0.039 1.518 22.200 0.042 -0.224 ; 0.062 -0.004 1.518 22.236 0.029 0.751 ; 0.065 0.033 1.519 22.255 0.038 1.487 ; 0.065 0.058 1.520 22.257 0.037 1.042 ; 0.061 0.079 1.519 22.279 0.053 1.303 ; 0.063 0.070 1.519 22.286 0.066 0.527 ; 0.060 0.116 1.516 22.341 0.064 0.107 ; 0.061 0.144 1.520 22.351 0.061 0.189 ; 0.061 0.143 1.524 22.395 0.046 0.444 ; 0.065 0.179 1.525 22.418 0.046 0.771 ; 0.064 0.188 1.528 22.446 0.064 1.296 ; 0.066 0.183 1.527 22.488 0.072 0.553 ; 0.067 0.171 1.528 22.562 0.087 0.183 ; 0.067 0.209 1.530 22.629 0.086 -0.316 ; 0.066 0.266 1.535 22.661 0.093 -0.254 ; 0.066 0.273 1.541 22.716 0.074 -0.382 ; 0.060 0.307 1.548 22.750 0.076 0.214 ; 0.063 0.318 1.551 22.806 0.087 0.240 ; 0.062 0.332 1.556 22.860 0.080 -0.280 ; 0.058 0.359 1.556 22.914 0.081 -0.034 ; 0.057 0.403 1.564 23.000 0.081 -0.096 ; 0.057 0.428 1.571 23.063 0.091 -0.048 ; 0.057 0.466 1.580 23.116 0.090 -0.395 ; 0.057 0.484 1.583 23.161 0.079 -0.280 ; 0.054 0.469 1.588 23.239 0.075 0.273 ; 0.050 0.467 1.594 23.309 0.082 0.291 ; 0.051 0.466 1.598 23.383 0.092 -0.271 ; 0.047 0.523 1.604 23.474 0.080 -0.469 ; 0.048 0.529 1.613 23.593 0.061 -0.074 ; 0.045 0.557 1.615 23.670 0.071 0.516 ; 0.041 0.536 1.619 23.754 0.061 0.168 ; 0.036 0.540 1.621 23.828 0.064 0.406 ; 0.034 0.506 1.626 23.923 0.057 0.160 ; 0.030 0.458 1.633 24.027 0.054 0.150 ; 0.031 0.394 1.635 24.087 0.050 0.117 ; 0.032 0.374 1.641 24.154 0.048 0.137 ; 0.031 0.368 1.643 24.219 0.035 0.016 ; 0.031 0.375 1.647 24.288 0.032 0.444 ; 0.031 0.410 1.652 24.329 0.025 0.686 ; 0.028 0.407 1.654 24.403 0.032 1.100 ; 0.026 0.416 1.655 24.469 0.035 0.806 ; 0.024 0.391 1.661 24.549 0.062 0.393 ; 0.022 0.382 1.666 24.630 0.058 -0.762 ; 0.022 0.365 1.666 24.676 0.036 -0.539 ; 0.022 0.302 1.674 24.734 0.031 0.011 ; 0.019 0.248 1.681 24.792 0.022 0.313 ; 0.019 0.200 1.684 24.858 0.017 0.430 ; 0.018 0.196 1.688 24.888 0.009 0.547 ; 0.021 0.166 1.687 24.926 0.001 0.636 ; 0.019 0.153 1.691 24.949 0.003 1.143 ; 0.019 0.168 1.693 24.979 0.023 0.859 ; 0.019 0.217 1.694 24.976 0.028 0.309 ; 0.015 0.180 1.695 25.020 0.015 0.042 ; 0.017 0.146 1.693 25.044 0.033 0.433 ; 0.018 0.145 1.694 25.083 0.017 -0.325 ; 0.016 0.094 1.694 25.100 0.022 0.362 ; 0.019 0.128 1.697 25.124 0.023 0.122 ; 0.016 0.133 1.698 25.131 0.031 0.019 ; 0.015 0.104 1.699 25.167 0.029 -0.312 ; 0.015 0.100 1.701 25.212 0.031 -0.360 ; 0.017 0.068 1.703 25.265 0.019 -0.409 ; 0.016 0.063 1.704 25.297 0.013 -0.060 ; 0.013 0.060 1.705 25.322 0.018 0.210 ; 0.015 0.050 1.702 25.361 0.014 -0.150 ; 0.011 0.024 1.703 25.386 0.029 0.100 ; 0.013 0.008 1.700 25.373 0.022 -0.633 ; 0.015 -0.015 1.700 25.410 0.023 -0.384 ; 0.016 -0.004 1.703 25.420 0.020 -0.259 ; 0.020 0.047 1.704 25.415 0.009 -0.112 ; 0.018 0.036 1.702 25.420 0.020 0.665 ; 0.020 0.003 1.705 25.464 0.026 -0.044 ; 0.020 0.021 1.707 25.504 0.018 -0.241 ; 0.022 0.004 1.705 25.478 0.017 0.184 ; 0.023 0.046 1.708 25.458 0.024 0.400 ; 0.021 0.092 1.709 25.496 0.019 0.131 ; 0.020 0.058 1.710 25.513 0.016 0.421 ; 0.017 0.110 1.709 25.545 0.020 0.311 ; 0.017 0.100 1.712 25.575 0.006 0.084 ; 0.018 0.086 1.713 25.588 0.021 0.595 ; 0.019 0.076 1.714 25.580 0.031 0.038 ; 0.016 0.106 1.712 25.584 0.028 -0.179 ; 0.012 0.078 1.717 25.616 0.024 -0.236 ; 0.016 0.058 1.717 25.657 0.024 -0.422 ; 0.018 0.088 1.720 25.674 0.008 -0.235 ; 0.017 0.091 1.719 25.682 0.011 0.485 ; 0.016 0.087 1.716 25.727 0.020 0.358 ; 0.018 0.140 1.714 25.745 0.006 -0.052 ; 0.017 0.131 1.715 25.787 0.018 0.647 ; 0.014 0.113 1.717 25.830 0.003 0.328 ; 0.013 0.112 1.721 25.831 0.012 0.733 ; 0.015 0.073 1.718 25.828 0.024 0.254 ; 0.012 0.085 1.717 25.862 0.037 -0.142 ; 0.011 0.088 1.717 25.856 0.021 -0.683 ; 0.011 0.088 1.717 25.881 -0.001 -0.324 ; 0.013 0.107 1.720 25.904 0.018 0.634 ; 0.014 0.099 1.717 25.921 0.030 -0.132 ; 0.015 0.086 1.717 25.930 0.029 -0.378 ; 0.015 0.080 1.716 25.931 0.031 -0.330 ; 0.011 0.115 1.719 25.960 0.021 -0.417 ; 0.014 0.137 1.719 25.984 0.022 -0.211 ; 0.013 0.098 1.720 26.023 0.022 -0.226 ; 0.016 0.101 1.719 26.019 0.024 -0.161 ; 0.015 0.109 1.718 26.042 0.019 -0.039 ; 0.014 0.086 1.720 26.096 0.008 0.126 ; 0.011 0.107 1.724 26.148 0.017 0.509 ; 0.008 0.150 1.728 26.155 0.025 0.140 ; 0.007 0.168 1.728 26.175 0.040 -0.448 ; 0.009 0.148 1.730 26.204 0.030 -0.818 ; 0.005 0.129 1.733 26.197 0.037 -0.580 ; 0.006 0.144 1.735 26.233 0.026 -1.012 ; 0.001 0.162 1.736 26.254 0.019 -0.448 ; 0.001 0.186 1.734 26.310 0.010 -0.502 ; -0.001 0.180 1.739 26.315 0.006 0.163 ; -0.003 0.205 1.745 26.339 0.003 -0.080 ; -0.008 0.180 1.747 26.372 0.018 0.132 ; -0.010 0.197 1.745 26.407 0.008 -0.740 ; -0.011 0.202 1.747 26.446 0.020 -0.329 ; -0.012 0.189 1.751 26.474 0.012 -0.746 ; -0.013 0.182 1.751 26.507 0.001 -0.561 ; -0.014 0.147 1.754 26.527 -0.007 -0.187 ; -0.014 0.155 1.758 26.546 0.008 -0.094 ; -0.015 0.170 1.755 26.575 0.003 -0.485 ; -0.019 0.155 1.756 26.619 0.002 -0.237 ; -0.017 0.134 1.758 26.648 0.017 -0.506 ; -0.015 0.144 1.761 26.684 0.009 -1.028 ; -0.011 0.156 1.764 26.698 0.017 -0.662 ; -0.015 0.154 1.768 26.695 -0.001 -0.757 ; -0.016 0.126 1.769 26.733 -0.012 -0.121 ; -0.016 0.059 1.770 26.782 -0.007 0.133 ; -0.013 0.069 1.768 26.806 -0.010 -0.235 ; -0.014 0.028 1.771 26.793 -0.013 0.037 ; -0.013 0.043 1.775 26.818 -0.012 -0.112 ; -0.013 0.028 1.774 26.854 -0.010 -0.101 ; -0.013 -0.032 1.775 26.832 -0.015 -0.079 ; -0.015 -0.011 1.776 26.812 -0.011 -0.152 ; -0.017 -0.030 1.776 26.787 -0.019 -0.335 ; -0.015 -0.072 1.776 26.764 -0.025 0.088 ; -0.013 -0.041 1.775 26.737 -0.030 0.167 ; -0.010 -0.012 1.774 26.699 -0.010 0.584 ; -0.012 -0.002 1.774 26.714 0.007 -0.019 ; -0.012 0.039 1.776 26.692 0.001 -0.724 ; -0.011 -0.011 1.774 26.725 0.001 -0.377 ; -0.011 -0.037 1.772 26.705 -0.005 -0.486 ; -0.011 -0.013 1.776 26.696 -0.004 -0.435 ; -0.015 0.010 1.772 26.690 -0.010 -0.356 ; -0.011 0.041 1.770 26.705 -0.014 -0.210 ; -0.011 0.062 1.773 26.722 -0.016 0.177 ; -0.013 0.095 1.774 26.708 -0.020 0.275 ; -0.014 0.096 1.772 26.755 -0.012 0.663 ; -0.016 0.109 1.771 26.787 -0.007 0.251 ; -0.016 0.083 1.777 26.782 -0.007 -0.249 ; -0.021 0.041 1.778 26.828 -0.004 -0.332 ; -0.022 0.012 1.777 26.848 -0.018 -0.574 ; -0.025 0.031 1.776 26.867 -0.012 -0.231 ; -0.028 0.039 1.779 26.880 -0.007 -0.464 ; -0.027 0.023 1.779 26.878 -0.008 -0.807 ; -0.026 -0.018 1.778 26.875 -0.006 -0.896 ; -0.027 -0.014 1.776 26.884 -0.020 -0.836 ; -0.023 -0.001 1.777 26.880 -0.019 -0.393 ; -0.024 0.006 1.776 26.873 -0.024 -0.010 ; -0.026 0.042 1.772 26.856 -0.024 -0.012 ; -0.025 -0.024 1.772 26.835 -0.020 -0.088 ; -0.023 -0.070 1.776 26.829 -0.020 -0.211 ; -0.020 -0.080 1.778 26.830 -0.005 -0.230 ; -0.018 -0.079 1.777 26.826 -0.018 -0.902 ; -0.021 -0.075 1.773 26.805 -0.017 -0.210 ; -0.022 -0.054 1.774 26.771 -0.013 -0.252 ; -0.025 -0.007 1.776 26.750 -0.024 -0.314 ; -0.024 -0.045 1.779 26.715 -0.026 -0.177 ; -0.024 -0.060 1.779 26.677 -0.028 -0.084 ; -0.025 -0.096 1.778 26.653 -0.025 0.149 ; -0.022 -0.143 1.779 26.636 -0.021 -0.181 ; -0.024 -0.127 1.777 26.623 -0.036 -0.374 ; -0.027 -0.168 1.779 26.597 -0.036 0.135 ; -0.023 -0.191 1.774 26.570 -0.030 -0.063 ; -0.018 -0.184 1.770 26.531 -0.018 -0.214 ; -0.017 -0.142 1.767 26.503 -0.020 -0.304 ; -0.019 -0.136 1.763 26.465 -0.013 -0.307 ; -0.015 -0.174 1.764 26.458 -0.036 -0.483 ; -0.018 -0.203 1.761 26.456 -0.042 0.465 ; -0.020 -0.151 1.761 26.435 -0.037 0.539 ; -0.020 -0.142 1.760 26.434 -0.036 0.157 ; -0.018 -0.153 1.757 26.396 -0.026 0.287 ; -0.016 -0.110 1.755 26.381 -0.031 0.022 ; -0.019 -0.091 1.751 26.348 -0.025 0.321 ; -0.015 -0.093 1.754 26.329 -0.018 0.068 ; -0.017 -0.088 1.752 26.327 -0.019 0.059 ; -0.015 -0.072 1.751 26.321 -0.022 -0.251 ; -0.013 -0.076 1.750 26.304 -0.025 -0.073 ; -0.013 -0.090 1.750 26.327 -0.024 0.473 ; -0.014 -0.055 1.748 26.339 -0.033 0.293 ; -0.014 -0.090 1.746 26.332 -0.032 0.698 ; -0.013 -0.118 1.744 26.317 -0.033 0.431 ; -0.016 -0.115 1.739 26.292 -0.031 0.581 ; -0.013 -0.134 1.742 26.292 -0.028 0.457 ; -0.013 -0.144 1.741 26.233 -0.020 0.234 ; -0.013 -0.112 1.739 26.224 -0.025 -0.043 ; -0.010 -0.115 1.735 26.188 -0.014 0.247 ; -0.007 -0.102 1.732 26.181 -0.026 -0.086 ; -0.003 -0.113 1.732 26.191 -0.019 0.571 ; 0.001 -0.114 1.734 26.145 -0.031 0.369 ; 0.003 -0.100 1.729 26.112 -0.023 1.011 ; 0.004 -0.126 1.726 26.095 -0.011 0.925 ; 0.007 -0.110 1.725 26.082 -0.012 0.453 ; 0.009 -0.113 1.719 26.106 -0.015 0.444 ; 0.011 -0.072 1.717 26.087 -0.005 0.602 ; 0.012 -0.068 1.717 26.090 -0.003 0.516 ; 0.012 -0.068 1.712 26.102 -0.014 0.616 ; 0.014 -0.063 1.712 26.097 0.005 0.796 ; 0.015 -0.043 1.714 26.123 0.014 0.308 ; 0.017 -0.084 1.715 26.083 0.019 0.099 ; 0.016 -0.020 1.715 26.063 0.015 -0.227 ; 0.016 0.002 1.713 26.069 0.015 -0.042 ; 0.019 -0.031 1.715 26.076 0.023 0.189 ; 0.020 -0.054 1.719 26.045 0.005 -0.245 ] }, ...
{ [ 0.000 ; 0.010 ; 0.020 ; 0.030 ; 0.040 ; 0.050 ; 0.060 ; 0.070 ; 0.080 ; 0.090 ; 0.100 ; 0.110 ; 0.120 ; 0.130 ; 0.140 ; 0.150 ; 0.160 ; 0.170 ; 0.180 ; 0.190 ; 0.200 ; 0.210 ; 0.220 ; 0.230 ; 0.240 ; 0.250 ; 0.260 ; 0.270 ; 0.280 ; 0.290 ; 0.300 ; 0.310 ; 0.320 ; 0.330 ; 0.340 ; 0.350 ; 0.360 ; 0.370 ; 0.380 ; 0.390 ; 0.400 ; 0.410 ; 0.420 ; 0.430 ; 0.440 ; 0.450 ; 0.460 ; 0.470 ; 0.480 ; 0.490 ; 0.500 ; 0.510 ; 0.520 ; 0.530 ; 0.540 ; 0.550 ; 0.560 ; 0.570 ; 0.580 ; 0.590 ; 0.600 ; 0.610 ; 0.620 ; 0.630 ; 0.640 ; 0.650 ; 0.660 ; 0.670 ; 0.680 ; 0.690 ; 0.700 ; 0.710 ; 0.720 ; 0.730 ; 0.740 ; 0.750 ; 0.760 ; 0.770 ; 0.780 ; 0.790 ; 0.800 ; 0.810 ; 0.820 ; 0.830 ; 0.840 ; 0.850 ; 0.860 ; 0.870 ; 0.880 ; 0.890 ; 0.900 ; 0.910 ; 0.920 ; 0.930 ; 0.940 ; 0.950 ; 0.960 ; 0.970 ; 0.980 ; 0.990 ; 1.000 ; 1.010 ; 1.020 ; 1.030 ; 1.040 ; 1.050 ; 1.060 ; 1.070 ; 1.080 ; 1.090 ; 1.100 ; 1.110 ; 1.120 ; 1.130 ; 1.140 ; 1.150 ; 1.160 ; 1.170 ; 1.180 ; 1.190 ; 1.200 ; 1.210 ; 1.220 ; 1.230 ; 1.240 ; 1.250 ; 1.260 ; 1.270 ; 1.280 ; 1.290 ; 1.300 ; 1.310 ; 1.320 ; 1.330 ; 1.340 ; 1.350 ; 1.360 ; 1.370 ; 1.380 ; 1.390 ; 1.400 ; 1.410 ; 1.420 ; 1.430 ; 1.440 ; 1.450 ; 1.460 ; 1.470 ; 1.480 ; 1.490 ; 1.500 ; 1.510 ; 1.520 ; 1.530 ; 1.540 ; 1.550 ; 1.560 ; 1.570 ; 1.580 ; 1.590 ; 1.600 ; 1.610 ; 1.620 ; 1.630 ; 1.640 ; 1.650 ; 1.660 ; 1.670 ; 1.680 ; 1.690 ; 1.700 ; 1.710 ; 1.720 ; 1.730 ; 1.740 ; 1.750 ; 1.760 ; 1.770 ; 1.780 ; 1.790 ; 1.800 ; 1.810 ; 1.820 ; 1.830 ; 1.840 ; 1.850 ; 1.860 ; 1.870 ; 1.880 ; 1.890 ; 1.900 ; 1.910 ; 1.920 ; 1.930 ; 1.940 ; 1.950 ; 1.960 ; 1.970 ; 1.980 ; 1.990 ; 2.000 ; 2.010 ; 2.020 ; 2.030 ; 2.040 ; 2.050 ; 2.060 ; 2.070 ; 2.080 ; 2.090 ; 2.100 ; 2.110 ; 2.120 ; 2.130 ; 2.140 ; 2.150 ; 2.160 ; 2.170 ; 2.180 ; 2.190 ; 2.200 ; 2.210 ; 2.220 ; 2.230 ; 2.240 ; 2.250 ; 2.260 ; 2.270 ; 2.280 ; 2.290 ; 2.300 ; 2.310 ; 2.320 ; 2.330 ; 2.340 ; 2.350 ; 2.360 ; 2.370 ; 2.380 ; 2.390 ; 2.400 ; 2.410 ; 2.420 ; 2.430 ; 2.440 ; 2.450 ; 2.460 ; 2.470 ; 2.480 ; 2.490 ; 2.500 ; 2.510 ; 2.520 ; 2.530 ; 2.540 ; 2.550 ; 2.560 ; 2.570 ; 2.580 ; 2.590 ; 2.600 ; 2.610 ; 2.620 ; 2.630 ; 2.640 ; 2.650 ; 2.660 ; 2.670 ; 2.680 ; 2.690 ; 2.700 ; 2.710 ; 2.720 ; 2.730 ; 2.740 ; 2.750 ; 2.760 ; 2.770 ; 2.780 ; 2.790 ; 2.800 ; 2.810 ; 2.820 ; 2.830 ; 2.840 ; 2.850 ; 2.860 ; 2.870 ; 2.880 ; 2.890 ; 2.900 ; 2.910 ; 2.920 ; 2.930 ; 2.940 ; 2.950 ; 2.960 ; 2.970 ; 2.980 ; 2.990 ; 3.000 ; 3.010 ; 3.020 ; 3.030 ; 3.040 ; 3.050 ; 3.060 ; 3.070 ; 3.080 ; 3.090 ; 3.100 ; 3.110 ; 3.120 ; 3.130 ; 3.140 ; 3.150 ; 3.160 ; 3.170 ; 3.180 ; 3.190 ; 3.200 ; 3.210 ; 3.220 ; 3.230 ; 3.240 ; 3.250 ; 3.260 ; 3.270 ; 3.280 ; 3.290 ; 3.300 ; 3.310 ; 3.320 ; 3.330 ; 3.340 ; 3.350 ; 3.360 ; 3.370 ; 3.380 ; 3.390 ; 3.400 ; 3.410 ; 3.420 ; 3.430 ; 3.440 ; 3.450 ; 3.460 ; 3.470 ; 3.480 ; 3.490 ] }, ...
[], ...
{}, ...
{} ...
);

end

% ------------------------------ END OF CODE ------------------------------
