import re

def preprocess_text(text: str) -> str:
    """
    Preprocess the input text by:
      1. Converting to lowercase.
      2. Removing HTML tags.
      3. Removing URLs and email addresses.
      4. Removing non-ASCII characters.
      5. Replacing '-' and '/' with a space.
      6. Replacing all punctuation marks with a space (and removing digits).
      7. Normalizing whitespace.
    
    Args:
        text (str): The original text.
    
    Returns:
        str: The cleaned text.
    """
    # 1. Convert to lowercase
    text = text.lower()
    
    # 2. Remove HTML tags
    text = re.sub(r'<[^>]+>', '', text)
    
    # 3. Remove URLs and email addresses
    text = re.sub(r'http\S+|www\.\S+', '', text)
    text = re.sub(r'\S+@\S+', '', text)
    
    # 4. Remove non-ASCII characters
    text = re.sub(r'[^\x00-\x7F]+', '', text)
    
    # 5. Replace '-' and '/' with a space
    text = text.replace('-', ' ').replace('/', ' ')
    
    # 6. Replace all punctuation (and digits) with a space by keeping only alphabetic characters and whitespace.
    text = re.sub(r'[^a-zA-Z\s]', ' ', text)
    
    # 7. Normalize whitespace (convert multiple spaces/newlines into a single space)
    text = re.sub(r'\s+', ' ', text).strip()
    
    return text

# Example usage:
if __name__ == "__main__":
    sample_text = """
    <html>
    <body>
    Hello World! Visit http://example.com or contact me@example.com.
    This is a sample-text/with numbers 12345 and non-ASCII: ü, é, ñ.
    </body>
    </html>
    """
    cleaned = preprocess_text(sample_text)
    print("Cleaned Text:")
    print(cleaned)

