Copyright 2022-2024

# AnonymousRepo1
Instance generator for various OPF problems (ACOPF & DCOPF currently supported)

- [AnonymousRepo1](#anonymousrepo1)
  - [Installation instructions](#installation-instructions)
    - [Using HSL solvers (ma27, ma57)](#using-hsl-solvers-ma27-ma57)
  - [Quick start](#quick-start)
    - [Generating random instances](#generating-random-instances)
    - [Building and solving OPF problems](#building-and-solving-opf-problems)
  - [Generating datasets](#generating-datasets)
  - [Solution format](#solution-format)
    - [ACOPF](#ACOPF)
    - [DCOPF](#DCOPF)
    - [SOCWRPowerModel](#socwrpowermodel)
  - [Datasets](#datasets)
    - [Format](#format)
    - [Loading from julia](#loading-from-julia)
    - [Loading from python](#loading-from-python)
  - [Loading and saving JSON files](#loading-and-saving-json-files)

## Installation instructions

This repository is a non-registered Julia package.

* Option 1: install as a Julia package. You can use it, but not modify the code
    ```julia
    using Pkg
    Pkg.add("git@github.com:AnonymousOrg/AnonymousRepo1.git")
    ```

* Option 2: clone the repository. Use this if you want to change the code
    ```bash
    git clone git@github.com:AnonymousOrg/AnonymousRepo1.git
    ```
    To use the package after cloning the repo
    ```bash
    $ cd AnonymousRepo1
    $ julia --project=.
    julia> using AnonymousRepo1
    ```

    If you are modifying the source code, it is recommened to use the package [`Revise.jl`](https://github.com/timholy/Revise.jl)
    so that you can use the changes without having to start Julia.
    Make sure you load `Revise` before loading `AnonymousRepo1` in your julia session.
    ```julia
    using Revise
    using AnonymousRepo1
    ```

### Using HSL solvers (ma27, ma57)

Please refer to [`Ipopt.jl` installation instructions](https://github.com/jump-dev/Ipopt.jl?tab=readme-ov-file#linear-solvers)
    for how to install non-default linear solvers, e.g., HSL, Pardiso, etc...
Note that a specific license may be required.

To use HSL linear solvers when solving OPF instances, set the parameter "linear_solver" to "ma27" or "ma57" in the config file.
The recommended solver for Ipopt is `ma27`.
```toml
solver.name = "Ipopt"
solver.attributes.linear_solver = "ma27"
```

## Quick start

### Generating random instances

```julia
using Random, PGLib, PowerModels
using AnonymousRepo1
PowerModels.silence()

rng = MersenneTwister(42)

old_data = make_basic_network(pglib("3_lmbd"))

# Load scaler using global scaling + uncorrelated LogNormal noise
config = Dict(
    "load" => Dict(
        "noise_type" => "ScaledLogNormal",
        "l" => 0.8,
        "u" => 1.2,
        "sigma" => 0.05,        
    )
)
opf_sampler  = SimpleOPFSampler(old_data, config)

# Generate a new instance
new_data = rand(rng, opf_sampler)

old_data["load"]["1"]["pd"]  # old 
1.1

new_data["load"]["1"]["pd"]  # new
1.1596456429775048
```

To generate multiple instances, run the above code in a loop
```julia
dataset = [
    AnonymousRepo1.rand(rng, opf_sampler)
    for i in 1:100
]
```

### Building and solving OPF problems

`AnonymousRepo1` supports multiple OPF formulations, based on [PowerModels](https://lanl-ansi.github.io/PowerModels.jl/stable/).

```julia
using PowerModels
using PGLib

using JuMP
using Ipopt

using AnonymousRepo1

data = make_basic_network(pglib("14_ieee"))
acopf = AnonymousRepo1.build_opf(ACOPF, data, Ipopt.Optimizer)
optimize!(acopf.model)
res = AnonymousRepo1.extract_result(acopf)

res["primal_objective_value"]  # should be close to 2178.08041
```

## Generating datasets

A script for generating multiple ACOPF instances is given in [`exp/sampler.jl`](exp/sampler.jl).

It is called from the command-line as follows:
```bash
julia --project=. exp/sampler.jl <path/to/config.toml> <seed_min> <seed_max>
```
where
* `<path/to/config.toml>` is a path to a valid configuration file in TOML format (see [`exp/config.toml`](exp/config.toml) for an example)
* `<seed_min>` and `<seed_max>` are minimum and maximum values for the random seed. Must be integer values.
    The script will generate instances for values `smin, smin+1, ..., smax-1, smax`.

### Loading from julia

```julia
using HDF5

D = h5read("dataset.h5", "/")  # read all the dataset into a dictionary
```

### Loading from python

The following code provides a starting point to load h5 datasets in python
```py

import h5py
import numpy as np


def parse_hdf5(path: str, preserve_shape: bool=False):
    dat = dict()

    def read_direct(dataset: h5py.Dataset):
        arr = np.empty(dataset.shape, dtype=dataset.dtype)
        dataset.read_direct(arr)
        return arr

    def store(name: str, obj):
        if isinstance(obj, h5py.Group):
            return
        elif isinstance(obj, h5py.Dataset):
            dat[name] = read_direct(obj)
        else:
            raise ValueError(f"Unexcepted type: {type(obj)} under name {name}. Expected h5py.Group or h5py.Dataset.")

    with h5py.File(path, "r") as f:
        f.visititems(store)

    if preserve_shape:
        # recursively correct the shape of the dictionary
        ret = dict()

        def r_correct_shape(d: dict, ret: dict):
            for k in list(d.keys()):
                if "/" in k:
                    k1, k2 = k.split("/", 1)
                    if k1 not in ret:
                        ret[k1] = dict()
                    r_correct_shape({k2: d[k]}, ret[k1])
                    del d[k]
                else:
                    ret[k] = d[k]

        r_correct_shape(dat, ret)

        return ret
    else:
        return dat
```
