Copyright AnonymousOrg 2024

# AnonymousRepo2

AnonymousRepo2 is a Python package for developing machine learning proxy models for optimal power flow. It is built on top of PyTorch and PyTorch Lightning, and is designed to be modular and extensible. The main components are:

- [`formulations`](anonymousrepo2/formulations): The main interface to the OPF formulations [ACOPF](anonymousrepo2/formulations/ac), [DCOPF](anonymousrepo2/formulations/dc), and [Economic Dispatch](anonymousrepo2/formulations/ed).

  - Each OPF formulation has three main component classes: `OPFProblem`, `OPFViolation`, and `OPFModel`. The `OPFProblem` class loads and parses data from disk, the `OPFViolation` class calculates constraints residuals, incidence matrices, objective value, etc., and the `OPFModel` class is an abstract base class for proxy models.

<!-- - [`functional`](anonymousrepo2/functional): The functional interface underlying the object-oriented interface in `formulations`, which makes less assumptions (i.e., all problem data may vary but the user is responsible for keeping track of it). -->

- [`loss_functions`](anonymousrepo2/loss_functions): Various loss functions including [LDFLoss](anonymousrepo2/loss_functions/ldf.py) and the self-supervised [ObjectiveLoss](anonymousrepo2/loss_functions/objective.py).

- [`layers`](anonymousrepo2/layers): Various feasibility recovery layers including [BoundRepair](anonymousrepo2/layers/bound_repair.py) and [HyperSimplexRepair](anonymousrepo2/layers/hypersimplex_repair.py).

- [`models`](anonymousrepo2/models): Various proxy models including [BasicNeuralNet](anonymousrepo2/models/basic_nn), [LDFNeuralNet](anonymousrepo2/models/ldf_nn), and [E2ELR](anonymousrepo2/models/e2elr).

- [`parsers`](anonymousrepo2/parsers): Parsers for data generated by [AnonymousOrg/AnonymousRepo1](https://github.com/AnonymousOrg/AnonymousRepo1). <!-- Note that when selecting the test set the seed is fixed to ensure reproducibility. -->

- [`viz`](anonymousrepo2/viz): Visualization helpers for plots and tables.

Documentation based on docstrings is live [here](https://anonymousorg.github.io/AnonymousRepo2/).


## Installation

To install `anonymousrepo2` on macOS (CPU/MPS) and Windows (CPU), run:
```bash
pip install git+ssh://git@github.com/AnonymousOrg/AnonymousRepo2.git

# or, to install with optional dependencies (options: "all", "dev", "viz"):
pip install "anonymousrepo2[all] @ git+ssh://git@github.com/AnonymousOrg/AnonymousRepo2.git"
```
If you don't already have PyTorch on Linux (CPU/CUDA/ROCm) or Windows (CUDA), make sure to provide the correct `--index-url` which you can find [here](https://pytorch.org/get-started/locally/). For example, to install from scratch with CUDA 12.1 and all optional dependencies:
```bash
pip install "anonymousrepo2[all] @ git+ssh://git@github.com/AnonymousOrg/AnonymousRepo2.git" \
     --index-url https://download.pytorch.org/whl/cu121                \
     --extra-index-url https://pypi.python.org/simple/
```

For development, the recommended installation method is using Conda environment files provided at [environment.yml](environment.yml) and [environment_cuda.yml](environment_cuda.yml). Using [Mambaforge](https://mamba.readthedocs.io/en/latest/mamba-installation.html#mamba-install) is recommended for super fast installation:
```bash
cd AnonymousRepo2                                  # cd into the repo
mamba env create -f environment.yml        # create the environment
conda activate anonymousrepo2                      # activate the environment
pip install -e ".[all]"                    # install AnonymousRepo2
# git lfs pull                             # download data if running tests
```

## Usage


### Training a `BasicNeuralNet` for ACOPF
```python
import torch

# load data
from anonymousrepo2 import ACProblem

data_path = ...

problem = ACProblem(data_path)

# make a basic neural network model
from anonymousrepo2.models.basic_nn import ACBasicNeuralNet # requires pytorch-lightning

config = {
    "optimizer": "adam",
    "learning_rate": 1e-3,
    "loss": "mse",
    "hidden_sizes": [500,300,500],
    "activation": "sigmoid",
    "boundrepair": "none" # optionally clamp outputs to bounds (choices: "sigmoid", "relu", "clamp")
}

model = ACBasicNeuralNet(config, problem)

model.train(trainer_kwargs={'max_epochs': 100, 'accelerator': 'auto'}) # pass args to the PyTorch Lightning Trainer

evals = model.evaluate_model()

from anonymousrepo2.viz import make_stats_df
print(make_stats_df(evals))

model.save_checkpoint("./basic_300bus") # creates a folder called "basic_300bus" with a file "trainer.ckpt" in it.
```

### Manually Loading Data
```python
import torch

from anonymousrepo2 import ACProblem

data_path = ...

# parse HDF5/JSON
problem = ACProblem(data_path)

# get train/test set:
train_data = problem.train_data
test_data = problem.test_data

train_data['input/pd'].shape # torch.Size([52863, 201])
test_data['input/pd'].shape # torch.Size([5000, 201])

# if needed, convert the HDF5 data to a tree dictionary instead of a flat dictionary:
from anonymousrepo2.parsers import H5Parser
h5_tree = H5Parser.make_tree(train_data) # this tree structure should
                                         # exactly mimic the
                                         # structure of the HDF5 file.
h5_tree['input']['pd'].shape # torch.Size([52863, 201])
```
