import os
import glob
import pickle
import json
import numpy as np
from concurrent.futures import ProcessPoolExecutor, as_completed


def convert_pickle_to_npz_and_json(pickle_path: str, output_dir: str) -> str:
    """
    Converts a single pickle file containing a list of dictionaries 
    to two files in output_dir:
    
    1) 'metadata.json' containing everything *except* the mask arrays.
    2) 'masks.npz' containing all mask arrays, keyed by a unique string.
    """
    try:
        with open(pickle_path, 'rb') as f:
            data = pickle.load(f)  # data is presumably a list of length ~700

        metadata = []
        masks_dict = {}

        for i, frame_dict in data.items():
            frame_metadata = {}
            for obj_key, obj_info in frame_dict.items():
                mask_array = obj_info['mask']
                mask_key = f"{i}_{obj_key}"
                masks_dict[mask_key] = mask_array

                obj_info_without_mask = {k: v for k, v in obj_info.items() if k != 'mask'}
                frame_metadata[obj_key] = obj_info_without_mask

            metadata.append(frame_metadata)

        os.makedirs(output_dir, exist_ok=True)

        metadata_json_path = os.path.join(output_dir, "metadata.json")
        with open(metadata_json_path, 'w') as f:
            json.dump(metadata, f)

        masks_npz_path = os.path.join(output_dir, "masks.npz")
        # Changed from np.savez_compressed to np.savez to avoid compression
        np.savez(masks_npz_path, **masks_dict)

        return f"Success: {pickle_path}"
    
    except Exception as e:
        return f"Failed: {pickle_path} | Error: {e}"


def main():
    original_dir = #TODO
    new_base_dir = #TODO

    os.makedirs(new_base_dir, exist_ok=True)

    pickle_files = glob.glob(os.path.join(original_dir, "*.pkl"))
    
    failed = []

    with ProcessPoolExecutor(max_workers=80) as executor:
        future_to_file = {
            executor.submit(
                convert_pickle_to_npz_and_json, 
                pkl_path, 
                os.path.join(new_base_dir, os.path.splitext(os.path.basename(pkl_path))[0])
            ): pkl_path
            for pkl_path in pickle_files
        }

        for future in as_completed(future_to_file):
            result = future.result()
            if "Failed" in result:
                failed.append(result)
            print(result)

    print("Conversion finished!")
    if failed:
        print("Failed files:")
        for f in failed:
            print(f)


if __name__ == "__main__":
    main()
