# TAID: Temporally Adaptive Interpolated Distillation for Efficient Knowledge Transfer in Language Models

This is an official Pytorch implementation of "TAID: Temporally Adaptive Interpolated Distillation for Efficient Knowledge Transfer in Language Models".

## Models

We developed two models developed using TAID. Both models have achieved state-of-the-art performance in their respective size categories for large language models (LLMs) and vision-language models (VLMs), underscoring
TAID’s versatility and effectiveness in real-world applications.

### TAID-LLM-1.5B

TAID-LLM-1.5B is a new state-of-the-art model for LLMs with fewer than 2B parameters.
We distilled from [Qwen-72B-Instruct](https://huggingface.co/Qwen/Qwen2-72B-Instruct) to [Qwen-1.5B-Instruct](https://huggingface.co/Qwen/Qwen2-1.5B-Instruct).

The model weights are available [here](https://drive.google.com/drive/folders/1J5ZBDffCawtM3MobvO39phUFoCnCTDBb?usp=sharing).

**Quickstart:**

<details>

Please replace `path-to-model` with the path downloaded the model.

```python
from transformers import AutoModelForCausalLM, AutoTokenizer
device = "cuda" # the device to load the model onto

model = AutoModelForCausalLM.from_pretrained(
    "path-to-model",
    torch_dtype="auto",
    device_map="auto"
)
tokenizer = AutoTokenizer.from_pretrained("path-to-model")

prompt = "Give me a short introduction to large language model."
messages = [
    {"role": "system", "content": "You are a helpful assistant."},
    {"role": "user", "content": prompt}
]
text = tokenizer.apply_chat_template(
    messages,
    tokenize=False,
    add_generation_prompt=True
)
model_inputs = tokenizer([text], return_tensors="pt").to(device)

generated_ids = model.generate(
    model_inputs.input_ids,
    max_new_tokens=512
)
generated_ids = [
    output_ids[len(input_ids):] for input_ids, output_ids in zip(model_inputs.input_ids, generated_ids)
]

response = tokenizer.batch_decode(generated_ids, skip_special_tokens=True)[0]
```

</details>

### TAID-VLM-2B

TAID-VLM-2B is a new state-of-the-art model for VLMs up to 4B parameters. We distilled from [InternVL2-8B](https://huggingface.co/OpenGVLab/InternVL2-8B) to [InternVL2-2B](https://huggingface.co/OpenGVLab/InternVL2-2B). 

The model weights are available [here](https://drive.google.com/drive/folders/1_xNcKPy65J4bySbud6TN3MwtvkmssOZa?usp=sharing).

**Quickstart:**

<details>

Please replace `path-to-model` with the path downloaded the model.

```python
import numpy as np
import torch
import torchvision.transforms as T
from decord import VideoReader, cpu
from PIL import Image
from torchvision.transforms.functional import InterpolationMode
from transformers import AutoModel, AutoTokenizer

IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)

def build_transform(input_size):
    MEAN, STD = IMAGENET_MEAN, IMAGENET_STD
    transform = T.Compose([
        T.Lambda(lambda img: img.convert('RGB') if img.mode != 'RGB' else img),
        T.Resize((input_size, input_size), interpolation=InterpolationMode.BICUBIC),
        T.ToTensor(),
        T.Normalize(mean=MEAN, std=STD)
    ])
    return transform

def find_closest_aspect_ratio(aspect_ratio, target_ratios, width, height, image_size):
    best_ratio_diff = float('inf')
    best_ratio = (1, 1)
    area = width * height
    for ratio in target_ratios:
        target_aspect_ratio = ratio[0] / ratio[1]
        ratio_diff = abs(aspect_ratio - target_aspect_ratio)
        if ratio_diff < best_ratio_diff:
            best_ratio_diff = ratio_diff
            best_ratio = ratio
        elif ratio_diff == best_ratio_diff:
            if area > 0.5 * image_size * image_size * ratio[0] * ratio[1]:
                best_ratio = ratio
    return best_ratio

def dynamic_preprocess(image, min_num=1, max_num=12, image_size=448, use_thumbnail=False):
    orig_width, orig_height = image.size
    aspect_ratio = orig_width / orig_height

    # calculate the existing image aspect ratio
    target_ratios = set(
        (i, j) for n in range(min_num, max_num + 1) for i in range(1, n + 1) for j in range(1, n + 1) if
        i * j <= max_num and i * j >= min_num)
    target_ratios = sorted(target_ratios, key=lambda x: x[0] * x[1])

    # find the closest aspect ratio to the target
    target_aspect_ratio = find_closest_aspect_ratio(
        aspect_ratio, target_ratios, orig_width, orig_height, image_size)

    # calculate the target width and height
    target_width = image_size * target_aspect_ratio[0]
    target_height = image_size * target_aspect_ratio[1]
    blocks = target_aspect_ratio[0] * target_aspect_ratio[1]

    # resize the image
    resized_img = image.resize((target_width, target_height))
    processed_images = []
    for i in range(blocks):
        box = (
            (i % (target_width // image_size)) * image_size,
            (i // (target_width // image_size)) * image_size,
            ((i % (target_width // image_size)) + 1) * image_size,
            ((i // (target_width // image_size)) + 1) * image_size
        )
        # split the image
        split_img = resized_img.crop(box)
        processed_images.append(split_img)
    assert len(processed_images) == blocks
    if use_thumbnail and len(processed_images) != 1:
        thumbnail_img = image.resize((image_size, image_size))
        processed_images.append(thumbnail_img)
    return processed_images

def load_image(image_file, input_size=448, max_num=12):
    image = Image.open(image_file).convert('RGB')
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(image, image_size=input_size, use_thumbnail=True, max_num=max_num)
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values

# If you want to load a model using multiple GPUs, please refer to the `Multiple GPUs` section.
path = 'path-to-model'
model = AutoModel.from_pretrained(
    path,
    torch_dtype=torch.bfloat16,
    low_cpu_mem_usage=True,
    use_flash_attn=True,
    trust_remote_code=True).eval().cuda()
tokenizer = AutoTokenizer.from_pretrained(path, trust_remote_code=True, use_fast=False)

# set the max number of tiles in `max_num`
pixel_values = load_image('./examples/image1.jpg', max_num=12).to(torch.bfloat16).cuda()
generation_config = dict(max_new_tokens=1024, do_sample=True)

# pure-text conversation 
question = 'Hello, who are you?'
response, history = model.chat(tokenizer, None, question, generation_config, history=None, return_history=True)
print(f'User: {question}\nAssistant: {response}')
```

</details>

## Reproducing Instruction-tuning

### 1. Installation

```bash
pip install torch==2.2.2 --index-url https://download.pytorch.org/whl/cu121
pip install -r requirements.txt
git clone https://github.com/Dao-AILab/flash-attention.git
cd flash-attention
python setup.py install

# make sure to login huggingface and wandb
huggingface-cli login
wandb login
```

We conducted our experiments in the following environment: Python Version 3.10.12 and CUDA Version 12.3 on 8 * A100 80GB.

### 2. Data Preparation

This is the script to prepare data for [Phi-3-mini](https://huggingface.co/microsoft/Phi-3-mini-4k-instruct).

```bash
python prepare_ultrachat.py --model_type phi-3 --output_dir data
```

### 3. Train

We provide bash scripts in the [scripts](./scripts) directory, covering all experimental setups used in this paper. For example, the scripts for the experiments distilling from Phi-3-mini to TinyLlama can be found in [scripts/phi-3](./scripts/phi-3) directory. For instance, running the following command will execute training with TAID.

```bash
bash scripts/phi-3/taid.sh
```

### 4. Evaluation

Our evaluation uses MT-Bench, a benchmark designed to evaluate model's instruction-following ability, with scoring conducted by GPT-4.

```bash
git clone https://github.com/lm-sys/FastChat.git
cd FastChat
git checkout 5095615810cf613dba7f27dd155f571fcff976d8
# To support Phi-3-mini and StableLM-Zephyr-3B
cd ..
cp scripts/fastchat/conversation.py FastChat/fastchat/conversation.py
cp scripts/fastchat/model/model_adapter.py FastChat/fastchat/model/model_adapter.py
```

To evaluate the trained models in MT-Bench, you need to convert Lightning's checkpoints to HuggingFace format. This scripts creates `hf_model` foloder, the HF-compatible model, in the best checkpoint folder.

```bash
TEACHER="microsoft/Phi-3-mini-4k-instruct"
STUDENT="TinyLlama/TinyLlama_v1.1"
CKPT="logs/phi-3-xxx"
python convert_l_to_hf.py --tokenizer_path $TEACHER --model_path $STUDENT --ckpt $CKPT
```

Please follow [the official instruction](https://github.com/lm-sys/FastChat/tree/5095615810cf613dba7f27dd155f571fcff976d8/fastchat/llm_judge#mt-bench) to evaluate on MT-Bench. The `hf_model` folder goes to `--model-path`.

## Acknowledgement

We would like to thank the developers of the source models for their contributions and for making their work available. 
The implmentation of baseline losses in [distil_losses](./src/distil_losses) and the sampling generation in [sampler.py](./src/sampler.py) is based on the following repositories, and we are grateful for their work.

- [DistiLLM: Towards Streamlined Distillation for Large Language Models (ICML 2024)](https://github.com/jongwooko/distillm)
- [MiniLLM: Knowledge Distillation of Large Language Models (ICLR 2024)](https://github.com/microsoft/LMOps/tree/main/minillm)
