import datetime
import multiprocessing
import socket
import json

from ConfigSpace import ConfigurationSpace, UniformFloatHyperparameter, Constant
from smac.scenario import Scenario
from smac.facade.hyperparameter_optimization_facade import HyperparameterOptimizationFacade

import sys
from pathlib import Path

# Add the base path to the Python module search path
base_path = Path(__file__).resolve().parents[2]
sys.path.append(str(base_path))

from scheduling.helper_functions import load_parameters
from scheduling.run_genetic_algorithm import initialize_run, run_algo

PARAMETERS_FILE = '/hpc/za64617/projects/GNN4APC_dev/code/src/benchmarking/config_files/tune_smac3_scheduling.json'
DEFAULT_RESULTS_ROOT = "smac_tuning_results/"


def tune_algo(config, instance, seed):
    parameters = {'problem_instance': instance,
                  'nr_of_objectives': config['nr_of_objectives'],
                  'population_size': config['population_size'],
                  'ngen': config['ngen'],
                  'indpb': config['indpb'],
                  'cr': config['cr'],
                  'logbook': False,
                  'plotting': False}

    pool = multiprocessing.Pool()
    population, toolbox, stats, hof, jobShopEnv = initialize_run(pool, **parameters)
    hypervolume = run_algo(jobShopEnv, population, toolbox, None, None, stats, hof, **parameters)
    return -hypervolume


def main(param_file=PARAMETERS_FILE):
    parameters = load_parameters(param_file)
    folder = DEFAULT_RESULTS_ROOT

    # Define Configuration Space
    cr = UniformFloatHyperparameter("cr", float(parameters['min_cr']), float(parameters['max_cr']))
    indpb = UniformFloatHyperparameter("indpb", float(parameters['min_indpb']), float(parameters['max_indpb']))

    # Define Static Parameters
    nr_of_objectives = Constant("nr_of_objectives", parameters["nr_of_objectives"])
    population_size = Constant("population_size", parameters["population_size"])
    ngen = Constant("ngen", parameters["ngen"])

    cs = ConfigurationSpace()
    cs.add_hyperparameters([cr, indpb, nr_of_objectives, population_size, ngen])

    job_name = parameters['job_name'] + '_' + datetime.datetime.now().strftime("%b%d_%H-%M-%S")
    scenario = Scenario(cs, deterministic=False, n_trials=1000000, walltime_limit=parameters["walltime_limit"], name=job_name, output_directory=folder, instances=parameters['problem_instances'], n_workers=5)
    smac = HyperparameterOptimizationFacade(scenario, tune_algo)
    best_found_config = smac.optimize()
    print(' ')
    print('Done:')
    print(best_found_config)

    # Save the best configuration to a JSON file
    config_filename = f"{folder}/{job_name}/best_config.json"
    with open(config_filename, 'w') as file:
        json.dump(dict(best_found_config), file, indent=4)


if __name__ == '__main__':
    main()