import gym

from models.tf.tf_modelv2 import TFModelV2
from models.tf.fcnet import FullyConnectedNetwork
from models.tf.transformer import TransformerNetwork
from utils.framework import try_import_tf
from utils.torch_utils import FLOAT_MIN
from utils.typing import ModelConfigDict

tf1, tf, tfv = try_import_tf()


class TFActionMaskModel(TFModelV2):
    """PyTorch version of above ActionMaskingModel."""

    def __init__(
        self,
        obs_space: gym.spaces.Space,
        action_space: gym.spaces.Space,
        num_outputs: int,
        model_config: ModelConfigDict,
        name: str,
    ):
        super(TFActionMaskModel, self).__init__(
            obs_space, action_space, num_outputs, model_config, name)
        orig_space = getattr(obs_space, "original_space", obs_space)
        assert (
            isinstance(orig_space, gym.spaces.Dict)
            and "action_mask" in orig_space.spaces
            and "observations" in orig_space.spaces
        )
        # if "type" in model_config["custom_model_config"]:
        #     if "tf_transformer" == model_config["custom_model_config"]["type"]:
        #         internal_model = TransformerNetwork
        #     elif "tf_fcnet" == model_config["custom_model_config"]["type"]:
        #         internal_model = FullyConnectedNetwork
        #     else:
        #         raise NotImplementedError
        
        self.internal_model = TransformerNetwork(
            orig_space["observations"],
            action_space,
            num_outputs,
            model_config,
            name + "_internal",
        )
        self.register_variables(self.internal_model.variables())

        # disable action masking --> will likely lead to invalid actions
        self.no_masking = False
        if "no_masking" in model_config["custom_model_config"]:
            self.no_masking = model_config["custom_model_config"]["no_masking"]

    def forward(self, input_dict, state, seq_lens):
        # Extract the available actions tensor from the observation.
        action_mask = input_dict["obs"]["action_mask"]

        # Compute the unmasked logits.
        logits, _ = self.internal_model({"obs": input_dict["obs"]["observations"]})

        # If action masking is disabled, directly return unmasked logits
        if self.no_masking:
            return logits, state

        # Convert action_mask into a [0.0 || -inf]-type mask.
        inf_mask = tf.clip_by_value(tf.math.log(action_mask), clip_value_min=FLOAT_MIN, clip_value_max=0.0)
        masked_logits = logits + inf_mask

        # Return masked logits.
        return masked_logits, state

    def value_function(self):
        return self.internal_model.value_function()
    
    # TODO: add partial import code
    def import_partially(self, checkpoint):
        load_dict = self.internal_model.load_weights(checkpoint)
        return load_dict
