import torch
from jaxtyping import Float
from torch import Tensor, nn
from torch_geometric.data import Batch

from src.utils.sparse_utils import make_sparse_matrix, pad_values_of_sparse_matrix


class Ancestor(nn.Module):
    """Probabilistic Ancestors Features."""

    def __init__(self) -> None:
        """Initialize Probabilistic Ancestors Features."""
        super().__init__()

    @property
    def d(self) -> int:
        """Return dimension of Probabilistic Ancestors Features.

        :return: Dimension of Probabilistic Ancestors Features.
        """
        return 2

    def forward(
        self, data: Batch, fully_connected_index: Float[Tensor, "2 n_fully_connected_edges"]
    ) -> Float[Tensor, "n_edges 2"]:
        """Compute the Adjacency Matrix Features.

        :param data: PyG batch object.
        :return: Adjacency Matrix Features.
        """
        p_anc = make_sparse_matrix(data.edge_index, data.p_anc)
        p_anc_feature = pad_values_of_sparse_matrix(p_anc, fully_connected_index)
        p_anc_transposed_feature = pad_values_of_sparse_matrix(p_anc.T, fully_connected_index)
        return torch.stack([p_anc_feature, p_anc_transposed_feature], dim=1)
