import os
import random
import time
from dataclasses import dataclass, field
import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
import metaworld
from torch.optim import Optimizer
from math import sqrt

@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 1
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""

    # Algorithm specific arguments
    env_id: str = "mt4"
    """the environment id of the task"""
    env_names: list = field(
        default_factory=lambda: ['window-open-v2', 'window-close-v2', 'drawer-open-v2', 'drawer-close-v2'])
    """the environment id of the task"""
    total_timesteps: int = 500000
    """total timesteps of the experiments"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(1e6)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 0.005
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 256 # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 1e4
    """timestep to start learning"""
    policy_lr: float = 3e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 1e-3
    """the learning rate of the Q network network optimizer"""
    policy_frequency: int = 2
    """the frequency of training policy (delayed)"""
    target_network_frequency: int = 1  # Denis Yarats' implementation delays this by 2.
    """the frequency of updates for the target nerworks"""
    noise_clip: float = 0.5
    """noise clip parameter of the Target Policy Smoothing Regularization"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    average_num: int = 100
    """the number of episode to compute average success rate"""
    threshold: float = 0.9
    """the threshold representing finishing training"""
    cycle_num: int = 2
    """the number of cycling"""


def make_env(env_name, gamma):
    def thunk():
        mt1 = metaworld.MT1(env_name)  # Construct the benchmark, sampling tasks
        env = mt1.train_classes[env_name]()  # Create an environment with task `pick_place`
        task = mt1.train_tasks[0]
        env.set_task(task)  # Set task
        env = gym.wrappers.RecordEpisodeStatistics(env)
        return env

    return thunk


# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod() + np.prod(env.single_action_space.shape), 768)
        self.fc2 = nn.Linear(768, 768)
        self.fc3 = nn.Linear(768, 768)
        self.fc4 = nn.Linear(768, 1)
        self.layers = [self.fc1, self.fc2, self.fc3]
        self.layers_plus = [self.fc4]

    def forward(self, x, a):
        x = torch.cat([x, a], 1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = F.relu(self.fc3(x))
        x = self.fc4(x)
        return x

    def forward_with_feature(self, x, a):
        x = torch.cat([x, a], 1)
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        x3 = F.relu(self.fc3(x2))
        x4 = self.fc4(x3)
        return x4, [x1, x2, x3]

LOG_STD_MAX = 10
LOG_STD_MIN = -10

class Actor(nn.Module):
    def __init__(self, env):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod(), 768)
        self.fc2 = nn.Linear(768, 768)
        self.fc3 = nn.Linear(768, 768)
        self.fc_mean = nn.Linear(768, np.prod(env.single_action_space.shape))
        self.fc_logstd = nn.Linear(768, np.prod(env.single_action_space.shape))
        # action rescaling
        self.register_buffer(
            "action_scale", torch.tensor((env.action_space.high - env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.register_buffer(
            "action_bias", torch.tensor((env.action_space.high + env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.layers = [self.fc1, self.fc2, self.fc3]
        self.layers_plus = [self.fc_mean
                            , self.fc_logstd
                           ]

    def forward(self, x):
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        x3 = F.relu(self.fc3(x2))
        mean = self.fc_mean(x3)
        log_std = self.fc_logstd(x3)
        log_std = torch.tanh(log_std)
        log_std = LOG_STD_MIN + 0.5 * (LOG_STD_MAX - LOG_STD_MIN) * (log_std + 1)  # From SpinUp / Denis Yarats

        return mean, log_std, [x1, x2, x3]

    def get_action(self, x):
        mean, log_std, _ = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean
    
    def get_action_with_feature(self, x):
        mean, log_std, feature = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean, feature


class AdamGnT(Optimizer):
    def __init__(self, params, lr=1e-3, betas=(0.9, 0.999), eps=1e-8,
                 weight_decay=0, amsgrad=False):
        if not 0.0 <= lr:
            raise ValueError("Invalid learning rate: {}".format(lr))
        if not 0.0 <= eps:
            raise ValueError("Invalid epsilon value: {}".format(eps))
        if not 0.0 <= betas[0] < 1.0:
            raise ValueError("Invalid beta parameter at index 0: {}".format(betas[0]))
        if not 0.0 <= betas[1] < 1.0:
            raise ValueError("Invalid beta parameter at index 1: {}".format(betas[1]))
        defaults = dict(lr=lr, betas=betas, eps=eps,
                        weight_decay=weight_decay, amsgrad=amsgrad)
        super(AdamGnT, self).__init__(params, defaults)

    def __setstate__(self, state):
        super(AdamGnT, self).__setstate__(state)
        for group in self.param_groups:
            group.setdefault('amsgrad', False)

    def step(self, closure=None):
        """Performs a single optimization step.

        Arguments:
            closure (callable, optional): A closure that reevaluates the model
                and returns the loss.
        """
        loss = None
        if closure is not None:
            loss = closure()

        for group in self.param_groups:
            for p in group['params']:
                if p.grad is None:
                    continue
                grad = p.grad.data
                if grad.is_sparse:
                    raise RuntimeError('Adam does not support sparse gradients, please consider SparseAdam instead')
                amsgrad = group['amsgrad']

                state = self.state[p]

                # State initialization
                if len(state) == 0:
                    # state['step'] = 0
                    state['step'] = torch.zeros_like(p.data)
                    # Exponential moving average of gradient values
                    state['exp_avg'] = torch.zeros_like(p.data)
                    # Exponential moving average of squared gradient values
                    state['exp_avg_sq'] = torch.zeros_like(p.data)
                    if amsgrad:
                        # Maintains max of all exp. moving avg. of sq. grad. values
                        state['max_exp_avg_sq'] = torch.zeros_like(p.data)

                exp_avg, exp_avg_sq = state['exp_avg'], state['exp_avg_sq']
                if amsgrad:
                    max_exp_avg_sq = state['max_exp_avg_sq']
                beta1, beta2 = group['betas']

                state['step'] += 1

                if group['weight_decay'] != 0:
                    grad.add_(p.data, alpha=group['weight_decay'])

                # Decay the first and second moment running average coefficient
                exp_avg.mul_(beta1).add_(grad, alpha=1 - beta1)
                exp_avg_sq.mul_(beta2).addcmul_(grad, grad, value=1 - beta2)
                if amsgrad:
                    # Maintains the maximum of all 2nd moment running avg. till now
                    torch.max(max_exp_avg_sq, exp_avg_sq, out=max_exp_avg_sq)
                    # Use the max. for normalizing running avg. of gradient
                    denom = max_exp_avg_sq.sqrt().add_(group['eps'])
                else:
                    denom = exp_avg_sq.sqrt().add_(group['eps'])

                bias_correction1 = 1 - beta1 ** state['step']
                bias_correction = 1 - beta2 ** state['step']
                # step_size = group['lr'] * math.sqrt(bias_correction2) / bias_correction1
                bias_correction.sqrt().div_(bias_correction1)

                denom.div_(exp_avg)

                p.data.addcdiv_(bias_correction, denom, value=-group['lr'])
        return loss
    
    
class GnT(object):
    """
    Generate-and-Test algorithm for feed forward neural networks, based on maturity-threshold based replacement
    """
    def __init__(
            self,
            net,
            opt,
            net_plus,
            hidden_activation='relu',
            decay_rate=0.99,
            replacement_rate=1e-4,
            init='kaiming',
            device="cpu",
            maturity_threshold=20,
            util_type='contribution',
            accumulate=False,
    ):
        super(GnT, self).__init__()
        self.device = device
        self.net = net
        self.net_plus = net_plus
        self.num_hidden_layers = len(self.net)
        self.accumulate = accumulate
        self.opt = opt
        self.opt_type = 'adam'

        """
        Define the hyper-parameters of the algorithm
        """
        self.replacement_rate = replacement_rate
        self.decay_rate = decay_rate
        self.maturity_threshold = maturity_threshold
        self.util_type = util_type

        """
        Utility of all features/neurons
        """
        self.util = [torch.zeros(self.net[i].out_features).to(self.device) for i in range(self.num_hidden_layers)]
        self.bias_corrected_util = \
            [torch.zeros(self.net[i].out_features).to(self.device) for i in range(self.num_hidden_layers)]
        self.ages = [torch.zeros(self.net[i].out_features).to(self.device) for i in range(self.num_hidden_layers)]
        self.m = torch.nn.Softmax(dim=1)
        self.mean_feature_act = [torch.zeros(self.net[i].out_features).to(self.device) for i in range(self.num_hidden_layers)]
        self.accumulated_num_features_to_replace = [0 for i in range(self.num_hidden_layers)]

        """
        Calculate uniform distribution's bound for random feature initialization
        """
        self.bounds = self.compute_bounds(hidden_activation=hidden_activation, init=init)

    def compute_bounds(self, hidden_activation, init='kaiming'):
        if hidden_activation in ['swish', 'elu']: hidden_activation = 'relu'
        if init == 'default':
            bounds = [sqrt(1 / self.net[i].in_features) for i in range(self.num_hidden_layers)]
        elif init == 'xavier':
            bounds = [torch.nn.init.calculate_gain(nonlinearity=hidden_activation) *
                      sqrt(6 / (self.net[i].in_features + self.net[i].out_features)) for i in
                      range(self.num_hidden_layers)]
        else:
            bounds = [torch.nn.init.calculate_gain(nonlinearity=hidden_activation) *
                      sqrt(3 / self.net[i].in_features) for i in range(self.num_hidden_layers)]
        bounds.append(1 * sqrt(3 / self.net_plus[0].in_features))
        return bounds

    def update_utility(self, layer_idx=0, features=None, next_features=None):
        with torch.no_grad():
            self.util[layer_idx] *= self.decay_rate
            """
            Adam-style bias correction
            """
            bias_correction = 1 - self.decay_rate ** self.ages[layer_idx]

            self.mean_feature_act[layer_idx] *= self.decay_rate
            self.mean_feature_act[layer_idx] -= - (1 - self.decay_rate) * features.mean(dim=0)
            bias_corrected_act = self.mean_feature_act[layer_idx] / bias_correction

            current_layer = self.net[layer_idx]
            if layer_idx == self.num_hidden_layers - 1:
                next_layer = self.net_plus[0]
                output_wight_mag = next_layer.weight.data.abs().mean(dim=0)
                input_wight_mag = current_layer.weight.data.abs().mean(dim=1)

                if self.util_type == 'weight':
                    new_util = output_wight_mag
                elif self.util_type == 'contribution':
                    new_util = output_wight_mag * features.abs().mean(dim=0)
                elif self.util_type == 'adaptation':
                    new_util = 1/input_wight_mag
                elif self.util_type == 'zero_contribution':
                    new_util = output_wight_mag * (features - bias_corrected_act).abs().mean(dim=0)
                elif self.util_type == 'adaptable_contribution':
                    new_util = output_wight_mag * (features - bias_corrected_act).abs().mean(dim=0) / input_wight_mag
                elif self.util_type == 'feature_by_input':
                    input_wight_mag = self.net[layer_idx].weight.data.abs().mean(dim=1)
                    new_util = (features - bias_corrected_act).abs().mean(dim=0) / input_wight_mag
                else:
                    new_util = 0
            else:
                next_layer = self.net[layer_idx+1]
                output_wight_mag = next_layer.weight.data.abs().mean(dim=0)
                input_wight_mag = current_layer.weight.data.abs().mean(dim=1)

                if self.util_type == 'weight':
                    new_util = output_wight_mag
                elif self.util_type == 'contribution':
                    new_util = output_wight_mag * features.abs().mean(dim=0)
                elif self.util_type == 'adaptation':
                    new_util = 1/input_wight_mag
                elif self.util_type == 'zero_contribution':
                    new_util = output_wight_mag * (features - bias_corrected_act).abs().mean(dim=0)
                elif self.util_type == 'adaptable_contribution':
                    new_util = output_wight_mag * (features - bias_corrected_act).abs().mean(dim=0) / input_wight_mag
                elif self.util_type == 'feature_by_input':
                    input_wight_mag = self.net[layer_idx].weight.data.abs().mean(dim=1)
                    new_util = (features - bias_corrected_act).abs().mean(dim=0) / input_wight_mag
                else:
                    new_util = 0

            self.util[layer_idx] += (1 - self.decay_rate) * new_util

            """
            Adam-style bias correction
            """
            self.bias_corrected_util[layer_idx] = self.util[layer_idx] / bias_correction

            if self.util_type == 'random':
                self.bias_corrected_util[layer_idx] = torch.rand(self.util[layer_idx].shape)

    def test_features(self, features):
        """
        Args:
            features: Activation values in the neural network
        Returns:
            Features to replace in each layer, Number of features to replace in each layer
        """
        features_to_replace = [torch.empty(0, dtype=torch.long).to(self.device) for _ in range(self.num_hidden_layers)]
        num_features_to_replace = [0 for _ in range(self.num_hidden_layers)]
    
        for i in range(self.num_hidden_layers):
            self.ages[i] += 1
            """
            Update feature utility
            """
            self.update_utility(layer_idx=i, features=features[i])
            """
            Find the no. of features to replace
            """
            eligible_feature_indices = torch.where(self.ages[i] > self.maturity_threshold)[0]
            if eligible_feature_indices.shape[0] == 0:
                continue
            num_new_features_to_replace = self.replacement_rate*eligible_feature_indices.shape[0]
            self.accumulated_num_features_to_replace[i] += num_new_features_to_replace

            """
            Case when the number of features to be replaced is between 0 and 1.
            """
            
            if num_new_features_to_replace < 1:
                if torch.rand(1) <= num_new_features_to_replace:
                    num_new_features_to_replace = 1
            num_new_features_to_replace = int(num_new_features_to_replace)
    
            if num_new_features_to_replace == 0:
                continue

            """
            Find features to replace in the current layer
            """
            new_features_to_replace = torch.topk(-self.bias_corrected_util[i][eligible_feature_indices],
                                                 num_new_features_to_replace)[1]
            new_features_to_replace = eligible_feature_indices[new_features_to_replace]

            """
            Initialize utility for new features
            """
            self.util[i][new_features_to_replace] = 0
            self.mean_feature_act[i][new_features_to_replace] = 0.

            features_to_replace[i] = new_features_to_replace
            num_features_to_replace[i] = num_new_features_to_replace

        return features_to_replace, num_features_to_replace

    def gen_new_features(self, features_to_replace, num_features_to_replace):
        """
        Generate new features: Reset input and output weights for low utility features
        """
        with torch.no_grad():
            for i in range(self.num_hidden_layers):
                if num_features_to_replace[i] == 0:
                    continue
                current_layer = self.net[i]
                if i == self.num_hidden_layers - 1:
                    
                    current_layer.weight.data[features_to_replace[i], :] *= 0.0
                    # noinspection PyArgumentList
                    current_layer.weight.data[features_to_replace[i], :] += \
                        torch.empty(num_features_to_replace[i], current_layer.in_features).uniform_(
                            -self.bounds[i], self.bounds[i]).to(self.device)
                    current_layer.bias.data[features_to_replace[i]] *= 0
                    """
                    # Update bias to correct for the removed features and set the outgoing weights and ages to zero
                    """
                    for next_layer in self.net_plus:
                        next_layer.bias.data += (next_layer.weight.data[:, features_to_replace[i]] * \
                                                        self.mean_feature_act[i][features_to_replace[i]] / \
                                                        (1 - self.decay_rate ** self.ages[i][features_to_replace[i]])).sum(dim=1)
                        next_layer.weight.data[:, features_to_replace[i]] = 0
                    self.ages[i][features_to_replace[i]] = 0
                    
                else:
                    next_layer = self.net[i + 1]
                    current_layer.weight.data[features_to_replace[i], :] *= 0.0
                    # noinspection PyArgumentList
                    current_layer.weight.data[features_to_replace[i], :] += \
                        torch.empty(num_features_to_replace[i], current_layer.in_features).uniform_(
                            -self.bounds[i], self.bounds[i]).to(self.device)
                    current_layer.bias.data[features_to_replace[i]] *= 0
                    """
                    # Update bias to correct for the removed features and set the outgoing weights and ages to zero
                    """
                    next_layer.bias.data += (next_layer.weight.data[:, features_to_replace[i]] * \
                                                    self.mean_feature_act[i][features_to_replace[i]] / \
                                                    (1 - self.decay_rate ** self.ages[i][features_to_replace[i]])).sum(dim=1)
                    next_layer.weight.data[:, features_to_replace[i]] = 0
                    self.ages[i][features_to_replace[i]] = 0


    def update_optim_params(self, features_to_replace, num_features_to_replace):
        """
        Update Optimizer's state
        """
        if self.opt_type == 'adam':
            for i in range(self.num_hidden_layers):
                # input weights
                if num_features_to_replace[i] == 0:
                    continue
                self.opt.state[self.net[i].weight]['exp_avg'][features_to_replace[i], :] = 0.0
                self.opt.state[self.net[i].bias]['exp_avg'][features_to_replace[i]] = 0.0
                self.opt.state[self.net[i].weight]['exp_avg_sq'][features_to_replace[i], :] = 0.0
                self.opt.state[self.net[i].bias]['exp_avg_sq'][features_to_replace[i]] = 0.0
                self.opt.state[self.net[i].weight]['step'][features_to_replace[i], :] = 0
                self.opt.state[self.net[i].bias]['step'][features_to_replace[i]] = 0
                # output weights
                if i == self.num_hidden_layers - 1:
                    for next_layer in self.net_plus:
                        self.opt.state[next_layer.weight]['exp_avg'][:, features_to_replace[i]] = 0.0
                        self.opt.state[next_layer.weight]['exp_avg_sq'][:, features_to_replace[i]] = 0.0
                        self.opt.state[next_layer.weight]['step'][:, features_to_replace[i]] = 0
                else:
                    self.opt.state[self.net[i + 1].weight]['exp_avg'][:, features_to_replace[i]] = 0.0
                    self.opt.state[self.net[i + 1].weight]['exp_avg_sq'][:, features_to_replace[i]] = 0.0
                    self.opt.state[self.net[i + 1].weight]['step'][:, features_to_replace[i]] = 0

    def gen_and_test(self, features):
        """
        Perform generate-and-test
        :param features: activation of hidden units in the neural network
        """
        features_to_replace, num_features_to_replace = self.test_features(features=features)
        self.gen_new_features(features_to_replace, num_features_to_replace)
        self.update_optim_params(features_to_replace, num_features_to_replace)

def train_env(envs, device, args, env_name, task_index):
    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -torch.prod(torch.Tensor(envs.single_action_space.shape).to(device)).item()
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr)
    else:
        alpha = args.alpha
    envs.single_observation_space.dtype = np.float32
    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the game
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    end = False
    for global_step in range(args.total_timesteps):
        if end == True:
            break
        # ALGO LOGIC: put action logic here
        if global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    if rate > args.threshold:
                        end = True
                    episode_total = success_total = 0
                print(
                    f"{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"{env_name}/{task_index}/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/{task_index}/episodic_length", info["episode"]["l"], global_step)
                writer.add_scalar(f"{env_name}/{task_index}/success_rate", rate, global_step)

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            data = rb.sample(args.batch_size)
            with torch.no_grad():
                next_state_actions, next_state_log_pi, _ = actor.get_action(data.next_observations.float())
                qf1_next_target = qf1_target(data.next_observations.float(), next_state_actions)
                qf2_next_target = qf2_target(data.next_observations.float(), next_state_actions)
                min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                    min_qf_next_target).view(-1)

            qf1_a_values, qf1_features = qf1.forward_with_feature(data.observations.to(torch.float32),
                                                                  data.actions.to(torch.float32))
            qf1_a_values = qf1_a_values.view(-1)
            qf2_a_values, qf2_features = qf2.forward_with_feature(data.observations.to(torch.float32),
                                                                  data.actions.to(torch.float32))
            qf2_a_values = qf2_a_values.view(-1)
            qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
            qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
            qf_loss = qf1_loss + qf2_loss

            # optimize the model
            qf1_optimizer.zero_grad()
            qf2_optimizer.zero_grad()
            qf_loss.backward()
            qf1_optimizer.step()
            qf2_optimizer.step()

            qf1_gnt.gen_and_test(features=qf1_features)
            qf2_gnt.gen_and_test(features=qf2_features)

            if global_step % args.policy_frequency == 0:  # TD 3 Delayed update support
                for _ in range(
                        args.policy_frequency
                ):  # compensate for the delay by doing 'actor_update_interval' instead of 1
                    pi, log_pi, _, actor_features = actor.get_action_with_feature(data.observations.float())
                    qf1_pi = qf1(data.observations.float(), pi)
                    qf2_pi = qf2(data.observations.float(), pi)
                    min_qf_pi = torch.min(qf1_pi, qf2_pi)
                    actor_loss = ((alpha * log_pi) - min_qf_pi).mean()

                    actor_optimizer.zero_grad()
                    actor_loss.backward()
                    actor_optimizer.step()

                    actor_gnt.gen_and_test(features=actor_features)

                    if args.autotune:
                        with torch.no_grad():
                            _, log_pi, _ = actor.get_action(data.observations.float())
                        alpha_loss = (-log_alpha.exp() * (log_pi + target_entropy)).mean()

                        a_optimizer.zero_grad()
                        alpha_loss.backward()
                        a_optimizer.step()
                        alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf1_values", qf1_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf2_values", qf2_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/{type}/{task_index}/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/{type}/{task_index}/SPS", int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/{type}/losses/alpha_loss", alpha_loss.item(), global_step)
    envs.close()

def test_env(envs, device, args, env_name, task_index):
        obs, _ = envs.reset(seed=args.seed)
        success_total = 0
        episode_total = 0
        for global_step in range(args.test_timesteps):

            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

            # TRY NOT TO MODIFY: execute the game and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # TRY NOT TO MODIFY: record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    if info['success'] == 1:
                        success_total += 1
                    episode_total += 1
                    rate = success_total / episode_total
                    if episode_total % args.average_num == 0:
                        episode_total = success_total = 0
                    print(
                        f"test_{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                    writer.add_scalar(f"test_{env_name}/{task_index}/episodic_return", info["episode"]["r"],
                                      global_step)
                    writer.add_scalar(f"test_{env_name}/{task_index}/episodic_length", info["episode"]["l"],
                                      global_step)
                    writer.add_scalar(f"test_{env_name}/{task_index}/success_rate", rate, global_step)
                    # break
            # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
            obs = next_obs
        envs.close()

if __name__ == "__main__":
    import stable_baselines3 as sb3

    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.gamma)])
            for env_name in args.env_names]

    max_action = float(envs[0].single_action_space.high[0])

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    qf1_optimizer = AdamGnT(list(qf1.parameters()), lr=args.q_lr)
    qf2_optimizer = AdamGnT(list(qf2.parameters()), lr=args.q_lr)
    actor_optimizer = AdamGnT(list(actor.parameters()), lr=args.policy_lr)
    
    qf1_gnt = GnT(net=qf1.layers, net_plus=qf1.layers_plus, opt=qf1_optimizer, device=device)
    qf2_gnt = GnT(net=qf2.layers, net_plus=qf2.layers_plus, opt=qf2_optimizer, device=device)
    actor_gnt = GnT(net=actor.layers, net_plus=actor.layers_plus, opt=actor_optimizer, device=device)

    env_num = len(envs)
    for i in range(args.cycle_num):
        for j in range(len(envs)):
            train_env(envs[j], device, args, args.env_names[j], i*env_num+j)
            for k in range(env_num):
                test_env(envs[k], device, args, args.env_names[k], i*env_num+j)
            
    writer.close()
