import argparse

from src.eval.toolfuzz.utils.setup import setup_env_vars
from src.eval.toolfuzz.envs.reset_context import DelayResetContext
from src.eval.toolfuzz.utils.tools import get_langchain_tools, get_composio_tools
from src.toolfuzz.agent_executors.langchain.openai_function import OpenAIFuncFactory, OpenAIFunction, ToolCallingAgent
from src.toolfuzz.result_classes import Budget
from src.toolfuzz.agent_executors.langchain.react_new import ReactAgentNew
from src.toolfuzz.runtime.runtime_fuzzer import RuntimeErrorTester
from src.toolfuzz.tools.info_extractors.tool_wrapper_factory import ToolWrapperFactory


def args():
    parser = argparse.ArgumentParser(description="Test agent tools.")
    parser.add_argument('-am', dest='agent_model')
    parser.add_argument('-pm', dest='prompt_model')
    parser.add_argument('-t', dest='tool')
    parser.add_argument('-l', dest='langchain', action='store_true')
    parser.add_argument('-agent', dest='agent')
    return parser.parse_args()


cl_args = args()


def test_tools(tools):
    tool_to_test = None
    for tool in tools:
        tool_wrapper = ToolWrapperFactory.create_extractor(tool)
        if tool_wrapper.get_tool_name() == cl_args.tool:
            tool_to_test = tool
            break
    assert tool_to_test is not None, f"Tool {cl_args.tool} not found in langchain tools"
    budget = Budget(time_limit=300, agent_token_limit=25_000, prompt_token_limit=25_000, agent_cost_limit=0.1, prompt_cost_limit=0.1)

    try:
        agent = ReactAgentNew(tool_to_test, cl_args.agent_model)
        tester = RuntimeErrorTester(agent_llm=cl_args.agent_model,prompt_llm=cl_args.prompt_model, tool=tool_to_test, agent=agent, budget=budget, fuzzer_iters=200, context_resetter=DelayResetContext(tool_wrapper))
        tester.test()
    except IndexError as e:
        print(f"Global level error for {tool}: {e}")


def test_tools_openai(tools):
    print("Testing OpenAI function calling")
    tool_to_test = None
    for tool in tools:
        tool_wrapper = ToolWrapperFactory.create_extractor(tool)
        if tool_wrapper.get_tool_name() == cl_args.tool:
            tool_to_test = tool
            break
    assert tool_to_test is not None, f"Tool {cl_args.tool} not found in langchain tools"
    budget = Budget(time_limit=300, agent_token_limit=25_000, prompt_token_limit=25_000, agent_cost_limit=0.1, prompt_cost_limit=0.1)

    try:
        agent = OpenAIFunction(tool_to_test, cl_args.agent_model)
        tester = RuntimeErrorTester(agent_llm=cl_args.agent_model,prompt_llm=cl_args.prompt_model, tool=tool_to_test, agent=agent, budget=budget, fuzzer_iters=200, context_resetter=DelayResetContext(tool_wrapper))
        tester.test()
    except IndexError as e:
        print(f"Global level error for {tool}: {e}")


def test_tools_tool_calling(tools):
    print("Testing Tool Callling")
    tool_to_test = None
    for tool in tools:
        tool_wrapper = ToolWrapperFactory.create_extractor(tool)
        if tool_wrapper.get_tool_name() == cl_args.tool:
            tool_to_test = tool
            break
    assert tool_to_test is not None, f"Tool {cl_args.tool} not found in langchain tools"
    budget = Budget(time_limit=300, agent_token_limit=25_000, prompt_token_limit=25_000, agent_cost_limit=0.1, prompt_cost_limit=0.1)

    try:
        agent = ToolCallingAgent(tool_to_test, cl_args.agent_model)
        tester = RuntimeErrorTester(agent_llm=cl_args.agent_model,prompt_llm=cl_args.prompt_model, tool=tool_to_test, agent=agent, budget=budget, fuzzer_iters=200, context_resetter=DelayResetContext(tool_wrapper))
        tester.test()
    except IndexError as e:
        print(f"Global level error for {tool}: {e}")


def main():
    setup_env_vars()
    if cl_args.langchain:
        if cl_args.agent == 'tool_calling':
            test_tools_tool_calling(get_langchain_tools())
        elif cl_args.agent == 'openai_function':
            test_tools_openai(get_langchain_tools())
        else:
            test_tools(get_langchain_tools())
    else:
        if cl_args.agent == 'tool_calling':
            test_tools_tool_calling(get_composio_tools())
        elif cl_args.agent == 'openai_function':
            test_tools_openai(get_composio_tools())
        else:
            test_tools(get_composio_tools())


if __name__ == '__main__':
    main()
