import argparse
import json

from src.eval.toolfuzz.utils.setup import setup_env_vars
from src.eval.toolfuzz.envs.reset_context import DelayResetContext
from src.eval.toolfuzz.utils.tools import get_langchain_tools, get_composio_tools
from src.toolfuzz.agent_executors.langchain.openai_function import OpenAIFuncFactory, OpenAIFunction, ToolCallingAgent
from src.toolfuzz.result_classes import Budget
from src.toolfuzz.agent_executors.langchain.react_new import ReactAgentNew
from src.toolfuzz.runtime.runtime_fuzzer import RuntimeErrorTester
from src.toolfuzz.tools.info_extractors.tool_wrapper_factory import ToolWrapperFactory


def args():
    parser = argparse.ArgumentParser(description="Test agent tools.")
    parser.add_argument('-am', dest='agent_model')
    parser.add_argument('-pm', dest='prompt_model')
    parser.add_argument('-t', dest='tool')
    parser.add_argument('-l', dest='langchain', action='store_true')
    parser.add_argument('-draft_file', dest='draft_file')
    return parser.parse_args()

cl_args = args()

def test_tools(tools):
    tool_to_test = None
    for tool in tools:
        tool_wrapper = ToolWrapperFactory.create_extractor(tool)
        if tool_wrapper.get_tool_name() == cl_args.tool:
            tool_to_test = tool
            break
    if tool_to_test is None:
        print(f"Tool not found in langchain tools: {cl_args.tool}")
        return
    assert tool_to_test is not None, f"Tool {cl_args.tool} not found in langchain tools"
    budget = Budget(time_limit=300, agent_token_limit=25_000, prompt_token_limit=25_000, agent_cost_limit=0.1, prompt_cost_limit=0.1)

    with open(cl_args.draft_file, "r") as draft_file:
        draft_descriptions = json.load(draft_file)
    # Now find the tool in the draft descriptions
    for draft_tool in draft_descriptions:
        tool_found = False
        if draft_tool['tool_name'].lower() == tool_wrapper.get_tool_name().lower():
            for _, value in draft_tool['tool_guidelines'].items():
                if value['name'].lower() == tool_wrapper.get_tool_name().lower():
                    tool = value['description']
                    tool_found = True
                    break
            break
        for _, values in draft_tool['tool_guidelines'].items():
            if values['name'].lower() == tool_wrapper.get_tool_name().lower():
                tool.description = values['description']
                tool_found = True
                break
        if tool_found:
            break
    else:
        raise ValueError(f"Tool {tool_wrapper.get_tool_name()} not found in draft descriptions")

    try:
        agent = ReactAgentNew(tool_to_test, cl_args.agent_model)
        agent.agent_name = "DRAFTAgent"
        tester = RuntimeErrorTester(agent_llm=cl_args.agent_model,prompt_llm=cl_args.prompt_model, tool=tool_to_test, agent=agent, budget=budget, fuzzer_iters=200, context_resetter=DelayResetContext(tool_wrapper))
        tester.test()
    except IndexError as e:
        print(f"Global level error for {tool}: {e}")


def main():
    setup_env_vars()
    if cl_args.langchain:
        test_tools(get_langchain_tools())
    else:
        test_tools(get_composio_tools())


if __name__ == '__main__':
    main()
