import abc
from langchain_community.utilities import WikipediaAPIWrapper

from src.eval.toolfuzz.loaders.composio_tool_loader import load_tools
from src.eval.buggy_tools.fixed_tools.pubmed import PubmedQueryRunFixed
from src.eval.buggy_tools.fixed_tools.wikipedia import WikipediaQueryRunFixed
from src.eval.toolfuzz.loaders.langchain_loader import LangChainToolLoader, LangChainToolkitLoader

from src.eval.toolfuzz.loaders.module_tool_loader import ToolLoader


def get_subclasses(cls):
    subclasses = set()
    for subclass in cls.__subclasses__():
        subclasses.add(subclass)
        subclasses.update(get_subclasses(subclass))
    return subclasses


def get_langchain_loaders():
    # Get all loaders which inherit from LangchainToolLoader
    import src.eval.toolfuzz.loaders.langchain_tool_loaders

    loaders = get_subclasses(LangChainToolLoader)
    relevant_loaders = []
    for loader in loaders:
        is_abc = False
        for superclass in loader.__bases__:
            if superclass == abc.ABC:
                is_abc = True
                break
        if not is_abc:
            relevant_loaders.append(loader)

    import src.eval.toolfuzz.loaders.langchain_toolkit_loaders
    toolkit_loaders = LangChainToolkitLoader.__subclasses__()

    return relevant_loaders, toolkit_loaders


def get_langchain_tools():
    loaders, toolkit_loaders = get_langchain_loaders()
    tools = []
    for loader in loaders:
        tools.append(loader().get_tool())

    for loader in toolkit_loaders:
        tools.extend(loader().get_tools())
    
    return tools


def get_composio_tools():
    return load_tools()


def get_fixed_tools():
    tools = [
        WikipediaQueryRunFixed(api_wrapper=WikipediaAPIWrapper()), PubmedQueryRunFixed()
    ]
    return tools


if __name__ == '__main__':
    from src.toolfuzz.tools.info_extractors.tool_wrapper_factory import ToolWrapperFactory
    for tool in get_composio_tools():
        tool_wrapper = ToolWrapperFactory.create_extractor(tool)
        print(tool_wrapper.get_tool_name())
