import math
import random
from typing import List, Optional


class LargeIndexRandomizer:
    def __init__(self, length: int, seed: int, levels: int = 4,
                 window_ratios: Optional[List[float]] = None):
        self._validate_inputs(length, seed, levels, window_ratios)

        self.length = length
        self.seed = seed
        self.levels = levels

        self._rng = random.Random(seed)

        self._multipliers = self._generate_multipliers()
        self._addends = self._generate_addends()

    def _validate_inputs(self, length: int, seed: int, levels: int,
                         window_ratios: Optional[List[float]]) -> None:
        if not isinstance(length, (int, float)) or length <= 0:
            raise ValueError("Length must be positive")
        if not isinstance(seed, int):
            raise ValueError("Seed must be integer")
        if not isinstance(levels, int) or levels < 2:
            raise ValueError("Levels must be at least 2")

    def _generate_multipliers(self) -> List[int]:
        multipliers = []
        for _ in range(self.levels):
            while True:
                mult = self._rng.randint(1, self.length - 1)
                if math.gcd(mult, self.length) == 1:
                    multipliers.append(mult)
                    break
        return multipliers

    def _generate_addends(self) -> List[int]:
        return [self._rng.randint(0, self.length - 1)
                for _ in range(self.levels)]

    def __call__(self, index: int) -> int:
        if not 0 <= index < self.length:
            raise ValueError(f"Index {index} out of bounds [0, {self.length})")

        current = index
        for level in range(self.levels):
            current = (current * self._multipliers[level]) % self.length
            current = (current + self._addends[level]) % self.length

        return current

    def reverse(self, output: int) -> int:
        if not 0 <= output < self.length:
            raise ValueError(f"Output {output} out of bounds [0, {self.length})")

        current = output
        for level in range(self.levels - 1, -1, -1):
            current = (current - self._addends[level]) % self.length
            mult_inverse = pow(self._multipliers[level], -1, self.length)
            current = (current * mult_inverse) % self.length

        return current
