from data_provider.data_factory import data_provider
from exp.exp_basic import Exp_Basic
from utils.tools import EarlyStopping, adjust_learning_rate, visual
from utils.metrics import metric
import torch
import torch.nn as nn
from torch import optim
import os
import time
import warnings
import numpy as np

warnings.filterwarnings('ignore')


class Exp_MeKong(Exp_Basic):
    def __init__(self, args, phase, verbose=True, device='cpu'):
        super(Exp_MeKong, self).__init__(args, phase, verbose=verbose, device=device)
        self.model = self._build_model()
        self.phase = phase
        num_param = self.numel(self.model, True)
        if self.verbose:
            print('model number of parameters:', num_param)

    def numel(self, m: torch.nn.Module, only_trainable: bool = False):
        """
        Returns the total number of parameters used by `m` (only counting
        shared parameters once); if `only_trainable` is True, then only
        includes parameters with `requires_grad = True`
        """
        parameters = list(m.parameters())
        if only_trainable:
            parameters = [p for p in parameters if p.requires_grad]
        unique = {p.data_ptr(): p for p in parameters}.values()
        return sum(p.numel() for p in unique)

    def _build_model(self):
        if self.args.features == 'S':
            num_channels = 1
        else:
            num_channels = len(self.station_channels_dict[self.target_station])
        self.args.enc_in = num_channels
        self.args.dec_in = num_channels
        self.args.c_out = num_channels
        model = self.model_dict[self.args.model].Model(self.args).float().to(self.device)
        return model

    def _get_data(self, flag):
        data_set, data_loader = data_provider(self.args, flag, self.verbose, 'diff_range_for_target')
        return data_set, data_loader

    def _select_optimizer(self):
        model_optim = optim.Adam(self.model.parameters(), lr=self.args.learning_rate)
        return model_optim

    def _select_criterion(self):
        criterion = nn.MSELoss()
        return criterion

    def vali(self, criterion, data_loader):
        total_loss = []
        self.model.eval()
        with torch.no_grad():
            for i, (batch_x, batch_y, batch_x_mark, batch_y_mark, cycle_index) in enumerate(data_loader):
                batch_x = batch_x.float().to(self.device)
                batch_x_mark = batch_x_mark.float().to(self.device)
                batch_y_mark = batch_y_mark.float().to(self.device)
                batch_y = batch_y.float()
                # decoder input
                dec_inp = torch.zeros_like(batch_y[:, -self.args.pred_len:, -1:]).float()
                dec_inp = torch.cat([batch_y[:, :self.args.label_len, -1:], dec_inp], dim=1).float().to(self.device)
                # encoder - decoder
                if self.args.model in self.cycle_model_list:
                    cycle_index = cycle_index.to(self.device)
                    outputs = self.model(batch_x, batch_x_mark, dec_inp, batch_y_mark, cycle_index=cycle_index)
                else:
                    outputs = self.model(batch_x, batch_x_mark, dec_inp, batch_y_mark)
                outputs = outputs[:, -self.args.pred_len:, -1:]
                batch_y = batch_y[:, -self.args.pred_len:, -1:]
                pred = outputs.detach().cpu()
                true = batch_y.detach().cpu()
                loss = criterion(pred, true)
                total_loss.append(loss)
        total_loss = np.average(total_loss)
        self.model.train()
        return total_loss

    def train(self, setting):
        self.train_data, self.train_loader = self._get_data(flag='train')
        self.vali_data, self.vali_loader = self._get_data(flag='val')
        self.test_data, self.test_loader = self._get_data(flag='test')
        path = os.path.join(self.args.checkpoints, setting)
        if not os.path.exists(path + f'/{self.phase}/'):
            os.makedirs(path + f'/{self.phase}/')

        time_now = time.time()

        train_steps = len(self.train_loader)
        early_stopping = EarlyStopping(patience=self.args.patience, verbose=self.verbose, delta=1e-6)

        model_optim = self._select_optimizer()
        criterion = self._select_criterion()

        for epoch in range(self.args.train_epochs):
            iter_count = 0
            train_loss = []

            self.model.train()
            epoch_time = time.time()
            for i, (batch_x, batch_y, batch_x_mark, batch_y_mark, cycle_index) in enumerate(self.train_loader):
                iter_count += 1
                model_optim.zero_grad()
                batch_x = batch_x.float().to(self.device)
                batch_x_mark = batch_x_mark.float().to(self.device)
                batch_y_mark = batch_y_mark.float().to(self.device)
                batch_y = batch_y.float().to(self.device)

                # decoder input
                dec_inp = torch.zeros_like(batch_y[:, -self.args.pred_len:, -1:]).float()
                dec_inp = torch.cat([batch_y[:, :self.args.label_len, -1:], dec_inp], dim=1).float().to(self.device)

                # encoder - decoder
                if self.args.model in self.cycle_model_list:
                    cycle_index = cycle_index.to(self.device)
                    outputs = self.model(batch_x, batch_x_mark, dec_inp, batch_y_mark, cycle_index=cycle_index)
                else:
                    outputs = self.model(batch_x, batch_x_mark, dec_inp, batch_y_mark)
                outputs = outputs[:, -self.args.pred_len:, -1:]
                batch_y = batch_y[:, -self.args.pred_len:, -1:]
                loss = criterion(outputs, batch_y)
                train_loss.append(loss.item())

                if self.verbose:
                    if (i + 1) % 100 == 0:
                        print("\titers: {0}, epoch: {1} | loss: {2:.7f}".format(i + 1, epoch + 1, loss.item()))
                        speed = (time.time() - time_now) / iter_count
                        left_time = speed * ((self.args.train_epochs - epoch) * train_steps - i)
                        print('\tspeed: {:.4f}s/iter; left time: {:.4f}s'.format(speed, left_time))
                        iter_count = 0
                        time_now = time.time()
                loss.backward()
                model_optim.step()
            if self.verbose:
                print("Epoch: {} cost time: {}".format(epoch + 1, time.time() - epoch_time))
            train_loss = np.average(train_loss)
            vali_loss = self.vali(criterion, self.vali_loader)
            test_loss = self.vali(criterion, self.test_loader)
            if self.verbose:
                print("Epoch: {0}, Steps: {1} | Train Loss: {2:.7f} Vali Loss: {3:.7f} Test Loss: {4:.7f}".format(
                    epoch + 1, train_steps, train_loss, vali_loss, test_loss))
            early_stopping(vali_loss, self.model, path + f'/{self.phase}/')
            if early_stopping.early_stop:
                if self.verbose:
                    print("Early stopping")
                break

            adjust_learning_rate(model_optim, epoch + 1, self.args, self.verbose)

        best_model_path = path + f'/{self.phase}/' + 'checkpoint.pth'
        self.model.load_state_dict(torch.load(best_model_path))

    def test(self, setting, test=0):
        self.train_data, self.train_loader = self._get_data(flag='train')
        self.vali_data, self.vali_loader = self._get_data(flag='val')
        self.test_data, self.test_loader = self._get_data(flag='test')

        if self.verbose:
            print('loading model')
        self.model = self._build_model().to(self.device)
        self.model.load_state_dict(torch.load(os.path.join('./checkpoints/' + setting, self.phase, 'checkpoint.pth')))

        preds = []
        trues = []
        inputs = []
        folder_path = './test_results/' + setting + f'/{self.args.data}/'
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)

        self.model.eval()
        with torch.no_grad():
            for i, (batch_x, batch_y, batch_x_mark, batch_y_mark, cycle_index) in enumerate(self.test_loader):
                batch_x = batch_x.float().to(self.device)
                batch_x_mark = batch_x_mark.float().to(self.device)
                batch_y_mark = batch_y_mark.float().to(self.device)
                batch_y = batch_y.float().to(self.device)
                # decoder input
                dec_inp = torch.zeros_like(batch_y[:, -self.args.pred_len:, -1:]).float()
                dec_inp = torch.cat([batch_y[:, :self.args.label_len, -1:], dec_inp], dim=1).float().to(self.device)
                # encoder - decoder
                if self.args.model in self.cycle_model_list:
                    cycle_index = cycle_index.to(self.device)
                    outputs = self.model(batch_x, batch_x_mark, dec_inp, batch_y_mark, cycle_index=cycle_index)
                else:
                    outputs = self.model(batch_x, batch_x_mark, dec_inp, batch_y_mark)
                outputs = outputs[:, -self.args.pred_len:, -1:]
                batch_y = batch_y[:, -self.args.pred_len:, -1:]
                outputs = outputs.detach().cpu().numpy()
                batch_y = batch_y.detach().cpu().numpy()
                if self.args.inverse:
                    shape = outputs.shape
                    outputs = self.test_data.inverse_transform(outputs.reshape(shape[0] * shape[1], -1)).reshape(shape)
                    batch_y = self.test_data.inverse_transform(batch_y.reshape(shape[0] * shape[1], -1)).reshape(shape)

                pred = outputs
                true = batch_y

                preds.append(pred)
                trues.append(true)
                # if i % 20 == 0:
                input = batch_x.detach().cpu().numpy()
                if self.args.inverse:
                    shape = input.shape
                    input = self.test_data.inverse_transform(input.reshape(shape[0] * shape[1], -1)).reshape(shape)
                inputs.append(input)
                gt = np.concatenate((input[0, :, -1], true[0, :, -1]), axis=0)
                pd = np.concatenate((input[0, :, -1], pred[0, :, -1]), axis=0)
                visual(gt, pd, os.path.join(folder_path, str(i) + '.pdf'))

        preds = np.concatenate(preds, axis=0)
        trues = np.concatenate(trues, axis=0)
        inputs = np.concatenate(inputs, axis=0)
        if self.verbose:
            print('test shape:', preds.shape, trues.shape)
        preds = preds.reshape(-1, preds.shape[-2], preds.shape[-1])
        trues = trues.reshape(-1, trues.shape[-2], trues.shape[-1])
        inputs = inputs.reshape(-1, inputs.shape[-2], inputs.shape[-1])
        if self.verbose:
            print('test shape:', preds.shape, trues.shape)

        # result save
        folder_path = self.args.results + setting + f'/{self.args.data}/'
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)

        mae, mse, rmse, mape, mspe, r2, nse, pbias, kge, flv, fhv = metric(preds[:, :, -1], trues[:, :, -1])
        f = open("result_initial.txt", 'a')
        f.write(setting + "  \n")
        f.write(f'nse:{nse}, pbias:{pbias}, kge:{kge}, flv:{flv}, '
                f'fhv:{fhv}, mae: {mae}, mse: {mse}, rmse: {rmse}, '
                f'mape: {mape}, mspe: {mspe}, r2: {r2}')
        f.write('\n')
        f.write('\n')
        f.close()

        np.save(os.path.join(str(folder_path), 'metrics.npy'),
                np.array([mae, mse, rmse, mape, mspe, r2, nse, pbias, kge, flv,
                          fhv]))
        np.save(os.path.join(str(folder_path), 'pred.npy'), preds)
        np.save(os.path.join(str(folder_path), 'true.npy'), trues)
        return mae, mse, rmse, mape, mspe, r2, nse, pbias, kge, flv, fhv, \
            mae, mse, rmse, mape, mspe, r2, nse, pbias, kge, flv, fhv
