import torch
import torch.nn as nn
from layers.StandardNorm import myNormalize
from layers.Autoformer_EncDec import series_decomp
from copy import deepcopy


class SegRNN(nn.Module):
    """
    Paper link: https://arxiv.org/abs/2308.11200.pdf
    """
    def __init__(self, configs):
        super(SegRNN, self).__init__()

        # get parameters
        self.seq_len = configs.seq_len
        self.enc_in = configs.enc_in
        self.d_model = configs.d_model
        self.dropout = configs.dropout

        self.pred_len = configs.pred_len

        self.seg_len = 1
        self.seg_num_x = self.seq_len // self.seg_len
        self.seg_num_y = self.pred_len // self.seg_len

        # building model
        self.valueEmbedding = nn.Sequential(
            nn.Linear(self.seg_len, self.d_model),
            nn.ReLU()
        )
        self.rnn = nn.GRU(input_size=self.d_model, hidden_size=self.d_model, num_layers=1, bias=True,
                          batch_first=True, bidirectional=False)
        self.pos_emb = nn.Parameter(torch.randn(self.seg_num_y, self.d_model // 2))
        self.channel_emb = nn.Parameter(torch.randn(self.enc_in, self.d_model // 2))
        self.predict = nn.Sequential(
            nn.Dropout(self.dropout),
            nn.Linear(self.d_model, self.seg_len)
        )

    def encoder(self, x):
        # b:batch_size c:channel_size s:seq_len s:seq_len
        # d:d_model w:seg_len n:seg_num_x m:seg_num_y
        batch_size = x.size(0)
        # segment and embedding    b,c,s -> bc,n,w -> bc,n,d
        x = self.valueEmbedding(x.reshape(-1, self.seg_num_x, self.seg_len))
        # encoding
        _, hn = self.rnn(x)  # bc,n,d  1,bc,d
        pos_emb = torch.cat([
            self.pos_emb.unsqueeze(0).repeat(self.enc_in, 1, 1),
            self.channel_emb.unsqueeze(1).repeat(1, self.seg_num_y, 1)
        ], dim=-1).view(-1, 1, self.d_model).repeat(batch_size,1,1)
        _, hy = self.rnn(pos_emb, hn.repeat(1, 1, self.seg_num_y).view(1, -1, self.d_model))  # bcm,1,d  1,bcm,d
        y = self.predict(hy).view(-1, self.enc_in, self.pred_len)
        return y

    def forward(self, x_enc):
        return self.encoder(x_enc)


class Model(nn.Module):
    def __init__(self, configs):
        super(Model, self).__init__()
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        self.channels = configs.enc_in
        self.e_layers = configs.e_layers
        self.d_model = configs.d_model
        self.dropout = configs.dropout

        self.decomposition = series_decomp(configs.moving_avg)
        self.seasonal_layers = SegRNN(configs)
        self.trend_layers = SegRNN(configs)
        self.instance_norm_seasonal = myNormalize(self.channels, affine=True, subtract_last=True)
        self.instance_norm_trend = myNormalize(self.channels, affine=True, subtract_last=True)

    def forward(self, x_enc, x_enc_mark=None, x_dec=None, x_dec_mark=None, target_x=None):
        if target_x is None:
            seasonal_target, trend_target = None, None
        else:
            seasonal_target, trend_target = self.decomposition(target_x)
        seasonal_init, trend_init = self.decomposition(x_enc)
        seasonal_init = self.instance_norm_seasonal(seasonal_init, seasonal_target, 'norm')
        trend_init = self.instance_norm_trend(trend_init, trend_target, 'norm')
        seasonal_init, trend_init = seasonal_init.permute(0, 2, 1), trend_init.permute(0, 2, 1)
        seasonal_output = self.seasonal_layers(seasonal_init)
        trend_output = self.trend_layers(trend_init)
        seasonal_output = self.instance_norm_seasonal(seasonal_output.permute(0, 2, 1), mode='denorm')
        trend_output = self.instance_norm_trend(trend_output.permute(0, 2, 1), mode='denorm')
        dec_out = seasonal_output + trend_output
        return dec_out[:, -self.pred_len:, :]  # [B, L, D]
