import torch
import torch.nn as nn


class LSTM(nn.Module):
    def __init__(self, configs, feature_in, feature_out):
        super(LSTM, self).__init__()
        self.feature_in = feature_in
        self.feature_out = feature_out
        self.lstm = nn.LSTM(input_size=self.feature_in, hidden_size=configs.d_model, batch_first=True,
                            num_layers=configs.e_layers, dropout=configs.dropout, device=configs.device)
        self.feature_project = nn.Linear(configs.d_model, self.feature_out, device=configs.device)
        self.time_project = nn.Linear(configs.seq_len, configs.pred_len, device=configs.device)

    def forward(self, x):
        # x [B, L, C]
        x, *_ = self.lstm(x)  # [B, L, D]
        x = self.feature_project(x)  # [B, L, C]
        x = self.time_project(x.permute(0, 2, 1)).permute(0, 2, 1)  # [B, H, C]
        return x


class Model(nn.Module):
    def __init__(self, configs):
        super(Model, self).__init__()
        self.name = 'LSTM'
        self.configs = configs
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        feature_in = configs.enc_in
        feature_out = configs.c_out
        self.base_model = LSTM(configs, feature_in, feature_out)

    def forecast(self, x_enc):
        dec_out = self.base_model(x_enc)
        return dec_out  # [B, H, C]

    def forward(self, x_enc, x_mark_enc=None, x_dec=None, x_mark_dec=None, target_x=None):
        # Normalization from Non-stationary Transformer
        means = target_x[:, :, -1:].mean(1, keepdim=True).detach() \
            if target_x is not None \
            else x_enc.mean(1, keepdim=True).detach()
        x_enc = x_enc - means
        stdev = torch.sqrt(torch.var(target_x[:, :, -1:], dim=1, keepdim=True, unbiased=False) + 1e-5) \
            if target_x is not None \
            else torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
        x_enc /= stdev
        _, _, N = x_enc.shape

        dec_out = self.forecast(x_enc)

        # De-Normalization from Non-stationary Transformer
        dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
        dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
        return dec_out
