import argparse
import torch
import numpy as np
from numpy import *

# compute rollout between attention layers
def compute_rollout_attention(all_layer_matrices, start_layer=0):
    # adding residual consideration- code adapted from https://github.com/samiraabnar/attention_flow
    num_tokens = all_layer_matrices[0].shape[1]
    batch_size = all_layer_matrices[0].shape[0]
    eye = torch.eye(num_tokens).expand(batch_size, num_tokens, num_tokens).to(all_layer_matrices[0].device)
    all_layer_matrices = [all_layer_matrices[i] + eye for i in range(len(all_layer_matrices))]
    matrices_aug = [all_layer_matrices[i] / all_layer_matrices[i].sum(dim=-1, keepdim=True)
                          for i in range(len(all_layer_matrices))]
    joint_attention = matrices_aug[start_layer]
    for i in range(start_layer+1, len(matrices_aug)):
        joint_attention = matrices_aug[i].bmm(joint_attention)
    return joint_attention

class LRP:
    def __init__(self, model):
        self.model = model
        self.model.eval()

    def generate_LRP(self, input, index=None, method="transformer_attribution", is_ablation=False, start_layer=0, **kwargs):
        if method.startswith("predmap"):
            if type(index) == torch.Tensor and index.numel() > 1:
                index = index
            else:
                index = index.item() if type(index) == torch.Tensor else index
            # invoke "method" of self.model
            method_invoke = getattr(self.model, method)
            return method_invoke(input, index, **kwargs)
        output = self.model(input)
        kwargs = {"alpha": 1}
        if index == None:
            index = np.argmax(output.cpu().data.numpy(), axis=-1)

        one_hot = np.zeros((1, output.size()[-1]), dtype=np.float32)
        one_hot[0, index] = 1
        one_hot_vector = one_hot
        one_hot = torch.from_numpy(one_hot).requires_grad_(True)
        one_hot = torch.sum(one_hot.cuda() * output)

        self.model.zero_grad()
        one_hot.backward(retain_graph=True)

        return self.model.relprop(torch.tensor(one_hot_vector).to(input.device), method=method, is_ablation=is_ablation,
                                  start_layer=start_layer, **kwargs)



class Baselines:
    def __init__(self, model):
        self.model = model
        self.model.eval()

    def generate_cam_attn(self, input, index=None):
        output = self.model(input.cuda(), register_hook=True) # (batch, classes)
        if index == None:
            index = np.argmax(output.cpu().data.numpy())

        one_hot = np.zeros((1, output.size()[-1]), dtype=np.float32) # (1, classes)
        one_hot[0][index] = 1
        one_hot = torch.from_numpy(one_hot).requires_grad_(True)
        one_hot = torch.sum(one_hot.cuda() * output) # (1, )

        self.model.zero_grad()
        one_hot.backward(retain_graph=True)
        #################### attn
        grad = self.model.blocks[-1].attn.get_attn_gradients() # (batch, heads, tokens, tokens)
        cam = self.model.blocks[-1].attn.get_attention_map() # (batch, heads, tokens, tokens)
        cam = cam[0, :, 0, 1:].reshape(-1, 14, 14) # (heads, 14, 14)
        grad = grad[0, :, 0, 1:].reshape(-1, 14, 14) # (heads, 14, 14)
        grad = grad.mean(dim=[1, 2], keepdim=True) # (heads, 1, 1)
        cam = (cam * grad).mean(0).clamp(min=0) # (14, 14)
        cam = (cam - cam.min()) / (cam.max() - cam.min())

        return cam
        #################### attn

    def generate_rollout(self, input, start_layer=0):
        with torch.no_grad():
            self.model(input)
        # self.model(input)
        blocks = self.model.blocks
        all_layer_attentions = []
        for blk in blocks:
            attn_heads = blk.attn.get_attention_map() # (B, H, N, N)
            avg_heads = (attn_heads.sum(dim=1) / attn_heads.shape[1]).detach() # (B, N, N)
            all_layer_attentions.append(avg_heads)
        rollout = compute_rollout_attention(all_layer_attentions, start_layer=start_layer)
        return rollout[:,0, 1:]
    
    def generate_raw_attn(self, input):
        with torch.no_grad():
            temp = self.model.blocks_to_save_attnmaps
            self.model.blocks_to_save_attnmaps = [len(self.model.blocks)-1]
            self.model(input)
            self.model.blocks_to_save_attnmaps = temp
        # self.model(input)
        cam = self.model.blocks[-1].attn.get_attention_map() # (batch, heads, tokens, tokens)
        cam = cam[0].reshape(-1, cam.shape[-1], cam.shape[-1]) # (heads, tokens, tokens)
        cam = cam.clamp(min=0).mean(dim=0) # (tokens, tokens)
        cam = cam[0, 1:] # (tokens-1, )
        return cam
    
