# Copyright 2024-2025
# [ANONYMIZED_INSTITUTION],
# [ANONYMIZED_FACULTY],
# [ANONYMIZED_DEPARTMENT]
#
# Authors:
# AUTHOR_1 (author1@example.com)
# AUTHOR_2 (author2@example.com)
#
# Code generation tools and workflows:
# First versions of this code were potentially generated
# with the help of AI writing assistants including
# GitHub Copilot, ChatGPT, Microsoft Copilot, Google Gemini.
# Afterwards, the generated segments were manually reviewed and edited.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Basic path manager for the PEFT finetuning mode."""

import logging
import pathlib

from topollm.config_classes.constants import ITEM_SEP, KV_SEP, NAME_PREFIXES
from topollm.config_classes.finetuning.peft.peft_config import PEFTConfig
from topollm.path_management.convert_object_to_valid_path_part import convert_list_to_path_part
from topollm.typing.enums import DescriptionType, FinetuningMode, Verbosity

default_logger: logging.Logger = logging.getLogger(
    name=__name__,
)


class PEFTPathManagerBasic:
    """Path manager for the PEFT finetuning mode."""

    def __init__(
        self,
        peft_config: PEFTConfig,
        verbosity: Verbosity = Verbosity.NORMAL,
        logger: logging.Logger = default_logger,
    ) -> None:
        """Initialize the PEFTPathManagerBasic."""
        self.peft_config: PEFTConfig = peft_config

        self.verbosity: Verbosity = verbosity
        self.logger: logging.Logger = logger

    @property
    def peft_description_subdir(
        self,
    ) -> pathlib.Path:
        path = pathlib.Path(
            self.get_finetuning_mode_description(
                description_type=DescriptionType.LONG,
            ),
            self.get_lora_description(
                description_type=DescriptionType.LONG,
            ),
        )

        return path

    def get_config_description(
        self,
        description_type: DescriptionType = DescriptionType.LONG,
        short_description_separator: str = "-",
    ) -> str:
        match description_type:
            case DescriptionType.LONG:
                description: str = (
                    self.get_finetuning_mode_description(
                        description_type=DescriptionType.LONG,
                    )
                    + ITEM_SEP
                    + self.get_lora_description(
                        description_type=DescriptionType.LONG,
                    )
                )
            case DescriptionType.SHORT:
                description: str = (
                    self.get_finetuning_mode_description(
                        description_type=DescriptionType.SHORT,
                        short_description_separator=short_description_separator,
                    )
                    + short_description_separator
                    + self.get_lora_description(
                        description_type=DescriptionType.SHORT,
                        short_description_separator=short_description_separator,
                    )
                )
            case _:
                msg: str = f"Unknown {description_type = }"
                raise ValueError(
                    msg,
                )

        return description

    def get_finetuning_mode_description(
        self,
        description_type: DescriptionType = DescriptionType.LONG,
        short_description_separator: str = "-",  # noqa: ARG002 - not used but kept for consistent interface
    ) -> str:
        match description_type:
            case DescriptionType.LONG:
                match self.peft_config.finetuning_mode:
                    case FinetuningMode.STANDARD:
                        description: str = f"{NAME_PREFIXES['FinetuningMode']}{KV_SEP}standard"
                    case FinetuningMode.LORA:
                        description = f"{NAME_PREFIXES['FinetuningMode']}{KV_SEP}lora"
                    case _:
                        msg: str = f"Unknown {self.peft_config.finetuning_mode = }"
                        raise ValueError(
                            msg,
                        )
            case DescriptionType.SHORT:
                match self.peft_config.finetuning_mode:
                    case FinetuningMode.STANDARD:
                        description = "standard"
                    case FinetuningMode.LORA:
                        description = "lora"
                    case _:
                        msg: str = f"Unknown {self.peft_config.finetuning_mode = }"
                        raise ValueError(
                            msg,
                        )
            case _:
                msg: str = f"Unknown {description_type = }"
                raise ValueError(
                    msg,
                )

        return description

    def get_lora_description(
        self,
        description_type: DescriptionType = DescriptionType.LONG,
        short_description_separator: str = "-",
    ) -> str:
        match description_type:
            case DescriptionType.LONG:
                match self.peft_config.finetuning_mode:
                    case FinetuningMode.STANDARD:
                        description: str = "lora-None"
                    case FinetuningMode.LORA:
                        description: str = (
                            f"{NAME_PREFIXES['lora_r']}"
                            f"{KV_SEP}"
                            f"{str(object=self.peft_config.r)}"
                            f"{ITEM_SEP}"
                            f"{NAME_PREFIXES['lora_alpha']}"
                            f"{KV_SEP}"
                            f"{str(object=self.peft_config.lora_alpha)}"
                            f"{ITEM_SEP}"
                            f"{NAME_PREFIXES['lora_target_modules']}"
                            f"{KV_SEP}"
                            f"{target_modules_to_path_part(target_modules=self.peft_config.target_modules)}"
                            f"{ITEM_SEP}"
                            f"{NAME_PREFIXES['lora_dropout']}"
                            f"{KV_SEP}"
                            f"{self.peft_config.lora_dropout}"
                            f"{ITEM_SEP}"
                            f"{NAME_PREFIXES['use_rslora']}"
                            f"{KV_SEP}"
                            f"{self.peft_config.use_rslora}"
                        )
                    case _:
                        msg: str = f"Unknown finetuning_mode: {self.peft_config.finetuning_mode = }"
                        raise ValueError(
                            msg,
                        )
            case DescriptionType.SHORT:
                match self.peft_config.finetuning_mode:
                    case FinetuningMode.STANDARD:
                        description: str = "None"
                    case FinetuningMode.LORA:
                        description: str = (
                            f"{self.peft_config.r}"
                            f"{short_description_separator}"
                            f"{self.peft_config.lora_alpha}"
                            f"{short_description_separator}"
                            f"{target_modules_to_path_part(target_modules=self.peft_config.target_modules)}"
                            f"{short_description_separator}"
                            f"{self.peft_config.lora_dropout}"
                            f"{short_description_separator}"
                            f"{self.peft_config.use_rslora}"
                        )
                    case _:
                        msg: str = f"Unknown finetuning_mode: {self.peft_config.finetuning_mode = }"
                        raise ValueError(
                            msg,
                        )
            case _:
                msg: str = f"Unknown {description_type = }"
                raise ValueError(
                    msg,
                )

        return description


def target_modules_to_path_part(
    target_modules: list[str] | str | None,
) -> str:
    """Convert the target_modules to a path part."""
    if target_modules is None:
        target_modules_path_part: str = "None"
    elif isinstance(
        target_modules,
        str,
    ):
        target_modules_path_part: str = target_modules
    else:
        target_modules_path_part: str = convert_list_to_path_part(
            input_list=target_modules,
        )

    return target_modules_path_part
