# Copyright 2024-2025
# [ANONYMIZED_INSTITUTION],
# [ANONYMIZED_FACULTY],
# [ANONYMIZED_DEPARTMENT]
#
# Authors:
# AUTHOR_1 (author1@example.com)
# AUTHOR_2 (author2@example.com)
#
# Code generation tools and workflows:
# First versions of this code were potentially generated
# with the help of AI writing assistants including
# GitHub Copilot, ChatGPT, Microsoft Copilot, Google Gemini.
# Afterwards, the generated segments were manually reviewed and edited.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import itertools
import logging
import pathlib
import pickle
from collections.abc import Iterator
from typing import Any

from topollm.data_processing.dictionary_handling import (
    dictionary_to_partial_path,
    filter_list_of_dictionaries_by_key_value_pairs,
)
from topollm.typing.enums import Verbosity

default_logger: logging.Logger = logging.getLogger(
    name=__name__,
)


def get_fixed_parameter_combinations(
    loaded_data: list[dict[str, Any]],
    fixed_keys: list[str],
    additional_fixed_params: dict[str, Any] | None = None,
) -> Iterator[dict[str, Any]]:
    """Generate combinations of fixed parameters from loaded data.

    This function extracts unique values for each key in `fixed_keys` from the provided
    `loaded_data`, and optionally merges these with `additional_fixed_params` (where each
    key in additional_fixed_params is fixed to a single value).
    It returns an iterator over dictionaries, each representing a unique combination of fixed parameters.

    Args:
        loaded_data: List of dictionaries containing data.
        fixed_keys: List of keys for which to generate combinations of values.
        additional_fixed_params: Additional key-value pairs to be fixed.

    Yields:
        A dictionary representing a combination of fixed parameters.

    """
    # Step 1: Extract unique options for each fixed key from the loaded data.
    # For each key in fixed_keys, create a set of all values that appear in the loaded_data.
    # This ensures that we only consider distinct values for each parameter.
    fixed_options: dict[str, set] = {
        key: {entry[key] for entry in loaded_data if key in entry}  # set comprehension for unique values
        for key in fixed_keys
    }

    # Step 2: Incorporate additional fixed parameters.
    # If additional_fixed_params are provided, override or add those parameters
    # in the fixed_options dictionary.
    # Each key in additional_fixed_params is set to a singleton
    # set containing its fixed value, ensuring that only this value is used.
    if additional_fixed_params:
        for key, value in additional_fixed_params.items():
            fixed_options[key] = {value}

    # Step 3: Prepare for generating the Cartesian product.
    # Create a list of keys from the fixed_options dictionary. The order of keys will
    # determine the order of values in each combination generated by the Cartesian product.
    keys_list: list[str] = list(fixed_options.keys())

    # Step 4: Generate the Cartesian product of all fixed parameter options.
    # For each key in keys_list, we retrieve the set of possible values.
    # itertools.product then computes the Cartesian product over these sets,
    # yielding tuples where each tuple represents one unique combination of fixed parameter values.
    for combination in itertools.product(*(fixed_options[key] for key in keys_list)):
        # Step 5: Map the tuple of values back to a dictionary using the keys.
        # The dict(zip(...)) creates a dictionary where each key from keys_list is paired with
        # the corresponding value from the current combination tuple.
        yield dict(
            zip(
                keys_list,
                combination,
                strict=True,
            ),
        )


def derive_base_model_partial_name(
    model_partial_name: str,
) -> str:
    """Derive the base model partial name from the model partial name."""
    if "model=roberta-base" in model_partial_name:
        # This model is derived from a roberta-base model.
        base_model_model_partial_name = "model=roberta-base"
    elif "model=bert-base-uncased" in model_partial_name:
        # This model is derived from a bert-base-uncased model.
        base_model_model_partial_name = "model=bert-base-uncased"
    else:
        # This model partial name does not match any known model.
        # Use the roberta-base model as the base model.
        base_model_model_partial_name = "model=roberta-base"

    return base_model_model_partial_name


def write_sorted_data_to_disk(
    sorted_data: list[dict],
    plots_output_dir: pathlib.Path,
    verbosity: Verbosity,
    logger: logging.Logger,
) -> None:
    """Write the sorted data to disk."""
    data_to_save: list[dict] = sorted_data

    sorted_data_output_file_path: pathlib.Path = pathlib.Path(
        plots_output_dir,
        "sorted_data_list_of_dicts_with_arrays.pkl",
    )

    # Create the directory if it does not exist.
    sorted_data_output_file_path.parent.mkdir(
        parents=True,
        exist_ok=True,
    )

    if verbosity >= Verbosity.NORMAL:
        logger.info(
            msg=f"Saving sorted data to {sorted_data_output_file_path = } ...",  # noqa: G004 - low overhead
        )

    with sorted_data_output_file_path.open(
        mode="wb",
    ) as file:
        pickle.dump(
            obj=data_to_save,
            file=file,
        )

    if verbosity >= Verbosity.NORMAL:
        logger.info(
            msg=f"Saving sorted data to {sorted_data_output_file_path = } DONE",  # noqa: G004 - low overhead
        )


def construct_plots_over_checkpoints_output_dir_from_filter_key_value_pairs(
    output_root_dir: pathlib.Path,
    filter_key_value_pairs: dict[
        str,
        str | int,
    ],
    verbosity: Verbosity = Verbosity.NORMAL,
    logger: logging.Logger = default_logger,
) -> pathlib.Path:
    """Construct the plots output directory from the filter key-value pairs."""
    plots_output_dir: pathlib.Path = pathlib.Path(
        output_root_dir,
        "plots_over_checkpoints",
        dictionary_to_partial_path(
            dictionary=filter_key_value_pairs,
        ),
    )
    if verbosity >= Verbosity.NORMAL:
        logger.info(
            msg=f"{plots_output_dir = }",  # noqa: G004 - low overhead
        )

    return plots_output_dir


def construct_mean_plots_over_model_checkpoints_output_dir_from_filter_key_value_pairs(
    output_root_dir: pathlib.Path,
    filter_key_value_pairs: dict[
        str,
        str | int,
    ],
    verbosity: Verbosity = Verbosity.NORMAL,
    logger: logging.Logger = default_logger,
) -> pathlib.Path:
    """Construct the mean plots output directory from the filter key-value pairs."""
    plots_output_dir: pathlib.Path = pathlib.Path(
        output_root_dir,
        "mean_over_checkpoints_for_different_seeds",
        dictionary_to_partial_path(
            dictionary=filter_key_value_pairs,
        ),
    )
    if verbosity >= Verbosity.NORMAL:
        logger.info(
            msg=f"{plots_output_dir = }",  # noqa: G004 - low overhead
        )

    return plots_output_dir


def add_base_model_data(
    loaded_data: list[dict],
    base_model_model_partial_name: str,
    filter_key_value_pairs: dict[
        str,
        str | int,
    ],
    filtered_data: list[dict],
    logger: logging.Logger = default_logger,
) -> list[dict]:
    """Add the base model data to the list of data to plot."""
    filtered_data_with_added_base_model: list[dict] = filtered_data.copy()

    # Build the dictionary which matches the same parameters as in the filter
    # but has the model_partial_name from the base model.
    filter_key_value_pairs_base_model: dict = {
        **filter_key_value_pairs,
        "model_partial_name": base_model_model_partial_name,
        "model_seed": None,  # The base model always has model_seed=None.
    }
    filtered_data_base_model: list[dict] = filter_list_of_dictionaries_by_key_value_pairs(
        list_of_dicts=loaded_data,
        key_value_pairs=filter_key_value_pairs_base_model,
    )
    if len(filtered_data_base_model) == 0:
        logger.warning(
            msg=f"No base model data found for {filter_key_value_pairs = }.",  # noqa: G004 - low overhead
        )
        logger.warning(
            msg="Proceeding without adding base model data.",
        )
    elif len(filtered_data_base_model) == 1:
        logger.info(
            msg=f"Unique base model data found for {filter_key_value_pairs = }.",  # noqa: G004 - low overhead
        )
        filtered_data_base_model_dict: dict = filtered_data_base_model[0]
        # Add the base model data to the list of data to plot.
        filtered_data_with_added_base_model.append(
            filtered_data_base_model_dict,
        )
    elif len(filtered_data_base_model) > 1:
        logger.warning(
            f"Ambiguous base model data ({len(filtered_data_base_model)} entries) "  # noqa: G004 - low overhead
            f"found for {filter_key_value_pairs = }.",
        )
        logger.warning(
            msg="Will use the first entry.",
        )
        filtered_data_base_model_dict: dict = filtered_data_base_model[0]
        # Add the base model data to the list of data to plot.
        filtered_data_with_added_base_model.append(
            filtered_data_base_model_dict,
        )

    return filtered_data_with_added_base_model
