import json


def normalize_description(description_str):
    try:
        description = json.loads(description_str)
    except json.JSONDecodeError:
        return description_str

    if not isinstance(description, dict):
        return description_str

    return description


def normalize_annotation(anno_str):
    try:
        anno = json.loads(anno_str)
    except json.JSONDecodeError:
        error_message = "Invalid JSON format. "
        return None, error_message

    if not isinstance(anno, dict):
        error_message = "Annotation is not a dictionary. "
        return None, error_message

    ability_list = ["Recognition", "Understanding", "Grounding", "Reasoning"]
    for ability in ability_list:
        if ability not in anno:
            error_message = f"Missing required ability: {ability}. "
            return None, error_message

        ability_content = anno[ability]
        if ability_content == "NA":
            continue

        if not isinstance(ability_content, dict):
            error_message = f"The content of '{ability}' is not a dictionary. "
            return None, error_message

        if "Question" not in ability_content or not isinstance(ability_content["Question"], str):
            error_message = f"The 'Question' field of '{ability}' is missing or not a string. "
            return None, error_message

        if ("Options" not in ability_content or not isinstance(ability_content["Options"], dict)
                or set(ability_content["Options"].keys()) != {"A", "B", "C", "D"}):
            error_message = (f"The 'Options' field of '{ability}' is missing, not a dictionary, "
                             f"or does not contain exactly 'A', 'B', 'C', and 'D'. ")
            return None, error_message

        if "Answer" not in ability_content or ability_content["Answer"] not in ["A", "B", "C", "D"]:
            error_message = f"The 'Answer' field of '{ability}' is missing or not one of 'A', 'B', 'C', or 'D'. "
            return None, error_message

    return anno, None

