import torch
from torch import Tensor
import torch.nn as nn
from typing import Type, Any, Callable, Union, List, Optional


def conv3x3(in_planes: int, out_planes: int, stride: int = 1, groups: int = 1, dilation: int = 1) -> nn.Conv2d:
    """
    3x3 convolution with padding.

    Creates a 3x3 convolutional layer with specified parameters, ensuring consistent padding
    based on dilation rate.

    Args:
        in_planes (int): Number of input channels
        out_planes (int): Number of output channels
        stride (int): Stride for the convolution
        groups (int): Number of groups for grouped convolution
        dilation (int): Dilation rate for atrous convolution

    Returns:
        nn.Conv2d: 3x3 convolutional layer
    """
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes: int, out_planes: int, stride: int = 1) -> nn.Conv2d:
    """
    1x1 convolution.

    Args:
        in_planes (int): Number of input channels
        out_planes (int): Number of output channels
        stride (int): Stride for the convolution

    Returns:
        nn.Conv2d: 1x1 convolutional layer
    """
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    """
    Basic residual block for ResNet architectures.

    Implements the residual connection: output = activation(input + F(input)),
    where F is a sequence of convolutional layers.
    """
    expansion: int = 1

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[nn.Module] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., nn.Module]] = None, 
        has_bn = True,
    ) -> None:
        """
        Initialize a BasicBlock.

        Args:
            inplanes (int): Number of input channels
            planes (int): Number of output channels for the first convolution
            stride (int): Stride for the first convolution
            downsample (nn.Module, optional): Module to downsample input for residual connection
            groups (int): Number of groups for grouped convolution (must be 1 for BasicBlock)
            base_width (int): Base width for group convolution (must be 64 for BasicBlock)
            dilation (int): Dilation rate (must be 1 for BasicBlock)
            norm_layer (Callable, optional): Normalization layer (default: BatchNorm2d)
            has_bn (bool): Whether to include batch normalization layers
        """
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        self.conv1 = conv3x3(inplanes, planes, stride)
        if has_bn:
            self.bn2 = norm_layer(planes)
        else:
            self.bn2 = nn.Identity()
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        if has_bn:
            self.bn3 = norm_layer(planes)
        else:
            self.bn3 = nn.Identity()
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor) -> Tensor:
        """
        Forward pass of the BasicBlock.

        Applies two convolutional layers with normalization and activation,
        then adds the residual connection (input) to the output.

        Args:
            x (Tensor): Input tensor of shape (batch_size, inplanes, height, width)

        Returns:
            Tensor: Output tensor of shape (batch_size, planes * expansion, height', width')
        """
        identity = x

        out = self.conv1(x)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out
    

class Bottleneck(nn.Module):
    """
    Bottleneck residual block for deeper ResNet architectures.

    Implements a bottleneck structure with three convolutions and a residual connection.
    """
    expansion: int = 4

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[nn.Module] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., nn.Module]] = None,
        has_bn = True,
    ) -> None:
        """
        Initialize a Bottleneck block.

        Args:
            inplanes (int): Number of input channels
            planes (int): Number of output channels for the bottleneck
            stride (int): Stride for the convolution
            downsample (nn.Module, optional): Downsampling module for residual connection
            groups (int): Number of groups for grouped convolution
            base_width (int): Base width for group convolution
            dilation (int): Dilation rate
            norm_layer (Callable, optional): Normalization layer
            has_bn (bool): Whether to include batch normalization layers
        """
        super().__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.0)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        if has_bn:
            self.bn1 = norm_layer(width)
        else:
            self.bn1 = nn.Identity()
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        if has_bn:
            self.bn2 = norm_layer(width)
        else:
            self.bn2 = nn.Identity()
        self.conv3 = conv1x1(width, planes * self.expansion)
        if has_bn:
            self.bn3 = norm_layer(planes * self.expansion)
        else:
            self.bn3 = nn.Identity()
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor) -> Tensor:
        """
        Forward pass of the Bottleneck block.

        Applies three convolutional layers with normalization and activation,
        then adds the residual connection (input) to the output.

        Args:
            x (Tensor): Input tensor

        Returns:
            Tensor: Output tensor
        """
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out
    

class ResNet(nn.Module):
    """
    ResNet architecture for image classification.

    Base class for ResNet models with configurable depth (via block count) and output classes.
    """

    def __init__(
        self,
        block: BasicBlock,
        layers: List[int],
        features: List[int] = [64, 128, 256, 512],
        num_classes: int = 1000,
        zero_init_residual: bool = False,
        groups: int = 1,
        width_per_group: int = 64,
        replace_stride_with_dilation: Optional[List[bool]] = None,
        norm_layer: Optional[Callable[..., nn.Module]] = None, 
        has_bn = True,
        bn_block_num = 4, 
    ) -> None:
        """
        Initialize a ResNet model.

        Args:
            block (BasicBlock): Type of residual block to use
            layers (list[int]): Number of blocks in each layer
            features (list[int]): Output channels for each layer
            num_classes (int): Number of output classes
            zero_init_residual (bool): If True, zero-initialize residual connections
            groups (int): Number of groups for grouped convolutions
            width_per_group (int): Width per group for grouped convolutions
            replace_stride_with_dilation (list[bool]): Whether to replace stride with dilation
            norm_layer (Callable, optional): Normalization layer (default: BatchNorm2d)
            has_bn (bool): Whether to include batch normalization layers
            bn_block_num (int): Number of blocks with batch normalization
        """
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3, bias=False)
        if has_bn:
            self.bn1 = norm_layer(self.inplanes)
        else:
            self.bn1 = nn.Identity()
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

        self.layers = []
        self.layers.extend(self._make_layer(block, 64, layers[0], has_bn=has_bn and (bn_block_num > 0)))
        for num in range(1, len(layers)):
            self.layers.extend(self._make_layer(block, features[num], layers[num], stride=2,
                                       dilate=replace_stride_with_dilation[num-1], 
                                       has_bn=has_bn and (num < bn_block_num)))

        for i, layer in enumerate(self.layers):
            setattr(self, f'layer_{i}', layer)

        self.avgpool = nn.Sequential(
            nn.AdaptiveAvgPool2d((1, 1)),
            nn.Flatten()
        )
        self.fc = nn.Linear(features[len(layers)-1] * block.expansion, num_classes)

        # self.fc = nn.Sequential(
        #     nn.AdaptiveAvgPool2d((1, 1)), 
        #     nn.Flatten(), 
        #     nn.Linear(features[len(layers)-1] * block.expansion, num_classes)
        # )

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck) and m.bn3.weight is not None:
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock) and m.bn2.weight is not None:
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(self, block: BasicBlock, planes: int, blocks: int,
                    stride: int = 1, dilate: bool = False, has_bn=True) -> List:
        """
        Create a sequence of residual blocks for one layer of the ResNet.

        Args:
            block (BasicBlock): Block type
            planes (int): Output channels for the blocks
            blocks (int): Number of blocks
            stride (int): Stride for the first block
            dilate (bool): Whether to use dilation
            has_bn (bool): Whether to include batch normalization

        Returns:
            List: List of block modules
        """
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            if has_bn:
                downsample = nn.Sequential(
                    conv1x1(self.inplanes, planes * block.expansion, stride),
                    norm_layer(planes * block.expansion),
                )
            else:
                downsample = nn.Sequential(
                    conv1x1(self.inplanes, planes * block.expansion, stride),
                    nn.Identity(),
                )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer, has_bn))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer, has_bn=has_bn))

        return layers

    def _forward_impl(self, x: Tensor) -> Tensor:
        """
        Implementation of the forward pass for ResNet.

        Args:
            x (Tensor): Input tensor

        Returns:
            Tensor: Output logits
        """
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        for i in range(len(self.layers)):
            layer = getattr(self, f'layer_{i}')
            x = layer(x)

        x = self.avgpool(x)
        x = self.fc(x)

        return x

    def forward(self, x: Tensor) -> Tensor:
        """
        Forward pass for ResNet.

        Args:
            x (Tensor): Input tensor

        Returns:
            Tensor: Output logits
        """
        return self._forward_impl(x)


def resnet152(**kwargs: Any) -> ResNet: 
    """
    Construct a ResNet-152 model.

    Returns:
        ResNet: ResNet-152 instance
    """
    return ResNet(Bottleneck, [3, 8, 36, 3], **kwargs)

def resnet101(**kwargs: Any) -> ResNet: 
    """
    Construct a ResNet-101 model.

    Returns:
        ResNet: ResNet-101 instance
    """
    return ResNet(Bottleneck, [3, 4, 23, 3], **kwargs)

def resnet50(**kwargs: Any) -> ResNet: 
    """
    Construct a ResNet-50 model.

    Returns:
        ResNet: ResNet-50 instance
    """
    return ResNet(Bottleneck, [3, 4, 6, 3], **kwargs)

def resnet34(**kwargs: Any) -> ResNet: 
    """
    Construct a ResNet-34 model.

    Returns:
        ResNet: ResNet-34 instance
    """
    return ResNet(BasicBlock, [3, 4, 6, 3], **kwargs)

def resnet18(**kwargs: Any) -> ResNet: # 18 = 2 + 2 * (2 + 2 + 2 + 2)
    """
    Construct a ResNet-18 model.

    Returns:
        ResNet: ResNet-18 instance
    """
    return ResNet(BasicBlock, [2, 2, 2, 2], **kwargs)

def resnet10(**kwargs: Any) -> ResNet: # 10 = 2 + 2 * (1 + 1 + 1 + 1)
    """
    Construct a ResNet-10 model.

    Returns:
        ResNet: ResNet-10 instance
    """
    return ResNet(BasicBlock, [1, 1, 1, 1], **kwargs)

def resnet8(**kwargs: Any) -> ResNet: # 8 = 2 + 2 * (1 + 1 + 1)
    """
    Construct a ResNet-8 model.

    Returns:
        ResNet: ResNet-8 instance
    """
    return ResNet(BasicBlock, [1, 1, 1], **kwargs)

def resnet6(**kwargs: Any) -> ResNet: # 6 = 2 + 2 * (1 + 1)
    """
    Construct a ResNet-6 model.

    Returns:
        ResNet: ResNet-6 instance
    """
    return ResNet(BasicBlock, [1, 1], **kwargs)

def resnet4(**kwargs: Any) -> ResNet: # 4 = 2 + 2 * (1)
    """
    Construct a ResNet-4 model.

    Returns:
        ResNet: ResNet-4 instance
    """
    return ResNet(BasicBlock, [1], **kwargs)
