import numpy as np


# Helper functions

def _adjust_indices(array: np.ndarray, selected_indices: np.ndarray, start_idx: int) -> np.ndarray:
    """
    Adjust indices for partitioned calculations.

    Args:
    array (np.ndarray): Input array of 0s and 1s.
    selected_indices (np.ndarray): Indices chosen from the main array.
    start_idx (int): Start index of the partition.

    Returns:
    np.ndarray: Adjusted indices.
    """ 
    return[idx - start_idx for idx in selected_indices if start_idx <= idx < start_idx + len(array)]


def estimate_accuracy(array: np.ndarray, n_trials: int = 100000, n_draws: int = 3) -> float:
    """Calculate the accuracy metric based on draws from the array.

    Args:
    array (np.ndarray): Input array of 0s and 1s.
    n_trials (int): Number of trials to run for estimation.
    n_draws (int): Number of draws in each trial.

    Returns:
    float: Estimated accuracy.
    """
    # Ensure the number of draws is equal to the number of ones in the array
    assert n_draws == np.sum(array), "n_draws must be equal to the number of ones in the array."

    accum = 0
    total_length = len(array)
    
    for _ in range(n_trials):
        selected = np.random.choice(array, n_draws, replace=False)
        ones_drawn = np.sum(selected)
        count = total_length - 2 * (n_draws - ones_drawn)
        accum += count / total_length

    return accum / n_trials

def estimate_tnr(array: np.ndarray, n_trials: int = 100000, n_draws: int = 3) -> float:
    """Calculate the TNR metric based on draws from the array.

    Args:
    array (np.ndarray): Input array of 0s and 1s.
    n_trials (int): Number of trials to run for estimation.
    n_draws (int): Number of draws in each trial.

    Returns:
    float: Estimated TNR."""
    metric_accum = 0
    for _ in range(n_trials):
        selected = np.random.choice(array, n_draws, replace=False)
        count_zeros = n_draws - np.sum(selected)
        metric_accum += count_zeros / n_draws
    return metric_accum / n_trials

def estimate_tpr(array: np.ndarray, n_trials: int = 100000, n_draws: int = 3) -> float:
    """Calculate the TPR metric based on draws from the array.
        Args:
    array (np.ndarray): Input array of 0s and 1s.
    n_trials (int): Number of trials to run for estimation.
    n_draws (int): Number of draws in each trial.

    Returns:
    float: Estimated TPR."""
    metric_accum = 0
    for _ in range(n_trials):
        selected = np.random.choice(array, n_draws, replace=False)
        count_ones = np.sum(selected)
        metric_accum += count_ones / n_draws
    return metric_accum / n_trials


def partition_metric(array: np.ndarray, partition_sizes: list, metric_type: str = 'tnr', n_trials=100000, n_draws:int=3) -> list:
    """
    Estimate a metric (TNR or TPR) for multiple partitions of the input array.

    Parameters:
    - array (np.ndarray): Input array of 0s and 1s.
    - partition_sizes (list): List of sizes defining partitions within the array.
    - metric_type (str): Type of metric to estimate. Either 'tnr' (True Negative Rate) or 'tpr' (True Positive Rate).
    - n_trials (int): Number of random selections to make for estimating the metric.

    Returns:
    - list: A list of estimated metrics for each partition.
    """
    partition_metrics = []
    start_idx = 0

    for size in partition_sizes:
        metric_sum = 0  
        for _ in range(n_trials):
            selected_indices = np.random.choice(len(array), n_draws, replace=False)
            metric_value = estimate_partition_metric(array[start_idx:start_idx + size], selected_indices, start_idx, metric_type)
            metric_sum += metric_value

        avg_metric = metric_sum / n_trials
        partition_metrics.append(avg_metric)
        start_idx += size

    return partition_metrics

def estimate_partition_metric(array: np.ndarray, selected_indices: np.ndarray, start_idx:int, metric_type:str='tnr') -> float:
    """
    Compute the metric (TNR or TPR) for a specific partition of the input array based on given indices.

    Args:
    - array (np.ndarray): Input array of 0s and 1s.
    - selected_indices (np.ndarray): Indices chosen from the main array.
    - start_idx (int): Start index of the partition.
    - size (int): Size of the partition.
    - metric_type (str): Type of metric to estimate. Either 'tnr' (True Negative Rate) or 'tpr' (True Positive Rate).

    Returns:
    - float: Computed metric value for the partition.
    """
    # Adjust indices to partition frame of reference
    adjusted_indices = _adjust_indices(array, selected_indices, start_idx)
    chosen_indices = array[adjusted_indices]
    if metric_type == 'tnr':
        count = np.sum(array) - np.sum(chosen_indices)  # count zeros
    elif metric_type == 'tpr':
        count = np.sum(chosen_indices)  # count ones
    elif metric_type == 'accuracy':
        selection = np.full(len(array), 0)
        selection[adjusted_indices] = 1
        count = np.sum(array == selection) / len(array)
        return count
    else:
        raise ValueError(f"Invalid metric_type: {metric_type}. Expected 'tnr' or 'tpr' or 'accuracy.")
    return count / np.sum(array)


def calculate_chance_tnr(array: np.ndarray, partition_sizes: list = None) -> float:
    """ 
    Calculate the chance True Negative Rate (TNR) for a given array.
    
    Args:
        array (np.ndarray): Input array of 0s and 1s.
        partition_sizes (list, optional): List of sizes defining partitions within the array.

    Returns:
        float or list: TNR value (global or partitioned based on the input).
    """
    if partition_sizes:
        # Use the partitioned TNR calculation
        return partition_metric(array, partition_sizes, metric_type='tnr')
    else:
        # Use the global TNR calculation
        return estimate_tnr(array)

def calculate_chance_tpr(array: np.ndarray, partition_sizes: list = None) -> float:
    """ 
    Calculate the chance True Positive Rate (TPR) for a given array.
    
    Args:
        array (np.ndarray): Input array of 0s and 1s.
        partition_sizes (list, optional): List of sizes defining partitions within the array.

    Returns:
        float or list: TPR value (global or partitioned based on the input).
    """
    if partition_sizes:
        # Use the partitioned TPR calculation
        return partition_metric(array, partition_sizes,  metric_type='tpr')
    else:
        # Use the global TPR calculation
        return estimate_tpr(array)

def calculate_chance_acc(array: np.ndarray, partition_sizes: list = None) -> float:
    """Calculate the chance accuracy for a given array.
    Args:
        array (np.ndarray): Input array of 0s and 1s.
        partition_sizes (list, optional): List of sizes defining partitions within the array.
    Returns:
        float or list: Accuracy value (global or partitioned based on the input)."""
    
    if partition_sizes:
        # Use the partitioned TPR calculation
        return partition_metric(array, partition_sizes,  metric_type='accuracy')
    else:
        # Use the global TPR calculation
        return estimate_accuracy(array)

