// 生成示例数据
function generateDemoData() {
    const data = [];
    
    // 生成一些随机形状
    const shapes = [
        {type: 'sphere', x: 5, y: 5, z: 5, r: 4, category: 0},
        {type: 'cube', x: 15, y: 5, z: 5, size: 5, category: 3},
        {type: 'sphere', x: 5, y: 15, z: 15, r: 5, category: 6},
        {type: 'cube', x: 15, y: 15, z: 15, size: 4, category: 9},
    ];
    
    for (let x = 0; x < CONFIG.GRID_SIZE; x++) {
        for (let y = 0; y < CONFIG.GRID_SIZE; y++) {
            for (let z = 0; z < CONFIG.GRID_SIZE; z++) {
                let category = 0;
                
                // 检查点是否在任何形状内
                for (const shape of shapes) {
                    if (shape.type === 'sphere') {
                        const dx = x - shape.x;
                        const dy = y - shape.y;
                        const dz = z - shape.z;
                        const distance = Math.sqrt(dx*dx + dy*dy + dz*dz);
                        
                        if (distance <= shape.r) {
                            category = shape.category;
                            break;
                        }
                    } else if (shape.type === 'cube') {
                        if (Math.abs(x - shape.x) <= shape.size/2 &&
                            Math.abs(y - shape.y) <= shape.size/2 &&
                            Math.abs(z - shape.z) <= shape.size/2) {
                            category = shape.category;
                            break;
                        }
                    }
                }
                
                // 随机添加一些噪点
                if (category === 0 && Math.random() < 0.01) {
                    category = Math.floor(Math.random() * 11) + 1;
                }
                
                if (category > 0) {
                    data.push({x, y, z, category});
                }
            }
        }
    }
    
    return data;
}

// 生成示例中心点数据
function generateDemoCenters() {
    return [
        {x: 5, y: 5, z: 5, category: 0},
        {x: 15, y: 5, z: 5, category: 3},
        {x: 5, y: 15, z: 15, category: 6},
        {x: 15, y: 15, z: 15, category: 9}
    ];
}

// 计算中心点
function calculateCenters() {
    // 按类别分组
    const groups = {};
    state.voxelData.forEach(voxel => {
        if (!groups[voxel.category]) {
            groups[voxel.category] = [];
        }
        groups[voxel.category].push(voxel);
    });
    
    // 计算每个组的中心点
    state.centerPoints = [];
    Object.keys(groups).forEach(category => {
        const points = groups[category];
        const sum = points.reduce((acc, point) => {
            acc.x += point.x;
            acc.y += point.y;
            acc.z += point.z;
            return acc;
        }, {x: 0, y: 0, z: 0});
        
        state.centerPoints.push({
            x: sum.x / points.length,
            y: sum.y / points.length,
            z: sum.z / points.length,
            category: parseInt(category)
        });
    });
    
    elements.centerCount.textContent = `中心点: ${state.centerPoints.length}`;
}

// 处理上传的JSON文件
function handleJsonFile(file, isPointsFile = false) {
    elements.loading.style.display = 'block';
    elements.loading.textContent = '正在解析数据...';
    
    const reader = new FileReader();
    reader.onload = function(e) {
        try {
            const jsonData = JSON.parse(e.target.result);
            
            // 验证数据格式
            if (!Array.isArray(jsonData)) {
                throw new Error('数据格式错误: 应为数组');
            }
            
            if (jsonData.length > 0) {
                const firstItem = jsonData[0];
                const requiredFields = ['x', 'y', 'z', 'category'];
                    
                for (const field of requiredFields) {
                    if (!firstItem.hasOwnProperty(field)) {
                        throw new Error(`数据格式错误: 每个项应包含${requiredFields.join(', ')}字段`);
                    }
                }
            }
            
            elements.loading.style.display = 'none';
            
            if (isPointsFile) {
                state.centerPointsFromFile = jsonData;
                if (state.useFileCenters) {
                    state.centerPoints = state.centerPointsFromFile;
                    elements.centerSource.textContent = '中心点来源: 文件';
                    drawScene();
                }
            } else {
                // 检查是否有value字段来区分中心点数据和体素数据
                if (jsonData.length > 0 && jsonData[0].hasOwnProperty('value')) {
                    // 这是中心点数据
                    state.centerPointsFromFile = jsonData;
                    if (state.useFileCenters) {
                        state.centerPoints = state.centerPointsFromFile;
                        elements.centerSource.textContent = '中心点来源: 文件';
                        drawScene();
                    }
                } else {
                    // 这是体素数据
                    state.voxelData = jsonData;
                    if (!state.useFileCenters) {
                        calculateCenters();
                    }
                    drawScene();
                }
            }
        } catch (error) {
            elements.loading.innerHTML = `<div class="error">解析JSON失败: ${error.message}</div>`;
        }
    };
    
    reader.onerror = function() {
        elements.loading.innerHTML = '<div class="error">读取文件失败</div>';
    };
    
    reader.readAsText(file);
}

// 保存中心点数据
function saveCenterPoints() {
    // 创建要保存的数据结构
    const dataToSave = state.centerPoints.map(center => ({
        x: parseFloat(center.x.toFixed(2)),
        y: parseFloat(center.y.toFixed(2)),
        z: parseFloat(center.z.toFixed(2)),
        category: center.category
    }));
    
    // 创建JSON字符串
    const jsonString = JSON.stringify(dataToSave, null, 2);
    
    // 创建Blob对象
    const blob = new Blob([jsonString], { type: 'application/json' });
    
    // 创建下载链接
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'centers.json';
    a.click();
    
    // 清理URL
    URL.revokeObjectURL(url);
    
    alert('中心点数据已保存为centers.json文件');
}

// 切换中心点数据源
function toggleCenterSource(useFile) {
    state.useFileCenters = useFile;
    
    if (state.useFileCenters) {
        document.getElementById('calc-centers').classList.remove('active-source');
        document.getElementById('load-points-file').classList.add('active-source');
        document.getElementById('upload-points-area').style.display = 'block';
        
        if (state.centerPointsFromFile.length > 0) {
            state.centerPoints = state.centerPointsFromFile;
            elements.centerSource.textContent = '中心点来源: 文件';
        } else {
            elements.centerSource.textContent = '中心点来源: 文件(未上传)';
        }
    } else {
        document.getElementById('calc-centers').classList.add('active-source');
        document.getElementById('load-points-file').classList.remove('active-source');
        document.getElementById('upload-points-area').style.display = 'none';
        
        calculateCenters();
        elements.centerSource.textContent = '中心点来源: 自动计算';
    }
    
    drawScene();
}

// 导出函数
window.generateDemoData = generateDemoData;
window.generateDemoCenters = generateDemoCenters;
window.calculateCenters = calculateCenters;
window.handleJsonFile = handleJsonFile;
window.saveCenterPoints = saveCenterPoints;
window.toggleCenterSource = toggleCenterSource;