// 3D到2D投影 - 使用四元数避免万向节锁
function project(x, y, z) {
    // 将欧拉角转换为四元数
    const cy = Math.cos(state.rotationY * 0.5);
    const sy = Math.sin(state.rotationY * 0.5);
    const cp = Math.cos(state.rotationX * 0.5);
    const sp = Math.sin(state.rotationX * 0.5);
    const cr = Math.cos(state.rotationZ * 0.5);
    const sr = Math.sin(state.rotationZ * 0.5);
    
    // 四元数
    const w = cr * cp * cy + sr * sp * sy;
    const xq = sr * cp * cy - cr * sp * sy;
    const yq = cr * sp * cy + sr * cp * sy;
    const zq = cr * cp * sy - sr * sp * cy;
    
    // 使用四元数旋转点
    const ix = w * x + yq * z - zq * y;
    const iy = w * y + zq * x - xq * z;
    const iz = w * z + xq * y - yq * x;
    const iw = -xq * x - yq * y - zq * z;
    
    // 应用旋转
    const xr = ix * w + iw * -xq + iy * -zq - iz * -yq;
    const yr = iy * w + iw * -yq + iz * -xq - ix * -zq;
    const zr = iz * w + iw * -zq + ix * -yq - iy * -xq;
    
    // 应用缩放和平移
    const scaleFactor = 10 * state.scale;
    const screenX = (xr * scaleFactor) + elements.canvas.width/2 + state.offsetX;
    const screenY = (yr * scaleFactor) + elements.canvas.height/2 + state.offsetY;
    
    // 返回深度信息用于排序
    return { x: screenX, y: screenY, z: zr };
}

// 反向投影：从屏幕坐标转换为3D坐标（近似）
function unproject(screenX, screenY, depth) {
    // 转换为相对于画布中心的坐标
    const x = (screenX - elements.canvas.width/2 - state.offsetX) / (10 * state.scale);
    const y = (screenY - elements.canvas.height/2 - state.offsetY) / (10 * state.scale);
    const z = depth;
    
    // 反向旋转（使用四元数的逆）
    const cy = Math.cos(-state.rotationY * 0.5);
    const sy = Math.sin(-state.rotationY * 0.5);
    const cp = Math.cos(-state.rotationX * 0.5);
    const sp = Math.sin(-state.rotationX * 0.5);
    const cr = Math.cos(-state.rotationZ * 0.5);
    const sr = Math.sin(-state.rotationZ * 0.5);
    
    const w = cr * cp * cy + sr * sp * sy;
    const xq = sr * cp * cy - cr * sp * sy;
    const yq = cr * sp * cy + sr * cp * sy;
    const zq = cr * cp * sy - sr * sp * cy;
    
    // 使用四元数反向旋转点
    const ix = w * x + yq * z - zq * y;
    const iy = w * y + zq * x - xq * z;
    const iz = w * z + xq * y - yq * x;
    const iw = -xq * x - yq * y - zq * z;
    
    const xr = ix * w + iw * -xq + iy * -zq - iz * -yq;
    const yr = iy * w + iw * -yq + iz * -xq - ix * -zq;
    const zr = iz * w + iw * -zq + ix * -yq - iy * -xq;
    
    return { x: xr, y: yr, z: zr };
}

// 绘制场景
function drawScene() {
    // 清除画布
    elements.ctx.clearRect(0, 0, elements.canvas.width, elements.canvas.height);
    
    if (state.voxelData.length === 0) return;
    
    // 绘制背景网格
    drawGrid();
    
    // 对体素按深度排序（从远到近）
    const sortedVoxels = state.voxelData
        .filter(voxel => state.selectedCategories.has(voxel.category))
        .map(voxel => {
            const pos = project(voxel.x, voxel.y, voxel.z);
            return { ...voxel, screenX: pos.x, screenY: pos.y, depth: pos.z };
        })
        .sort((a, b) => b.depth - a.depth);
    
    // 绘制体素
    const voxelSize = parseInt(elements.voxelSize.value);
    const opacity = parseInt(elements.opacity.value) / 100;
    
    sortedVoxels.forEach(voxel => {
        const color = CONFIG.COLORS[voxel.category % CONFIG.COLORS.length];
        const alpha = opacity * (0.7 + 0.3 * (voxel.depth + CONFIG.GRID_SIZE) / (2 * CONFIG.GRID_SIZE));
        
        elements.ctx.fillStyle = color.replace(')', `, ${alpha})`).replace('rgb', 'rgba');
        elements.ctx.fillRect(
            voxel.screenX - voxelSize/2,
            voxel.screenY - voxelSize/2,
            voxelSize,
            voxelSize
        );
        
        // 绘制边框
        elements.ctx.strokeStyle = 'rgba(255, 255, 255, 0.2)';
        elements.ctx.lineWidth = 1;
        elements.ctx.strokeRect(
            voxel.screenX - voxelSize/2,
            voxel.screenY - voxelSize/2,
            voxelSize,
            voxelSize
        );
    });
    
    // 绘制中心点
    if (state.showCenters && state.centerPoints.length > 0) {
        const centerSize = parseInt(elements.centerSize.value);
        
        state.centerPoints
            .filter(center => state.selectedCategories.has(center.category))
            .forEach((center, index) => {
                const pos = project(center.x, center.y, center.z);
                
                // 绘制中心点
                elements.ctx.fillStyle = CONFIG.CENTER_COLOR;
                elements.ctx.beginPath();
                elements.ctx.arc(pos.x, pos.y, centerSize, 0, Math.PI * 2);
                elements.ctx.fill();
                
                // 绘制外圈
                elements.ctx.strokeStyle = CONFIG.COLORS[center.category % CONFIG.COLORS.length];
                elements.ctx.lineWidth = 2;
                elements.ctx.beginPath();
                elements.ctx.arc(pos.x, pos.y, centerSize + 2, 0, Math.PI * 2);
                elements.ctx.stroke();
                
                // 如果正在拖动这个中心点，绘制特殊效果
                if (state.isDraggingCenter && state.draggedCenterIndex === index) {
                    elements.ctx.strokeStyle = '#FFD700';
                    elements.ctx.lineWidth = 3;
                    elements.ctx.beginPath();
                    elements.ctx.arc(pos.x, pos.y, centerSize + 4, 0, Math.PI * 2);
                    elements.ctx.stroke();
                }
                
                // 如果点被选中，绘制特殊效果
                if (center.selected) {
                    elements.ctx.strokeStyle = '#FFD700'; // 金色边框表示选中
                    elements.ctx.lineWidth = 2;
                    elements.ctx.beginPath();
                    elements.ctx.arc(pos.x, pos.y, centerSize + 6, 0, Math.PI * 2);
                    elements.ctx.stroke();
                }
            });
    }
    
    // 更新状态信息
    elements.voxelCount.textContent = `体素数量: ${sortedVoxels.length}`;
    updateRotationValues();
}

// 更新旋转值显示
function updateRotationValues() {
    elements.rotationXValue.textContent = `${Math.round(state.rotationX * 180 / Math.PI)}°`;
    elements.rotationYValue.textContent = `${Math.round(state.rotationY * 180 / Math.PI)}°`;
    elements.rotationZValue.textContent = `${Math.round(state.rotationZ * 180 / Math.PI)}°`;
    elements.scaleValue.textContent = `${state.scale.toFixed(1)}x`;
}

// 绘制背景网格
function drawGrid() {
    elements.ctx.strokeStyle = 'rgba(255, 255, 255, 0.1)';
    elements.ctx.lineWidth = 1;
    
    // 绘制XY平面网格
    for (let i = -CONFIG.GRID_SIZE; i <= CONFIG.GRID_SIZE; i += 5) {
        // X轴
        const startX = project(-CONFIG.GRID_SIZE, i, 0);
        const endX = project(CONFIG.GRID_SIZE, i, 0);
        elements.ctx.beginPath();
        elements.ctx.moveTo(startX.x, startX.y);
        elements.ctx.lineTo(endX.x, endX.y);
        elements.ctx.stroke();
        
        // Y轴
        const startY = project(i, -CONFIG.GRID_SIZE, 0);
        const endY = project(i, CONFIG.GRID_SIZE, 0);
        elements.ctx.beginPath();
        elements.ctx.moveTo(startY.x, startY.y);
        elements.ctx.lineTo(endY.x, endY.y);
        elements.ctx.stroke();
    }
    
    // 绘制坐标轴
    const origin = project(0, 0, 0);
    const xAxis = project(CONFIG.GRID_SIZE/2, 0, 0);
    const yAxis = project(0, CONFIG.GRID_SIZE/2, 0);
    const zAxis = project(0, 0, CONFIG.GRID_SIZE/2);
    
    // X轴 (红色)
    elements.ctx.strokeStyle = '#FF5252';
    elements.ctx.lineWidth = 2;
    elements.ctx.beginPath();
    elements.ctx.moveTo(origin.x, origin.y);
    elements.ctx.lineTo(xAxis.x, xAxis.y);
    elements.ctx.stroke();
    
    // Y轴 (绿色)
    elements.ctx.strokeStyle = '#69F0AE';
    elements.ctx.lineWidth = 2;
    elements.ctx.beginPath();
    elements.ctx.moveTo(origin.x, origin.y);
    elements.ctx.lineTo(yAxis.x, yAxis.y);
    elements.ctx.stroke();
    
    // Z轴 (蓝色)
    elements.ctx.strokeStyle = '#448AFF';
    elements.ctx.lineWidth = 2;
    elements.ctx.beginPath();
    elements.ctx.moveTo(origin.x, origin.y);
    elements.ctx.lineTo(zAxis.x, zAxis.y);
    elements.ctx.stroke();
}

// 导出函数
window.project = project;
window.unproject = unproject;
window.drawScene = drawScene;
window.updateRotationValues = updateRotationValues;
window.drawGrid = drawGrid;