// 创建图例
function createLegend() {
    const legend = d3.select('#legend');
    legend.html(''); // 清空现有内容
    
    for (let i = 0; i < 12; i++) {
        const item = legend.append('div')
            .attr('class', 'color-item')
            .style('cursor', 'pointer')
            .on('click', function() {
                if (state.selectedCategories.has(i)) {
                    state.selectedCategories.delete(i);
                    d3.select(this).style('opacity', 0.4);
                } else {
                    state.selectedCategories.add(i);
                    d3.select(this).style('opacity', 1);
                }
                updateSelectedCategoriesText();
                drawScene();
            });
        
        item.append('div')
            .attr('class', 'color-box')
            .style('background-color', CONFIG.COLORS[i]);
        
        item.append('span')
            .text(`类别 ${i}`);
    }
}

// 更新选中的类别文本
function updateSelectedCategoriesText() {
    if (state.selectedCategories.size === 12) {
        elements.selectedCategories.textContent = '显示类别: 全部';
    } else if (state.selectedCategories.size === 0) {
        elements.selectedCategories.textContent = '显示类别: 无';
    } else {
        elements.selectedCategories.textContent = `显示类别: ${Array.from(state.selectedCategories).join(', ')}`;
    }
}

// 设置画布尺寸
function setupCanvas() {
    const container = elements.canvas.parentElement;
    elements.canvas.width = container.clientWidth;
    elements.canvas.height = container.clientHeight;
}

// 初始化事件监听
function initEvents() {
    // 鼠标拖动旋转
    elements.canvas.addEventListener('mousedown', (e) => {
        const mouseX = e.clientX - elements.canvas.getBoundingClientRect().left;
        const mouseY = e.clientY - elements.canvas.getBoundingClientRect().top;
        
        // 检查是否点击了中心点（在编辑模式下）
        if (state.isEditing && state.showCenters) {
            const centerSize = parseInt(elements.centerSize.value);
            
            for (let i = 0; i < state.centerPoints.length; i++) {
                const center = state.centerPoints[i];
                const pos = project(center.x, center.y, center.z);
                const distance = Math.sqrt(Math.pow(mouseX - pos.x, 2) + Math.pow(mouseY - pos.y, 2));
                
                if (distance <= centerSize + 5) {
                    // 右键点击删除中心点
                    if (e.button === 2) {
                        // 从中心点数组中移除该中心点
                        state.centerPoints.splice(i, 1);
                        // 更新中心点计数
                        elements.centerCount.textContent = `中心点: ${state.centerPoints.length}`;
                        // 重新绘制场景
                        drawScene();
                        e.preventDefault();
                        return;
                    }
                    
                    // 左键点击选择或拖动中心点
                    if (e.ctrlKey) {
                        // 按住Ctrl键点击切换选中状态
                        center.selected = !center.selected;
                        drawScene(); // 重新绘制以显示选中状态
                    } else {
                        // 直接拖动中心点
                        state.isDraggingCenter = true;
                        state.draggedCenterIndex = i;
                        state.originalCenterPos = {x: center.x, y: center.y, z: center.z};
                        elements.canvas.style.cursor = 'grabbing';
                    }
                    e.preventDefault();
                    return;
                }
            }
        }
        
        // 如果不是在拖动中心点，则进行正常的视角操作
        state.isDragging = true;
        state.lastX = e.clientX;
        state.lastY = e.clientY;
        elements.canvas.style.cursor = 'grabbing';
        state.isRightClick = e.button === 2; // 只有鼠标右键才设置为true
        
        // 检测Ctrl键是否按下，用于Z轴旋转
        state.isCtrlPressed = e.ctrlKey;
        
        // 防止事件冒泡
        e.preventDefault();
    });
    
    elements.canvas.addEventListener('mousemove', (e) => {
        const mouseX = e.clientX - elements.canvas.getBoundingClientRect().left;
        const mouseY = e.clientY - elements.canvas.getBoundingClientRect().top;
        
        // 处理中心点拖动
        if (state.isDraggingCenter && state.draggedCenterIndex >= 0) {
            const center = state.centerPoints[state.draggedCenterIndex];
            const originalPos = project(state.originalCenterPos.x, state.originalCenterPos.y, state.originalCenterPos.z);
            
            // 计算鼠标移动距离
            const dx = mouseX - originalPos.x;
            const dy = mouseY - originalPos.y;
            
            // 转换为3D空间中的移动
            // 使用unproject函数计算3D空间中的新位置
            const depth = originalPos.z;
            const newPos = unproject(mouseX, mouseY, depth);
            
            // 更新中心点位置
            center.x = newPos.x;
            center.y = newPos.y;
            center.z = newPos.z;
            
            e.preventDefault();
            return;
        }
        
        if (!state.isDragging) return;
        
        const deltaX = e.clientX - state.lastX;
        const deltaY = e.clientY - state.lastY;
        
        if (state.isShiftPressed) {
            // 平移视图
            state.offsetX += deltaX;
            state.offsetY += deltaY;
        } else if (state.isRightClick || state.isCtrlPressed) {
            // 右键或Ctrl键拖动 - Z轴旋转
            state.rotationZ += deltaX * 0.01;
            drawScene(); // 重新绘制场景以反映Z轴旋转
        } else {
            // 左键拖动 - X和Y轴旋转
            state.rotationY += deltaX * 0.01;
            state.rotationX += deltaY * 0.01;
            
            // 限制X旋转角度在-π/2到π/2之间
            state.rotationX = Math.max(-Math.PI/2, Math.min(Math.PI/2, state.rotationX));
            drawScene(); // 重新绘制场景以反映X/Y轴旋转
        }
        
        state.lastX = e.clientX;
        state.lastY = e.clientY;
    });
    
    elements.canvas.addEventListener('mouseup', (e) => {
        if (state.isDraggingCenter) {
            state.isDraggingCenter = false;
            state.draggedCenterIndex = -1;
            state.originalCenterPos = null;
            elements.canvas.style.cursor = state.isEditing ? 'pointer' : 'grab';
            return;
        }
        
        state.isDragging = false;
        elements.canvas.style.cursor = state.isEditing ? 'pointer' : 'grab';
        if (e.button !== 2) { // Only reset isRightClick if not right click
            state.isRightClick = false;
        }
    });
    
    elements.canvas.addEventListener('mouseleave', (e) => {
        if (state.isDraggingCenter) {
            state.isDraggingCenter = false;
            state.draggedCenterIndex = -1;
            state.originalCenterPos = null;
        }
        
        state.isDragging = false;
        elements.canvas.style.cursor = state.isEditing ? 'pointer' : 'grab';
        // Only reset isRightClick if not right click
        if (e.button === undefined || e.button !== 2) {
            state.isRightClick = false;
        }
    });
    
    // 阻止右键菜单
    elements.canvas.addEventListener('contextmenu', (e) => {
        e.preventDefault();
    });
    
    // 键盘事件 - 检测Shift键和Ctrl键
    window.addEventListener('keydown', (e) => {
        if (e.key === 'Shift') {
            state.isShiftPressed = true;
            elements.canvas.style.cursor = 'move';
        } else if (e.key === 'Control') {
            state.isCtrlPressed = true;
        }
    });
    
    window.addEventListener('keyup', (e) => {
        if (e.key === 'Shift') {
            state.isShiftPressed = false;
            elements.canvas.style.cursor = state.isEditing ? 'pointer' : 'grab';
        } else if (e.key === 'Control') {
            state.isCtrlPressed = false;
        }
    });
    
    // 鼠标滚轮缩放
    elements.canvas.addEventListener('wheel', (e) => {
        e.preventDefault();
        state.scale += e.deltaY * -0.001;
        state.scale = Math.max(CONFIG.MIN_SCALE, Math.min(CONFIG.MAX_SCALE, state.scale));
        drawScene();
    });
    
    // 重置视图按钮
    document.getElementById('reset-view').addEventListener('click', () => {
        state.rotationX = 0;
        state.rotationY = 0;
        state.rotationZ = 0;
        state.scale = CONFIG.DEFAULT_SCALE;
        state.offsetX = 0;
        state.offsetY = 0;
        drawScene();
    });
    
    // 自动旋转切换
    document.getElementById('toggle-rotation').addEventListener('click', function() {
        state.autoRotate = !state.autoRotate;
        this.textContent = `自动旋转: ${state.autoRotate ? '开启' : '关闭'}`;
        elements.rotationStatus.textContent = `旋转: ${state.autoRotate ? '开启' : '关闭'}`;
        drawScene();
    });
    
    // 显示中心点切换
    document.getElementById('toggle-centers').addEventListener('click', function() {
        state.showCenters = !state.showCenters;
        this.textContent = `显示中心点: ${state.showCenters ? '开启' : '关闭'}`;
        drawScene();
    });
    
    // 编辑模式切换
    document.getElementById('toggle-edit-mode').addEventListener('click', function() {
        state.isEditing = !state.isEditing;
        this.textContent = `编辑模式: ${state.isEditing ? '开启' : '关闭'}`;
        elements.editStatus.textContent = `编辑: ${state.isEditing ? '开启' : '关闭'}`;
        document.getElementById('edit-info').style.display = state.isEditing ? 'block' : 'none';
        elements.canvas.style.cursor = state.isEditing ? 'pointer' : 'grab';
    });
    
    // 保存中心点按钮
    document.getElementById('save-centers').addEventListener('click', saveCenterPoints);
    
    // 初始化类别选择下拉菜单
    function initCategorySelector() {
        const categorySelect = document.getElementById('category-select');
        
        // 添加类别选项
        for (let i = 0; i < 12; i++) {
            const option = document.createElement('option');
            option.value = i;
            option.textContent = `${i}: ${CONFIG.CATEGORY_NAMES[i]}`;
            categorySelect.appendChild(option);
        }
    }
    
    // 显示中心点坐标按钮
    document.getElementById('show-centers-coordinates').addEventListener('click', function() {
        const categorySelect = document.getElementById('category-select');
        const selectedCategory = categorySelect.value;
        
        // 创建坐标信息字符串
        let coordinatesInfo = '';
        
        if (selectedCategory === 'all') {
            coordinatesInfo = '全部中心点3D坐标:\n';
            state.centerPoints.forEach((center, index) => {
                coordinatesInfo += `类别 ${center.category} 中心点 ${index + 1}: (${center.x.toFixed(2)}, ${center.y.toFixed(2)}, ${center.z.toFixed(2)})\n`;
            });
        } else {
            const categoryId = parseInt(selectedCategory);
            const categoryName = CONFIG.CATEGORY_NAMES[categoryId];
            coordinatesInfo = `类别 ${categoryId} (${categoryName}) 中心点3D坐标:\n`;
            
            // 筛选出指定类别的中心点
            const filteredCenters = state.centerPoints.filter(center => center.category === categoryId);
            
            if (filteredCenters.length === 0) {
                coordinatesInfo += '该类别没有中心点。\n';
            } else {
                filteredCenters.forEach((center, index) => {
                    coordinatesInfo += `中心点 ${index + 1}: (${center.x.toFixed(2)}, ${center.y.toFixed(2)}, ${center.z.toFixed(2)})\n`;
                });
            }
        }
        
        // 显示坐标信息
        alert(coordinatesInfo);
    });
    
    // 删除中心点按钮
    document.getElementById('delete-center-point').addEventListener('click', function() {
        alert('要删除中心点，请在编辑模式下右键点击要删除的中心点。');
    });
    
    // 删除未选中点按钮
    document.getElementById('delete-unselected-points').addEventListener('click', function() {
        if (!state.isEditing) {
            alert('请先启用编辑模式。');
            return;
        }
        
        // 确认删除操作
        if (confirm('确定要删除所有未选中的点吗？此操作不可撤销。')) {
            // 保留选中的点，删除未选中的点
            state.centerPoints = state.centerPoints.filter(point => point.selected);
            
            // 更新中心点计数
            elements.centerCount.textContent = `中心点: ${state.centerPoints.length}`;
            
            // 重新绘制场景
            drawScene();
            
            alert(`已删除所有未选中的点，剩余 ${state.centerPoints.length} 个点。`);
        }
    });
    
    // 分割选中点按钮
    document.getElementById('split-point').addEventListener('click', function() {
        if (!state.isEditing) {
            alert('请先启用编辑模式。');
            return;
        }
        
        // 找到选中的点
        const selectedPoints = state.centerPoints.filter(point => point.selected);
        
        if (selectedPoints.length === 0) {
            alert('请先选择一个点。');
            return;
        }
        
        if (selectedPoints.length > 1) {
            alert('只能同时分割一个点，请只选择一个点。');
            return;
        }
        
        // 获取选中的点
        const selectedPoint = selectedPoints[0];
        const selectedIndex = state.centerPoints.findIndex(point => point.selected);
        
        // 确认分割操作
        if (confirm(`确定要将类别为"${CONFIG.CATEGORY_NAMES[selectedPoint.category]}"的选中点分割为两个点吗？`)) {
            // 创建两个新点，位置略有偏移
            const offset = 0.5; // 偏移量
            const newPoint1 = {
                x: selectedPoint.x - offset,
                y: selectedPoint.y,
                z: selectedPoint.z,
                category: selectedPoint.category,
                selected: true // 保持一个点选中，以便用户看到分割效果
            };
            
            const newPoint2 = {
                x: selectedPoint.x + offset,
                y: selectedPoint.y,
                z: selectedPoint.z,
                category: selectedPoint.category,
                selected: false
            };
            
            // 用两个新点替换原来的点
            state.centerPoints.splice(selectedIndex, 1, newPoint1, newPoint2);
            
            // 更新中心点计数
            elements.centerCount.textContent = `中心点: ${state.centerPoints.length}`;
            
            // 重新绘制场景
            drawScene();
            
            alert(`已将选中点分割为两个点（其中一个保持选中状态），现在共有 ${state.centerPoints.length} 个点。`);
        }
    });
    

    
    // 体素大小滑块
    elements.voxelSize.addEventListener('input', function() {
        elements.voxelSizeValue.textContent = this.value;
        drawScene();
    });
    
    // 透明度滑块
    elements.opacity.addEventListener('input', function() {
        elements.opacityValue.textContent = `${this.value}%`;
        drawScene();
    });
    
    // 中心点大小滑块
    elements.centerSize.addEventListener('input', function() {
        elements.centerSizeValue.textContent = this.value;
        drawScene();
    });
    
    // 数据源切换
    document.getElementById('load-demo').addEventListener('click', function() {
        document.getElementById('load-demo').classList.add('active-source');
        document.getElementById('load-file').classList.remove('active-source');
        document.getElementById('upload-area').style.display = 'none';
        
        elements.loading.style.display = 'block';
        elements.loading.textContent = '正在生成示例数据...';
        
        setTimeout(() => {
            state.voxelData = generateDemoData();
            state.centerPointsFromFile = generateDemoCenters();
            
            if (state.useFileCenters) {
                state.centerPoints = state.centerPointsFromFile;
                elements.centerSource.textContent = '中心点来源: 文件';
            } else {
                calculateCenters();
                elements.centerSource.textContent = '中心点来源: 自动计算';
            }
            
            elements.loading.style.display = 'none';
            drawScene();
        }, 500);
    });
    
    document.getElementById('load-file').addEventListener('click', function() {
        document.getElementById('load-demo').classList.remove('active-source');
        document.getElementById('load-file').classList.add('active-source');
        document.getElementById('upload-area').style.display = 'block';
    });
    
    // 初始化类别选择下拉菜单
    initCategorySelector();
    
    // 文件上传处理
    document.getElementById('upload-area').addEventListener('click', function() {
        elements.fileInput.click();
    });
    
    elements.fileInput.addEventListener('change', function(e) {
        if (e.target.files.length > 0) {
            // 根据用户选择的文件类型处理文件
            const fileType = elements.fileType.value;
            if (fileType === 'centers') {
                handleJsonFile(e.target.files[0], true);
            } else {
                handleJsonFile(e.target.files[0], false);
            }
        }
    });
    
    // 中心点数据源切换
    document.getElementById('calc-centers').addEventListener('click', function() {
        toggleCenterSource(false);
    });
    
    document.getElementById('load-points-file').addEventListener('click', function() {
        toggleCenterSource(true);
    });
    
    // 中心点文件上传处理
    document.getElementById('upload-points-area').addEventListener('click', function() {
        elements.pointsFileInput.click();
    });
    
    elements.pointsFileInput.addEventListener('change', function(e) {
        if (e.target.files.length > 0) {
            handleJsonFile(e.target.files[0], true);
        }
    });
}

// 动画循环
function animate() {
    if (state.autoRotate) {
        state.rotationY += 0.005;
        state.rotationX += 0.002;
        state.rotationZ += 0.003;
    }
    
    drawScene();
    state.animationId = requestAnimationFrame(animate);
}

// 导出函数
window.createLegend = createLegend;
window.updateSelectedCategoriesText = updateSelectedCategoriesText;
window.setupCanvas = setupCanvas;
window.initEvents = initEvents;
window.animate = animate;