import torch
from torch import nn

import utils


class DoubleQCritic(nn.Module):
    """Critic network, employes double Q-learning."""
    def __init__(self, obs_dim, t_obs_dim, action_dim,
                 hidden_dim, hidden_depth, use_t_obs=False,
                 use_t_vel=False):
        super().__init__()

        self.use_t_obs = use_t_obs
        self._use_t_vel = use_t_vel
        o_dim = obs_dim
        if use_t_obs:
            o_dim += t_obs_dim
        if use_t_vel:
            o_dim += t_obs_dim

        self.Q1 = utils.mlp(o_dim + action_dim,
                            hidden_dim, 1, hidden_depth)
        self.Q2 = utils.mlp(o_dim + action_dim,
                            hidden_dim, 1, hidden_depth)

        self.outputs = dict()
        self.apply(utils.weight_init)

    def forward(self, obs, t_obs, prev_obs, prev_t_obs, action):
        assert obs.size(0) == action.size(0)
        assert obs.size(0) == t_obs.size(0)
        assert obs.size(0) == prev_obs.size(0)
        assert obs.size(0) == prev_t_obs.size(0)

        to_cat = [obs, action]
        if self.use_t_obs:
            to_cat.append(t_obs)
        if self._use_t_vel:
            t_vel = t_obs - prev_t_obs
            to_cat.append(t_vel)
        obs_action = torch.cat(to_cat, dim=-1)
        q1 = self.Q1(obs_action)
        q2 = self.Q2(obs_action)

        self.outputs['q1'] = q1
        self.outputs['q2'] = q2

        return q1, q2

    def log(self, logger, step):
        for k, v in self.outputs.items():
            logger.log_histogram(f'train_critic/{k}_hist', v, step)

        assert len(self.Q1) == len(self.Q2)
        for i, (m1, m2) in enumerate(zip(self.Q1, self.Q2)):
            assert type(m1) == type(m2)
            if type(m1) is nn.Linear:
                logger.log_param(f'train_critic/q1_fc{i}', m1, step)
                logger.log_param(f'train_critic/q2_fc{i}', m2, step)
