import torch
from sklearn.metrics import roc_auc_score, classification_report

def evaluate_model(model, data_loader, device):
    model.eval()
    all_outputs = []
    all_labels = []
    with torch.no_grad():
        for inputs, times, labels in data_loader:
            inputs, times, labels = inputs.to(device), times.to(device), labels.to(device)
            outputs = model(inputs, times).squeeze()
            all_outputs.append(outputs)
            all_labels.append(labels)
    all_outputs = torch.cat(all_outputs).cpu()
    all_labels = torch.cat(all_labels).cpu()
    return all_outputs, all_labels

def compute_metrics(outputs, labels):
    outputs = torch.sigmoid(outputs).numpy()
    labels = labels.numpy()
    auc = roc_auc_score(labels, outputs)
    preds = (outputs > 0.5).astype(int)
    report = classification_report(labels, preds, output_dict=True)
    metrics = {
        'auc': auc,
        'accuracy': report['accuracy'],
        'precision': report['1']['precision'],
        'recall': report['1']['recall'],
        'f1': report['1']['f1-score']
    }
    return metrics

def print_classification_report(labels, preds):
    print(classification_report(labels, preds, digits=4))
