import torch
import numpy as np
from sklearn.metrics import accuracy_score, precision_score, recall_score, f1_score, roc_auc_score, classification_report

def evaluate_model(model, data_loader, device):
    model.eval()
    outputs = []
    labels = []
    with torch.no_grad():
        for inputs, targets, times in data_loader:
            inputs, targets, times = inputs.to(device), targets.to(device), times.to(device)
            output, _, _ = model(inputs, times)
            outputs.append(output)
            labels.append(targets)
    outputs = torch.cat(outputs).cpu().numpy()
    labels = torch.cat(labels).cpu().numpy()
    return outputs, labels

def compute_metrics(outputs, labels):
    predictions = (outputs > 0.5).astype(int)
    accuracy = accuracy_score(labels, predictions)
    precision = precision_score(labels, predictions)
    recall = recall_score(labels, predictions)
    f1 = f1_score(labels, predictions)
    auc = roc_auc_score(labels, outputs)
    loss = np.mean(labels * np.log(outputs) + (1 - labels) * np.log(1 - outputs))
    return {
        'accuracy': accuracy,
        'precision': precision,
        'recall': recall,
        'f1': f1,
        'auc': auc,
        'loss': loss
    }

def print_classification_report(labels, predictions):
    report = classification_report(labels, predictions, digits=4)
    print(report)
