import torch
from sklearn.metrics import roc_auc_score, classification_report, accuracy_score, precision_score, recall_score

def evaluate_model(model, dataloader, device):
    model.eval()
    outputs = []
    labels = []
    with torch.no_grad():
        for X_batch, y_batch in dataloader:
            X_batch, y_batch = X_batch.to(device), y_batch.to(device)
            tgt_batch = torch.zeros_like(X_batch).to(device)
            padding_mask = (X_batch == 0).all(dim=-1)

            output = model(X_batch, tgt_batch, src_mask=padding_mask)
            outputs.append(output)
            labels.append(y_batch)

    outputs = torch.cat(outputs).view(-1).cpu()
    labels = torch.cat(labels).view(-1).cpu()
    return outputs, labels

def compute_metrics(outputs, labels):
    outputs = outputs > 0.5
    accuracy = accuracy_score(labels, outputs)
    precision = precision_score(labels, outputs)
    recall = recall_score(labels, outputs)
    auc = roc_auc_score(labels, outputs)
    report = classification_report(labels, outputs, digits=4)
    return {
        'accuracy': accuracy,
        'precision': precision,
        'recall': recall,
        'auc': auc,
        'report': report
    }

def print_classification_report(metrics):
    print(metrics['report'])
    print(f"Accuracy: {metrics['accuracy']:.4f}")
    print(f"Precision: {metrics['precision']:.4f}")
    print(f"Recall: {metrics['recall']:.4f}")
    print(f"AUC: {metrics['auc']:.4f}")
