"""
Interfaces to pygraphviz to draw time series of graphs using dot.

"""

import numpy as np
import pygraphviz as pgv

# REVIEWER COPY; DO NOT DISTRIBUTE!

def draw_from_adjacency_matrix(A, filename, labels = None):
    """ Converts an adjacency matrix to pygraphviz graph format, draws it
    using dot and writes the result to the given filename. The given file
    extension determines the file format.

    Parameters
    ----------
    A: class numpy.array
        An adjacency matrix
    filename: str
        A filename for the output file.
    labels: list (default = list(range(len(A))))
        a list-like object of labels for each node.

    Returns
    -------
    G: class pygraphviz.AGraph
        The input graph in pygraphviz format.

    """
    directed = not np.all(A == A.T)
    G = pgv.AGraph(directed = directed)
    if(labels is None):
        labels = list(range(len(A)))
    G.add_nodes_from(labels)
    for i in range(len(A)):
        lo = i+1 if not directed else 0
        for j in np.where(A[i, lo:])[0]:
            G.add_edge(labels[i], labels[lo + j])
    G.draw(filename, prog = 'dot')
    return G

def draw_time_series(As, filename_pattern, labelss = None):
    """ Converts all input adjacency matrices to pygraphviz graph format,
    draws them using dot and writes the result to files with the given
    filename pattern. The given file extension determines the file format.

    Parameters
    ----------
    As: list
        A list of adjacency matrices.
    filename: str
        A filename pattern for the output files. Has to contain one
        %d for the time step.
    labelss: list (default = None)
        A list of label-lists for each graph in the time series.

    Returns
    -------
    Gs: list
        The input graphs in pygraphviz format.

    """
    Gs = []
    if(labelss is None):
        labelss = [None] * len(As)
    for t in range(len(As)):
        Gs.append(draw_from_adjacency_matrix(As[t], filename_pattern % t, labelss[t]))
    return Gs
