import random
from abc import abstractclassmethod
from copy import deepcopy
from typing import Dict, Union

import gin
import numpy as np
import scipy
import torch
import torch.nn.functional as F

import causal_discovery.logger as lg
from causal_discovery import enco, graph_fitting, new_approach
from causal_discovery.datasets import DynamicInterventionalDataset
from causal_discovery.sampling import (
    logpdf_interventionalSamples,
    sample_DAG,
    sample_interventionalSamples,
)


def logmeanexp(A, axis):
    return torch.logsumexp(A, dim=axis) - np.log(A.shape[axis])


LOGPDF_IDX = 1


class AcquisitionStrategy:
    @torch.no_grad()
    def acquire(self, gamma, theta, nodes):
        target, extra = self.score_for_value(gamma, theta, nodes)
        max_j_idx = int(torch.argmax(target, dim=None))
        max_j = nodes[max_j_idx]
        return max_j

    @torch.no_grad()
    def score_for_value(self, gamma, theta, nodes):
        raise NotImplementedError()

    @abstractclassmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "AcquisitionStrategy":
        pass


@gin.configurable
class BALDStrategy(AcquisitionStrategy):
    def __init__(
        self,
        model,
        num_categs: int,
        num_graphs: int,
        only_gamma: bool,
        num_int_samples: int,
    ):
        super(BALDStrategy, self).__init__()
        self.model = model
        self.num_categs = num_categs
        self.num_graphs = num_graphs
        self.num_int_samples = num_int_samples
        self.only_gamma = only_gamma

    @torch.no_grad()
    def score_for_value(self, gamma, theta, nodes):
        # Interventions x num_graphs x num_graphs_comp x samples
        logpdfs_val = self.get_logpdfs(gamma, theta, nodes)

        total_uncertainty = -logmeanexp(logpdfs_val, axis=2).mean((1, 2))
        aleatoric_uncertainty = -torch.diagonal(logpdfs_val, dim1=1, dim2=2).mean(
            (1, 2)
        )

        MI = total_uncertainty - aleatoric_uncertainty

        return MI, {}

    @torch.no_grad()
    def get_logpdfs(self, gamma, theta, nodes):
        sample_dag_gen = sample_DAG(
            gamma=gamma, theta=theta, only_gamma=self.only_gamma
        )
        graphs = [next(sample_dag_gen) for _ in range(self.num_graphs)]

        # Graphs x Number of Interventions x Samples x Num Nodes
        datapoints = torch.stack(
            [
                torch.stack(
                    [
                        sample_interventionalSamples(
                            config=g,
                            target_node=n,
                            model=self.model,
                            device=self.model.device,
                            nb_categs=self.num_categs,
                            batch_size=self.num_int_samples,
                        )
                        for n in nodes
                    ]
                )
                for g in graphs
            ]
        )

        logpdfs_vals = torch.stack(
            [
                torch.stack(
                    [
                        logpdf_interventionalSamples(
                            config=g1,
                            target_node=n,
                            model=self.model,
                            device=self.model.device,
                            samples=datapoints[:, n, :],
                        )
                        for g1 in graphs
                    ],
                    dim=1,
                )
                for n in nodes
            ]
        )

        return logpdfs_vals

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "BALDStrategy":
        num_categs = max([v.prob_dist.num_categs for v in gf_obj.graph.variables])
        return cls(
            model=gf_obj.model,
            num_categs=num_categs,
            num_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            num_int_samples=gf_obj.BALD_num_int_samples,
        )


@gin.configurable
class BatchBALDStrategy(BALDStrategy):
    def __init__(
        self,
        model,
        num_categs: int,
        num_graphs: int,
        only_gamma: bool,
        num_int_samples: int,
        batch_size: int,
    ):
        super(BatchBALDStrategy, self).__init__(
            model,
            num_categs,
            num_graphs,
            only_gamma=only_gamma,
            num_int_samples=num_int_samples,
        )
        self.batch_size = batch_size
        self.iter = 0
        self.targets = []

    @torch.no_grad()
    def score_for_value(self, gamma, theta, nodes, pnm1=None):
        logpdfs_val = self.get_logpdfs(gamma, theta, nodes)
        if pnm1 is not None:
            # the recursive equation (13) in batchbald paper
            logpdfs_val += pnm1[None, ...]

        total_uncertainty = -logmeanexp(logpdfs_val, axis=2).mean((1, 2))
        aleatoric_uncertainty = -torch.diagonal(logpdfs_val, dim1=1, dim2=2).mean(
            (1, 2)
        )

        MI = total_uncertainty - aleatoric_uncertainty

        return MI, {"logpdfs": logpdfs_val}

    @torch.no_grad()
    def acquire(self, gamma, theta, nodes):
        if self.iter == 0:
            self.targets = []
            pnm1 = None
            for _ in range(self.batch_size):
                temp_target, temp_extra = self.score_for_value(
                    gamma, theta, nodes, pnm1=pnm1
                )
                max_j_idx = int(torch.argmax(temp_target, dim=None))
                max_j = nodes[max_j_idx]
                pnm1 = temp_extra["logpdfs"][max_j]
                self.targets.append(max_j)
        max_j = self.targets[self.iter]
        self.iter = (self.iter + 1) % self.batch_size
        return max_j

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "BatchBALDStrategy":
        num_categs = max([v.prob_dist.num_categs for v in gf_obj.graph.variables])
        return cls(
            model=gf_obj.model,
            num_categs=num_categs,
            num_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            num_int_samples=gf_obj.BALD_num_int_samples,
            batch_size=gf_obj.BALD_batch_size,
        )


@gin.configurable
class SoftBALDStrategy(BALDStrategy):
    def __init__(
        self,
        model,
        num_categs: int,
        num_graphs: int,
        only_gamma: bool,
        bald_temperature: float,
        num_int_samples: int,
    ):
        super(SoftBALDStrategy, self).__init__(
            model, num_categs, num_graphs, only_gamma, num_int_samples=num_int_samples
        )
        self.temperature = bald_temperature

    @torch.no_grad()
    def acquire(self, gamma, theta, nodes):
        target, extra = self.score_for_value(gamma, theta, nodes)
        probs = F.softmax(target / self.temperature, dim=-1)
        max_j = int(torch.multinomial(probs, 1, replacement=True))
        return max_j

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "SoftBALDStrategy":
        num_categs = max([v.prob_dist.num_categs for v in gf_obj.graph.variables])
        return cls(
            model=gf_obj.model,
            num_categs=num_categs,
            num_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            num_int_samples=gf_obj.BALD_num_int_samples,
            bald_temperature=gf_obj.policy_softmax_temperature,
        )


@gin.configurable
class RoundRobinStrategy(AcquisitionStrategy):
    def __init__(self):
        super(RoundRobinStrategy, self).__init__()
        self.inter_vars = []

    def acquire(self, gamma, theta, possible_interventions):
        if len(self.inter_vars) == 0:  # If an epoch finished, reshuffle variables
            self.inter_vars = deepcopy(possible_interventions)
            random.shuffle(self.inter_vars)
        var_idx = self.inter_vars.pop()
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "RoundRobinStrategy":
        return cls()


@gin.configurable
class NonemptyRoundRobinStrategy(RoundRobinStrategy):
    def __init__(self, dataset: DynamicInterventionalDataset):
        super(NonemptyRoundRobinStrategy, self).__init__()
        if not isinstance(dataset, DynamicInterventionalDataset):
            raise ValueError(
                f"{self.__class__.__name__} makes sense only when "
                "DynamicInterventionalDataset is used"
            )
        self.dataset = dataset

    def acquire(self, gamma, theta, possible_interventions):
        while True:
            var_idx = super().acquire(gamma, theta, possible_interventions)
            if self.dataset.size(var_idx) > 0:
                break
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "RoundRobinStrategy":
        return cls(dataset=gf_obj.dataset)


@gin.configurable
class UniformStrategy(AcquisitionStrategy):
    def __init__(self):
        super(UniformStrategy, self).__init__()
        self.inter_vars = []

    def acquire(self, gamma, theta, possible_interventions):
        return random.choice(possible_interventions)

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "RoundRobinStrategy":
        return cls()


@gin.configurable
class CeSHDReductionStrategy(AcquisitionStrategy):
    def __init__(
        self,
        parent_cd_object: Union["enco.ENCO", "new_approach.NewApproach"],
        dataset_to_save: dict,
    ):
        super(CeSHDReductionStrategy, self).__init__()
        self.parent_cd_object = parent_cd_object
        self.dataset_to_save = dataset_to_save

    @torch.no_grad()
    def acquire(self, gamma, theta, possible_interventions):
        candidates = []
        saved_state = self.parent_cd_object.save_state_and_optimizers()
        metrics_before = self.parent_cd_object.get_metrics()

        gradient_magnitude = []
        shd_diff = []
        ce_shd_diff = []

        for idx in possible_interventions:
            metrics = self.parent_cd_object.graph_fitting_single_iteration(
                var_idx=idx, suppress_logging=True
            )
            self.parent_cd_object.load_state_and_optimizers(*saved_state)
            score = metrics["ce_shd"]
            candidates.append((score, idx))

            # needed for dataset to save
            gradient_magnitude.append(metrics["gradient_magnitude"])
            shd_diff.append(metrics_before["SHD"] - metrics["SHD"])
            ce_shd_diff.append(metrics_before["ce_shd"] - metrics["ce_shd"])

        if self.dataset_to_save is not None:
            self.dataset_to_save["gamma"].append(gamma.cpu().numpy())
            self.dataset_to_save["theta"].append(theta.cpu().numpy())
            self.dataset_to_save["gradient_magnitude"].append(gradient_magnitude)
            self.dataset_to_save["shd_diff"].append(shd_diff)
            self.dataset_to_save["ce_shd_diff"].append(ce_shd_diff)

        candidates.sort()

        score, var_idx = candidates[0]
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "CeSHDReductionStrategy":
        return cls(
            parent_cd_object=gf_obj.parent_enco_object,
            dataset_to_save=gf_obj.dataset_to_save,
        )


@gin.configurable
class AITStrategy(AcquisitionStrategy):
    def __init__(
        self,
        parent_gf_object: "graph_fitting.GraphFitting",
        num_hypothetical_graphs: int,
        only_gamma: bool,
        log_grads: bool,
    ):
        super(AITStrategy, self).__init__()
        self.parent_gf_object = parent_gf_object
        self.num_hypothetical_graphs = num_hypothetical_graphs
        self.only_gamma = only_gamma
        self.log_grads = log_grads

    @torch.no_grad()
    def score_for_value(self, gamma, theta, possible_interventions):
        scores = []

        # (0) Sample hypothesis DAGs
        sample_dag_gen = sample_DAG(
            gamma=gamma, theta=theta, only_gamma=self.only_gamma
        )
        graphs = [next(sample_dag_gen) for _ in range(self.num_hypothetical_graphs)]

        # (1) Sample data on  hypo. DAGs and compute AIT score
        for idx in possible_interventions:

            mask = np.ones(
                (len(self.parent_gf_object.graph.variables),), dtype=np.float32
            )
            mask[idx] = 0
            mask = torch.from_numpy(mask).to(self.parent_gf_object.get_device())

            samples_overall = []
            means_graph = []
            variances_graph = []

            # Collect hypothetical interventional samples + compute mean/variance per graph
            for g in graphs:
                int_samples = self.parent_gf_object.get_hypothetical_samples(
                    var_idx=idx, hypothetical_graph=g
                ).float()
                samples_overall.append(int_samples)
                means_graph.append(int_samples.mean(dim=0))
                variances_graph.append((int_samples.var(dim=0) * mask).sum())

            # Compute AIT score
            samples_overall = torch.stack(samples_overall).reshape(
                -1, len(self.parent_gf_object.graph.variables)
            )
            mean_overall = samples_overall.mean(dim=0)
            vbg = torch.stack(means_graph).sub(mean_overall).square().sum(dim=0)
            vbg = (vbg.div(self.num_hypothetical_graphs - 1) * mask).sum()
            vwg = torch.stack(variances_graph).sum()
            score = vbg / vwg
            scores.append(score.cpu())

        return scores, {}

    @torch.no_grad()
    def acquire(self, gamma, theta, nodes):
        scores, _ = self.score_for_value(gamma, theta, nodes)
        max_j_idx = int(np.argmax(scores))
        var_idx, score = nodes[max_j_idx], scores[max_j_idx]

        if self.log_grads:
            for candidate_score, candidate_idx in zip(scores, nodes):
                lg.NEPTUNE_LOGGER.log(
                    name=f"ait_score_{candidate_idx}",
                    value=candidate_score,
                )
            lg.NEPTUNE_LOGGER.log(
                name="ait_score",
                value=score,
            )

        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "AITStrategy":
        return cls(
            parent_gf_object=gf_obj,
            num_hypothetical_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            log_grads=gf_obj.log_grads,
        )


@gin.configurable
class SoftAITStrategy(AITStrategy):
    def __init__(self, policy_softmax_temperature: float = 1.0, **kwargs):
        super(SoftAITStrategy, self).__init__(**kwargs)
        self.policy_softmax_temperature = policy_softmax_temperature

    def acquire(self, gamma, theta, nodes):
        scores, _ = self.score_for_value(gamma, theta, nodes)
        scores = np.array(scores)

        normalized_scores = (scores - scores.min()) / (scores.max() - scores.min())
        probs = scipy.special.softmax(
            normalized_scores / self.policy_softmax_temperature
        )
        idx = random.choices(np.arange(len(nodes)), weights=probs, k=1)[0]
        var_idx, score = nodes[idx], scores[idx]

        if self.log_grads:
            for candidate_score, candidate_idx in zip(scores, nodes):
                lg.NEPTUNE_LOGGER.log(
                    name=f"ait_score_{candidate_idx}",
                    value=candidate_score,
                )
            lg.NEPTUNE_LOGGER.log(
                name="ait_score",
                value=score,
            )

        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "SoftAITStrategy":
        return cls(
            parent_gf_object=gf_obj,
            num_hypothetical_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            log_grads=gf_obj.log_grads,
            policy_softmax_temperature=gf_obj.policy_softmax_temperature,
        )


@gin.configurable
class TrainedStrategy(AcquisitionStrategy):
    def __init__(self, trained_policy, features_fn):
        super(TrainedStrategy, self).__init__()
        self.trained_policy = trained_policy
        self.features_fn = features_fn

    def score_for_value(self, gamma, theta, nodes):
        with torch.no_grad():
            inputs = self.features_fn(
                gamma=gamma.cpu().numpy()[None, ...],
                theta=theta.cpu().numpy()[None, ...],
            )
            scores = self.trained_policy(torch.from_numpy(inputs))[0]
        return scores, {}

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "TrainedStrategy":
        return cls(trained_policy=gf_obj.trained_policy, features_fn=gf_obj.features_fn)


@gin.configurable
class SoftTrainedStrategy(TrainedStrategy):
    def __init__(self, policy_softmax_temperature: float = 1.0, **kwargs):
        super(SoftTrainedStrategy, self).__init__(**kwargs)
        self.policy_softmax_temperature = policy_softmax_temperature

    def acquire(self, gamma, theta, nodes):
        scores, _ = self.score_for_value(gamma, theta, nodes)
        scores = (
            torch.nn.functional.softmax(scores / self.policy_softmax_temperature)
            .cpu()
            .numpy()
        )
        candidates = [(float(score), i) for (i, score) in zip(nodes, scores)]
        score, var_idx = random.choices(candidates, weights=scores, k=1)[0]
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "SoftTrainedStrategy":
        return cls(
            trained_policy=gf_obj.trained_policy,
            features_fn=gf_obj.features_fn,
            policy_softmax_temperature=gf_obj.policy_softmax_temperature,
        )


@gin.configurable
class GradientsL2Strategy(AcquisitionStrategy):
    def __init__(self, parent_gf_object: "graph_fitting.GraphFitting", log_grads: bool):
        super(GradientsL2Strategy, self).__init__()
        self.parent_gf_object = parent_gf_object
        self.log_grads = log_grads

    def score_for_value(self, gamma, theta, possible_interventions):
        scores = []
        for idx in possible_interventions:
            score = float(
                self.parent_gf_object.get_hypothetical_gradient_magnitude(
                    gamma, theta, idx
                ).cpu()
            )
            scores.append(score)
        return scores, {}

    def acquire(self, gamma, theta, nodes):
        scores, _ = self.score_for_value(gamma, theta, nodes)
        max_j_idx = int(np.argmax(scores))
        var_idx, score = nodes[max_j_idx], scores[max_j_idx]

        if self.log_grads:
            for candidate_score, candidate_idx in zip(scores, nodes):
                lg.NEPTUNE_LOGGER.log(
                    name=f"grad_l2_hypo_{candidate_idx}",
                    value=candidate_score,
                )
            lg.NEPTUNE_LOGGER.log(
                name="grad_l2_hypo",
                value=score,
            )
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "GradientsL2Strategy":
        return cls(parent_gf_object=gf_obj, log_grads=gf_obj.log_grads)


@gin.configurable
class SoftGradientsL2Strategy(GradientsL2Strategy):
    def __init__(self, policy_softmax_temperature: float = 1.0, **kwargs):
        super(SoftGradientsL2Strategy, self).__init__(**kwargs)
        self.policy_softmax_temperature = policy_softmax_temperature

    def acquire(self, gamma, theta, nodes):
        scores, _ = self.score_for_value(gamma, theta, nodes)
        scores = np.array(scores)
        normalized_scores = (scores - scores.min()) / (scores.max() - scores.min())
        probs = scipy.special.softmax(
            normalized_scores / self.policy_softmax_temperature
        )
        candidates = [(float(score), i) for (i, score) in zip(nodes, scores)]
        score, var_idx = random.choices(candidates, weights=probs, k=1)[0]

        if self.log_grads:
            for candidate_score, candidate_idx in zip(scores, nodes):
                lg.NEPTUNE_LOGGER.log(
                    name=f"grad_l2_hypo_{candidate_idx}",
                    value=candidate_score,
                )
            lg.NEPTUNE_LOGGER.log(
                name="grad_l2_hypo",
                value=score,
            )
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "SoftGradientsL2Strategy":
        return cls(
            parent_gf_object=gf_obj,
            log_grads=gf_obj.log_grads,
            policy_softmax_temperature=gf_obj.policy_softmax_temperature,
        )


@gin.configurable
class GradientsL2HypotheticalSamplesStrategy(GradientsL2Strategy):
    def __init__(self, num_hypothetical_graphs: int, only_gamma: bool, **kwargs):
        super(GradientsL2HypotheticalSamplesStrategy, self).__init__(**kwargs)
        self.num_hypothetical_graphs = num_hypothetical_graphs
        self.only_gamma = only_gamma

    def score_for_value(self, gamma, theta, possible_interventions):
        scores = []
        sample_dag_gen = sample_DAG(
            gamma=gamma, theta=theta, only_gamma=self.only_gamma
        )
        graphs = [next(sample_dag_gen) for _ in range(self.num_hypothetical_graphs)]

        for idx in possible_interventions:
            score = 0
            for g in graphs:
                score += float(
                    self.parent_gf_object.get_hypothetical_gradient_magnitude(
                        gamma, theta, idx, hypothetical_graph=g
                    ).cpu()
                )
            score /= self.num_hypothetical_graphs
            scores.append(score)
        return scores, {}

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "GradientsL2HypotheticalSamplesStrategy":
        return cls(
            num_hypothetical_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            parent_gf_object=gf_obj,
            log_grads=gf_obj.log_grads,
        )


@gin.configurable
class SoftGradientsL2HypotheticalSamplesStrategy(
    GradientsL2HypotheticalSamplesStrategy
):
    def __init__(self, policy_softmax_temperature: float = 1.0, **kwargs):
        super(SoftGradientsL2HypotheticalSamplesStrategy, self).__init__(**kwargs)
        self.policy_softmax_temperature = policy_softmax_temperature

    def acquire(self, gamma, theta, nodes):
        scores, _ = self.score_for_value(gamma, theta, nodes)
        scores = np.array(scores)
        normalized_scores = (scores - scores.min()) / (scores.max() - scores.min())
        probs = scipy.special.softmax(
            normalized_scores / self.policy_softmax_temperature
        )
        candidates = [(float(score), i) for (i, score) in zip(nodes, scores)]
        score, var_idx = random.choices(candidates, weights=probs, k=1)[0]

        if self.log_grads:
            for candidate_score, candidate_idx in zip(scores, nodes):
                lg.NEPTUNE_LOGGER.log(
                    name=f"grad_l2_hypo_{candidate_idx}",
                    value=candidate_score,
                )
            lg.NEPTUNE_LOGGER.log(
                name="grad_l2_hypo",
                value=score,
            )
        return var_idx

    @classmethod
    def from_graph_fitting_object(
        cls, gf_obj: "graph_fitting.GraphFitting"
    ) -> "SoftGradientsL2HypotheticalSamplesStrategy":
        return cls(
            num_hypothetical_graphs=gf_obj.num_hypothetical_graphs,
            only_gamma=gf_obj.only_gamma,
            parent_gf_object=gf_obj,
            log_grads=gf_obj.log_grads,
            policy_softmax_temperature=gf_obj.policy_softmax_temperature,
        )


strategies: Dict[str, AcquisitionStrategy] = {
    "bald": BALDStrategy,
    "batch_bald": BatchBALDStrategy,
    "soft_bald": SoftBALDStrategy,
    "round_robin": RoundRobinStrategy,
    "nonempty_round_robin": NonemptyRoundRobinStrategy,
    "uniform": UniformStrategy,
    "ce_shd_reduction": CeSHDReductionStrategy,
    "ait": AITStrategy,
    "soft_ait": SoftAITStrategy,
    "trained": TrainedStrategy,
    "soft_trained": SoftTrainedStrategy,
    "gradients_l2": GradientsL2Strategy,
    "soft_gradients_l2": SoftGradientsL2Strategy,
    "gradients_l2_hypothetical_samples": GradientsL2HypotheticalSamplesStrategy,
    "soft_gradients_l2_hypothetical_samples": SoftGradientsL2HypotheticalSamplesStrategy,
}


def get_strategy_from_name_and_graph_fitting_object(
    name: str, gf_obj: "graph_fitting.GraphFitting"
) -> AcquisitionStrategy:
    if name in strategies.keys():
        return strategies[name].from_graph_fitting_object(gf_obj)
    else:
        raise ValueError(f"Unknown strategy: {name}")
