import torch
from torch import optim
from tqdm import tqdm
import random
from sklearn.metrics import classification_report as sk_classification_report
from seqeval.metrics import classification_report
from transformers.optimization import get_linear_schedule_with_warmup

from .metrics import eval_result,compute_metrics


class BaseTrainer(object):  
    def train(self):
        raise NotImplementedError()

    def evaluate(self):
        raise NotImplementedError()

    def test(self):
        raise NotImplementedError()

class RTVTrainer(BaseTrainer):
    def __init__(self, train_data=None, dev_data=None, test_data=None, model=None, processor=None, args=None, logger=None, writer=None) -> None:
        self.train_data = train_data  
        self.dev_data = dev_data  
        self.test_data = test_data 
        self.model = model  
        self.processor = processor  
        self.re_dict = processor.get_relation_dict()  
        self.logger = logger
        self.writer = writer
        self.refresh_step = 2
        self.best_train_metric = 0
        self.best_dev_metric = 0
        self.best_test_metric = 0
        self.best_train_epoch = None
        self.best_dev_epoch = None
        self.best_test_epoch = None
        self.optimizer = None
        if self.train_data is not None:
            self.train_num_steps = len(self.train_data) * args.num_epochs
        self.step = 0
        self.args = args
        if self.args.use_prompt:  
            self.before_multimodal_train()  
        else:
            self.before_train()  

    def get_sim_mat(self):
        with torch.no_grad():
            x1,x2,x3,x4,x5,x6=[],[],[],[],[],[]
            for batch in self.train_data:
                batch = (tup.to(self.args.device) if isinstance(tup, torch.Tensor) else tup for tup in batch)
                
                input_ids1, token_type_ids1, attention_mask1, labels1, images1, aux_imgs1 = batch
                x1.append(input_ids1)
                x2.append(token_type_ids1)
                x3.append(attention_mask1)
                x4.append(labels1)
                x5.append(images1)
                x6.append(aux_imgs1)
            input_ids=torch.cat(x1, dim=0)
            attention_mask=torch.cat(x2, dim=0)
            token_type_ids=torch.cat(x3, dim=0)
            labels=torch.cat(x4, dim=0)
            images=torch.cat(x5, dim=0)
            aux_imgs=torch.cat(x6, dim=0)
            return self.model.get_sim_mat(input_ids,
            attention_mask,
            token_type_ids,
            labels,
            images,
            aux_imgs,)
    
    def train(self):
        self.step = 0
        self.model.train()
        self.logger.info("***** Running training *****")
        self.logger.info("  Num instance = %d", len(self.train_data)*self.args.batch_size)
        self.logger.info("  Num epoch = %d", self.args.num_epochs)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        self.logger.info("  Learning rate = {}".format(self.args.lr))
        self.logger.info("  Evaluate begin = %d", self.args.eval_begin_epoch)
        if self.args.load_path is not None:  
            self.logger.info("Loading model from {}".format(self.args.load_path))
            self.model.load_state_dict(torch.load(self.args.load_path))
            self.logger.info("Load model successful!")
        with tqdm(total=self.train_num_steps, postfix='loss:{0:<6.5f}', leave=False, dynamic_ncols=True, initial=self.step) as pbar:  
            self.pbar = pbar
            avg_loss = 0
            word_all=[]
            video_all=[]
            aux_all=[]
            for epoch in range(1, self.args.num_epochs+1): 
                pbar.set_description_str(desc="Epoch {}/{}".format(epoch, self.args.num_epochs))  
                total_loss = 0
                for batch in self.train_data:
                    self.step += 1
                    batch = (tup.to(self.args.device) if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    (loss, score) = self._step(batch, mode="train")  
                    tmp_loss = avg_loss + loss  
                    avg_loss = tmp_loss
                    tmp_loss = total_loss
                    total_loss = tmp_loss
                    loss_tensor = torch.tensor(loss, requires_grad=True)
                    loss_tensor.backward()  
                    self.optimizer.step()
                    self.scheduler.step()
                    self.optimizer.zero_grad()
                    if self.step % self.refresh_step == 0:  
                        avg_loss = float(avg_loss) / self.refresh_step
                        print_output = "loss:{:<6.5f}".format(avg_loss)
                        pbar.update(self.refresh_step)
                        pbar.set_postfix_str(print_output)  
                        if self.writer:
                            self.writer.add_scalar(tag='train_loss', scalar_value=avg_loss, global_step=self.step)    
                        avg_loss = 0
                    result = compute_metrics(self.get_sim_mat())
                    R1=result["R1"]
                    R5=result["R5"]
                    MR=result["MR"]
                if self.writer:
                    self.writer.add_scalar(tag='dev_R1', scalar_value=R1, global_step=epoch)    
                    self.writer.add_scalar(tag='dev_R5', scalar_value=R5, global_step=epoch) 
                    self.writer.add_scalar(tag='dev_MR', scalar_value=MR, global_step=epoch) 
                    self.writer.add_scalar(tag='dev_loss', scalar_value=total_loss/len(self.test_data), global_step=epoch)    
                self.logger.info("R1: {},R5: {}, MR: {}, tot_loss: {}"\
                            .format(R1, R5, MR, total_loss/len(self.test_data)))
                self.logger.info("Epoch {}/{}, best dev f1: {}, best epoch: {}, current dev R1 score: {}."\
                            .format(epoch, self.args.num_epochs, self.best_train_metric, self.best_train_epoch, R1))
                if R1 >= self.best_train_metric:  
                    self.logger.info("Get better performance at epoch {}".format(epoch))
                    self.best_train_epoch = epoch
                    self.best_train_metric = R1 
                if epoch >= self.args.eval_begin_epoch:  
                    self.evaluate(epoch)   
                    self.test()
            pbar.close()
            self.pbar = None
            self.logger.info("Get best dev performance at epoch {}, best dev f1 score is {}".format(self.best_dev_epoch, self.best_dev_metric))
            self.logger.info("Get best test performance at epoch {}, best test f1 score is {}".format(self.best_test_epoch, self.best_test_metric))

    def evaluate(self, epoch): 
        self.model.eval()
        self.logger.info("***** Running evaluate *****")
        self.logger.info("  Num instance = %d", len(self.dev_data)*self.args.batch_size)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        step = 0
        true_labels, pred_labels = [], []
        with torch.no_grad():
            with tqdm(total=len(self.dev_data), leave=False, dynamic_ncols=True) as pbar:
                pbar.set_description_str(desc="Dev")
                total_loss = 0
                for batch in self.dev_data:
                    step += 1
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    (loss, score)= self._step(batch, mode="dev")   
                    tmp_loss = total_loss + loss
                    total_loss = tmp_loss
                    pbar.update()
                pbar.close()
                result = compute_metrics(self.get_sim_mat())
                R1=result["R1"]
                R5=result["R5"]
                MR=result["MR"]
                if self.writer:
                    self.writer.add_scalar(tag='dev_R1', scalar_value=R1, global_step=epoch)    
                    self.writer.add_scalar(tag='dev_R5', scalar_value=R5, global_step=epoch) 
                    self.writer.add_scalar(tag='dev_MR', scalar_value=MR, global_step=epoch) 
                    self.writer.add_scalar(tag='dev_loss', scalar_value=total_loss/len(self.test_data), global_step=epoch)   
                self.logger.info("R1: {},R5: {}, MR: {}, tot_loss: {}"\
                            .format(R1, R5, MR, total_loss/len(self.test_data)))
                self.logger.info("Epoch {}/{}, best dev f1: {}, best epoch: {}, current dev R1 score: {}"\
                            .format(epoch, self.args.num_epochs, self.best_dev_metric, self.best_dev_epoch, R1))
                if R1 >= self.best_dev_metric:  
                    self.logger.info("Get better performance at epoch {}".format(epoch))
                    self.best_dev_epoch = epoch
                    self.best_dev_metric = R1 
                    if self.args.save_path is not None:
                        torch.save(self.model.state_dict(), self.args.save_path + self.args.file_name)
                        self.logger.info("Save best model at {}".format(self.args.save_path))

        self.model.train()

    def test(self):  
        self.model.eval()
        self.logger.info("\n***** Running testing *****")
        self.logger.info("  Num instance = %d", len(self.test_data)*self.args.batch_size)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        
        if self.args.load_path is not None:  
            self.logger.info("Loading model from {}".format(self.args.load_path))
            self.model.load_state_dict(torch.load(self.args.load_path))
            self.logger.info("Load model successful!")
        true_labels, pred_labels = [], []
        with torch.no_grad():
            with tqdm(total=len(self.test_data), leave=False, dynamic_ncols=True) as pbar:
                pbar.set_description_str(desc="Testing")
                total_loss = 0
                for batch in self.test_data:
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    (loss, score) = self._step(batch, mode="dev")    
                    tmp_loss = total_loss + loss
                    total_loss = tmp_loss 
                    pbar.update()
                pbar.close()
                result = compute_metrics(self.get_sim_mat())
                R1=result["R1"]
                R5=result["R5"]
                MR=result["MR"]
                if self.writer:
                    self.writer.add_scalar(tag='test_R1', scalar_value=R1)   
                    self.writer.add_scalar(tag='test_R5', scalar_value=R5) 
                    self.writer.add_scalar(tag='test_MR', scalar_value=MR) 
                    self.writer.add_scalar(tag='test_loss', scalar_value=total_loss/len(self.test_data))    
                total_loss = 0
                self.logger.info("Test R1 score: {}.".format(R1))           
        self.model.train()
        
    def _step(self, batch, mode="train"):
        if mode != "predict":
            if self.args.use_prompt:  
                input_ids, token_type_ids, attention_mask, labels, images, aux_imgs = batch  
            else:
                images, aux_imgs = None, None  
                input_ids, token_type_ids, attention_mask, labels = batch
            outputs = self.model(input_ids=input_ids, attention_mask=attention_mask,
                                 token_type_ids=token_type_ids, labels=labels,
                                 images=images, aux_imgs=aux_imgs)
            return outputs

    def before_train(self):
        no_decay = ['bias', 'LayerNorm.weight']
        optimizer_grouped_parameters = [
                {'params': [p for n, p in self.model.named_parameters() if not any(nd in n for nd in no_decay)], 'weight_decay': 1e-2},
                {'params': [p for n, p in self.model.named_parameters() if any(nd in n for nd in no_decay)], 'weight_decay': 0.0}
        ]
        self.optimizer = optim.AdamW(optimizer_grouped_parameters, lr=self.args.lr)
        self.scheduler = get_linear_schedule_with_warmup(optimizer=self.optimizer, 
                                                            num_warmup_steps=self.args.warmup_ratio*self.train_num_steps, 
                                                                num_training_steps=self.train_num_steps)
        self.model.to(self.args.device)

    
    def before_multimodal_train(self):
        optimizer_grouped_parameters = []
        params = {'lr':self.args.lr, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'bert' in name:
                params['params'].append(param)
        optimizer_grouped_parameters.append(params)
        params = {'lr':self.args.lr, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'encoder_conv' in name or 'gates' in name:
                params['params'].append(param)
        optimizer_grouped_parameters.append(params)
        for name, param in self.model.named_parameters():
            if 'image_model' in name:
                param.require_grad = False
        self.optimizer = optim.AdamW(optimizer_grouped_parameters, lr=self.args.lr)
        self.scheduler = get_linear_schedule_with_warmup(optimizer=self.optimizer, 
                                                            num_warmup_steps=self.args.warmup_ratio*self.train_num_steps, 
                                                                num_training_steps=self.train_num_steps)
        self.model.to(self.args.device)

class RETrainer(BaseTrainer):
    def __init__(self, train_data=None, dev_data=None, test_data=None, model=None, processor=None, args=None, logger=None, writer=None) -> None:
        self.train_data = train_data  
        self.dev_data = dev_data  
        self.test_data = test_data  
        self.model = model  
        self.processor = processor  
        self.re_dict = processor.get_relation_dict()  
        self.logger = logger
        self.writer = writer
        self.refresh_step = 2
        self.best_train_metric = 0
        self.best_dev_metric = 0
        self.best_test_metric = 0
        self.best_train_epoch = None
        self.best_dev_epoch = None
        self.best_test_epoch = None
        self.optimizer = None
        if self.train_data is not None:
            self.train_num_steps = len(self.train_data) * args.num_epochs
        self.step = 0
        self.args = args
        if self.args.use_prompt:  
            self.before_multimodal_train()  
        else:
            self.before_train()  

    def train(self):
        self.step = 0
        self.model.train()
        self.logger.info("***** Running training *****")
        self.logger.info("  Num instance = %d", len(self.train_data)*self.args.batch_size)
        self.logger.info("  Num epoch = %d", self.args.num_epochs)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        self.logger.info("  Learning rate = {}".format(self.args.lr))
        self.logger.info("  Evaluate begin = %d", self.args.eval_begin_epoch)
        if self.args.load_path is not None:  
            self.logger.info("Loading model from {}".format(self.args.load_path))
            self.model.load_state_dict(torch.load(self.args.load_path))
            self.logger.info("Load model successful!")
        with tqdm(total=self.train_num_steps, postfix='loss:{0:<6.5f}', leave=False, dynamic_ncols=True, initial=self.step) as pbar: 
            self.pbar = pbar
            avg_loss = 0
            for epoch in range(1, self.args.num_epochs+1):  
                pbar.set_description_str(desc="Epoch {}/{}".format(epoch, self.args.num_epochs))  
                total_loss = 0
                true_labels, pred_labels = [], []
                for batch in self.train_data:
                    self.step += 1
                    batch = (tup.to(self.args.device) if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    (loss, logits), labels = self._step(batch, mode="train")  
                    tmp_loss = avg_loss + loss.detach().cpu().item()  
                    avg_loss = tmp_loss
                    tmp_loss = total_loss + loss.detach().cpu().item()
                    total_loss = tmp_loss
                    preds = logits.argmax(-1)
                    true_labels.extend(labels.view(-1).detach().cpu().tolist())
                    pred_labels.extend(preds.view(-1).detach().cpu().tolist())
                    loss.backward()  
                    self.optimizer.step()
                    self.scheduler.step()
                    self.optimizer.zero_grad()
                    if self.step % self.refresh_step == 0:  
                        avg_loss = float(avg_loss) / self.refresh_step
                        print_output = "loss:{:<6.5f}".format(avg_loss)
                        pbar.update(self.refresh_step)
                        pbar.set_postfix_str(print_output)  
                        if self.writer:
                            self.writer.add_scalar(tag='train_loss', scalar_value=avg_loss, global_step=self.step)    
                        avg_loss = 0
                sk_result = sk_classification_report(y_true=true_labels, y_pred=pred_labels, labels=list(self.re_dict.values())[1:], target_names=list(self.re_dict.keys())[1:], digits=4)
                self.logger.info("%s\n", sk_result)
                result = eval_result(true_labels, pred_labels, self.re_dict, self.logger)
                acc, micro_f1 = round(result['acc']*100, 4), round(result['micro_f1']*100, 4)
                if self.writer:
                    self.writer.add_scalar(tag='dev_acc', scalar_value=acc, global_step=epoch)    
                    self.writer.add_scalar(tag='dev_f1', scalar_value=micro_f1, global_step=epoch)    
                    self.writer.add_scalar(tag='dev_loss', scalar_value=total_loss/len(self.test_data), global_step=epoch)    
                self.logger.info("Epoch {}/{}, best dev f1: {}, best epoch: {}, current dev f1 score: {}, acc: {}."\
                            .format(epoch, self.args.num_epochs, self.best_train_metric, self.best_train_epoch, micro_f1, acc))
                if micro_f1 >= self.best_train_metric:  
                    self.logger.info("Get better performance at epoch {}".format(epoch))
                    self.best_train_epoch = epoch
                    self.best_train_metric = micro_f1 
                if epoch >= self.args.eval_begin_epoch:  
                    self.evaluate(epoch)   
                    self.test()
            pbar.close()
            self.pbar = None
            self.logger.info("Get best dev performance at epoch {}, best dev f1 score is {}".format(self.best_dev_epoch, self.best_dev_metric))
            self.logger.info("Get best test performance at epoch {}, best test f1 score is {}".format(self.best_test_epoch, self.best_test_metric))

    def evaluate(self, epoch):  
        self.model.eval()
        self.logger.info("***** Running evaluate *****")
        self.logger.info("  Num instance = %d", len(self.dev_data)*self.args.batch_size)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        step = 0
        true_labels, pred_labels = [], []
        with torch.no_grad():
            with tqdm(total=len(self.dev_data), leave=False, dynamic_ncols=True) as pbar:
                pbar.set_description_str(desc="Dev")
                total_loss = 0
                for batch in self.dev_data:
                    step += 1
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    (loss, logits), labels = self._step(batch, mode="dev")   
                    tmp_loss = total_loss + loss.detach().cpu().item()
                    total_loss = tmp_loss
                    preds = logits.argmax(-1)
                    true_labels.extend(labels.view(-1).detach().cpu().tolist())
                    pred_labels.extend(preds.view(-1).detach().cpu().tolist())
                    pbar.update()
                pbar.close()
                sk_result = sk_classification_report(y_true=true_labels, y_pred=pred_labels, labels=list(self.re_dict.values())[1:], target_names=list(self.re_dict.keys())[1:], digits=4)
                self.logger.info("%s\n", sk_result)
                result = eval_result(true_labels, pred_labels, self.re_dict, self.logger)
                acc, micro_f1 = round(result['acc']*100, 4), round(result['micro_f1']*100, 4)
                if self.writer:
                    self.writer.add_scalar(tag='dev_acc', scalar_value=acc, global_step=epoch)   
                    self.writer.add_scalar(tag='dev_f1', scalar_value=micro_f1, global_step=epoch)    
                    self.writer.add_scalar(tag='dev_loss', scalar_value=total_loss/len(self.test_data), global_step=epoch)   
                self.logger.info("Epoch {}/{}, best dev f1: {}, best epoch: {}, current dev f1 score: {}, acc: {}."\
                            .format(epoch, self.args.num_epochs, self.best_dev_metric, self.best_dev_epoch, micro_f1, acc))
                if micro_f1 >= self.best_dev_metric:  
                    self.logger.info("Get better performance at epoch {}".format(epoch))
                    self.best_dev_epoch = epoch
                    self.best_dev_metric = micro_f1 
                    if self.args.save_path is not None:
                        torch.save(self.model.state_dict(), self.args.save_path + self.args.file_name)
                        self.logger.info("Save best model at {}".format(self.args.save_path))
        self.model.train()

    def test(self):  
        self.model.eval()
        self.logger.info("\n***** Running testing *****")
        self.logger.info("  Num instance = %d", len(self.test_data)*self.args.batch_size)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        if self.args.load_path is not None:  
            self.logger.info("Loading model from {}".format(self.args.load_path))
            self.model.load_state_dict(torch.load(self.args.load_path))
            self.logger.info("Load model successful!")
        true_labels, pred_labels = [], []
        with torch.no_grad():
            with tqdm(total=len(self.test_data), leave=False, dynamic_ncols=True) as pbar:
                pbar.set_description_str(desc="Testing")
                total_loss = 0
                for batch in self.test_data:
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    (loss, logits), labels = self._step(batch, mode="dev")    
                    tmp_loss = total_loss + loss.detach().cpu().item()
                    total_loss = tmp_loss            
                    preds = logits.argmax(-1)
                    true_labels.extend(labels.view(-1).detach().cpu().tolist())
                    pred_labels.extend(preds.view(-1).detach().cpu().tolist())         
                    pbar.update()
                pbar.close()
                sk_result = sk_classification_report(y_true=true_labels, y_pred=pred_labels, labels=list(self.re_dict.values())[1:], target_names=list(self.re_dict.keys())[1:], digits=4)
                self.logger.info("%s\n", sk_result)
                result = eval_result(true_labels, pred_labels, self.re_dict, self.logger)
                acc, micro_f1 = round(result['acc']*100, 4), round(result['micro_f1']*100, 4)
                if self.writer:
                    self.writer.add_scalar(tag='test_acc', scalar_value=acc)    
                    self.writer.add_scalar(tag='test_f1', scalar_value=micro_f1)    
                    self.writer.add_scalar(tag='test_loss', scalar_value=total_loss/len(self.test_data))    
                total_loss = 0
                self.logger.info("Test f1 score: {}, acc: {}.".format(micro_f1, acc))
                    
        self.model.train()
        
    def _step(self, batch, mode="train"):
        if mode != "predict":
            if self.args.use_prompt:  
                input_ids, token_type_ids, attention_mask, labels, images, aux_imgs = batch  
            else:
                images, aux_imgs = None, None  
                input_ids, token_type_ids, attention_mask, labels = batch
            outputs = self.model(input_ids=input_ids, attention_mask=attention_mask,
                                 token_type_ids=token_type_ids, labels=labels,
                                 images=images, aux_imgs=aux_imgs)
            return outputs, labels

    def before_train(self):
        no_decay = ['bias', 'LayerNorm.weight']
        optimizer_grouped_parameters = [
                {'params': [p for n, p in self.model.named_parameters() if not any(nd in n for nd in no_decay)], 'weight_decay': 1e-2},
                {'params': [p for n, p in self.model.named_parameters() if any(nd in n for nd in no_decay)], 'weight_decay': 0.0}
        ]
        self.optimizer = optim.AdamW(optimizer_grouped_parameters, lr=self.args.lr)
        self.scheduler = get_linear_schedule_with_warmup(optimizer=self.optimizer, 
                                                            num_warmup_steps=self.args.warmup_ratio*self.train_num_steps, 
                                                                num_training_steps=self.train_num_steps)
        self.model.to(self.args.device)

    
    def before_multimodal_train(self):
        optimizer_grouped_parameters = []
        params = {'lr':self.args.lr, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'bert' in name:
                params['params'].append(param)
        optimizer_grouped_parameters.append(params)
        params = {'lr':self.args.lr, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'encoder_conv' in name or 'gates' in name:
                params['params'].append(param)
        optimizer_grouped_parameters.append(params)
        for name, param in self.model.named_parameters():
            if 'image_model' in name:
                param.require_grad = False
        self.optimizer = optim.AdamW(optimizer_grouped_parameters, lr=self.args.lr)
        self.scheduler = get_linear_schedule_with_warmup(optimizer=self.optimizer, 
                                                            num_warmup_steps=self.args.warmup_ratio*self.train_num_steps, 
                                                                num_training_steps=self.train_num_steps)
        self.model.to(self.args.device)


class NERTrainer(BaseTrainer):
    def __init__(self, train_data=None, dev_data=None, test_data=None, model=None, processor=None, label_map=None, args=None, logger=None,  writer=None) -> None:
        self.train_data = train_data
        self.dev_data = dev_data
        self.test_data = test_data
        self.model = model
        self.processor = processor
        self.logger = logger
        self.label_map = label_map
        self.writer = writer
        self.refresh_step = 2
        self.best_dev_metric = 0
        self.best_test_metric = 0
        self.best_train_metric = 0
        self.best_dev_epoch = None
        self.best_test_epoch = None
        self.best_train_epoch = None
        self.optimizer = None
        if self.train_data is not None:
            self.train_num_steps = len(self.train_data) * args.num_epochs
        self.step = 0
        self.args = args

    def train(self):
        if self.args.use_prompt:
            self.multiModal_before_train()
        else:
            self.bert_before_train()
        self.step = 0
        self.model.train()
        self.logger.info("***** Running training *****")
        self.logger.info("  Num instance = %d", len(self.train_data)*self.args.batch_size)
        self.logger.info("  Num epoch = %d", self.args.num_epochs)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        self.logger.info("  Learning rate = {}".format(self.args.lr))
        self.logger.info("  Evaluate begin = %d", self.args.eval_begin_epoch)
        if self.args.load_path is not None:  
            self.logger.info("Loading model from {}".format(self.args.load_path))
            self.model.load_state_dict(torch.load(self.args.load_path))
            self.logger.info("Load model successful!")
        with tqdm(total=self.train_num_steps, postfix='loss:{0:<6.5f}', leave=False, dynamic_ncols=True, initial=self.step) as pbar:
            self.pbar = pbar
            avg_loss = 0
            for epoch in range(1, self.args.num_epochs+1):
                y_true, y_pred = [], []
                pbar.set_description_str(desc="Epoch {}/{}".format(epoch, self.args.num_epochs))
                for batch in self.train_data:
                    self.step += 1
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)
                    attention_mask, labels, logits, loss = self._step(batch, mode="train")
                    avg_loss += loss.detach().cpu().item()
                    loss.backward()
                    self.optimizer.step()
                    self.scheduler.step()
                    self.optimizer.zero_grad()
                    if isinstance(logits, torch.Tensor):    
                        logits = logits.argmax(-1).detach().cpu().numpy() 
                    label_ids = labels.to('cpu').numpy()
                    input_mask = attention_mask.to('cpu').numpy()
                    label_map = {idx:label for label, idx in self.label_map.items()}
                    for row, mask_line in enumerate(input_mask):
                        true_label = []
                        true_predict = []
                        for column, mask in enumerate(mask_line):
                            if column == 0:
                                continue
                            if mask:
                                if label_map[label_ids[row][column]] != "X" and label_map[label_ids[row][column]] != "[SEP]":
                                    true_label.append(label_map[label_ids[row][column]])
                                    true_predict.append(label_map[logits[row][column]])
                            else:
                                break
                        y_true.append(true_label)
                        y_pred.append(true_predict)
                    if self.step % self.refresh_step == 0:
                        avg_loss = float(avg_loss) / self.refresh_step
                        print_output = "loss:{:<6.5f}".format(avg_loss)
                        pbar.update(self.refresh_step)
                        pbar.set_postfix_str(print_output)
                        if self.writer:
                            self.writer.add_scalar(tag='train_loss', scalar_value=avg_loss, global_step=self.step)    
                        avg_loss = 0
                results = classification_report(y_true, y_pred, digits=4) 
                self.logger.info("***** Train Eval results *****")
                self.logger.info("\n%s", results)
                f1_score = float(results.split('\n')[-4].split('      ')[0].split('    ')[3])
                if self.writer:
                    self.writer.add_scalar(tag='train_f1', scalar_value=f1_score, global_step=epoch)  
                self.logger.info("Epoch {}/{}, best train f1: {}, best epoch: {}, current train f1 score: {}."\
                            .format(epoch, self.args.num_epochs, self.best_train_metric, self.best_train_epoch, f1_score))
                if f1_score > self.best_train_metric:
                    self.best_train_metric = f1_score
                    self.best_train_epoch = epoch
                if epoch >= self.args.eval_begin_epoch:
                    self.evaluate(epoch)  
            torch.cuda.empty_cache()
            pbar.close()
            self.pbar = None
            self.logger.info("Get best dev performance at epoch {}, best dev f1 score is {}".format(self.best_dev_epoch, self.best_dev_metric))
            self.logger.info("Get best test performance at epoch {}, best test f1 score is {}".format(self.best_test_epoch, self.best_test_metric))

    def evaluate(self, epoch):
        self.model.eval()
        self.logger.info("***** Running evaluate *****")
        self.logger.info("  Num instance = %d", len(self.dev_data)*self.args.batch_size)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        y_true, y_pred = [], []
        step = 0
        with torch.no_grad():
            with tqdm(total=len(self.dev_data), leave=False, dynamic_ncols=True) as pbar:
                pbar.set_description_str(desc="Dev")
                total_loss = 0
                for batch in self.dev_data:
                    step += 1
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    attention_mask, labels, logits, loss = self._step(batch, mode="dev")    
                    total_loss += loss.detach().cpu().item()

                    if isinstance(logits, torch.Tensor):
                        logits = logits.argmax(-1).detach().cpu().numpy()  
                    label_ids = labels.detach().cpu().numpy()
                    input_mask = attention_mask.detach().cpu().numpy()
                    label_map = {idx:label for label, idx in self.label_map.items()}
                    for row, mask_line in enumerate(input_mask):
                        true_label = []
                        true_predict = []
                        for column, mask in enumerate(mask_line):
                            if column == 0:
                                continue
                            if mask:
                                if label_map[label_ids[row][column]] != "X" and label_map[label_ids[row][column]] != "[SEP]":
                                    true_label.append(label_map[label_ids[row][column]])
                                    true_predict.append(label_map[logits[row][column]])
                            else:
                                break
                        y_true.append(true_label)
                        y_pred.append(true_predict)
                    pbar.update()
                pbar.close()
                results = classification_report(y_true, y_pred, digits=4)  
                self.logger.info("***** Dev Eval results *****")
                self.logger.info("\n%s", results)
                f1_score = float(results.split('\n')[-4].split('      ')[-2].split('    ')[-1]) 
                if self.writer:
                    self.writer.add_scalar(tag='dev_f1', scalar_value=f1_score, global_step=epoch)    
                    self.writer.add_scalar(tag='dev_loss', scalar_value=total_loss/step, global_step=epoch)   
                self.logger.info("Epoch {}/{}, best dev f1: {}, best epoch: {}, current dev f1 score: {}."\
                            .format(epoch, self.args.num_epochs, self.best_dev_metric, self.best_dev_epoch, f1_score))
                if f1_score >= self.best_dev_metric:  
                    self.logger.info("Get better performance at epoch {}".format(epoch))
                    self.best_dev_epoch = epoch
                    self.best_dev_metric = f1_score 
                    if self.args.save_path is not None:
                        torch.save(self.model.state_dict(), self.args.save_path + self.args.file_name)
                        self.logger.info("Save best model at {}".format(self.args.save_path))

        self.model.train()

    def test(self):
        self.model.eval()
        self.logger.info("\n***** Running testing *****")
        self.logger.info("  Num instance = %d", len(self.test_data)*self.args.batch_size)
        self.logger.info("  Batch size = %d", self.args.batch_size)
        if self.args.load_path is not None:  
            self.logger.info("Loading model from {}".format(self.args.load_path))
            self.model.load_state_dict(torch.load(self.args.load_path))
            self.logger.info("Load model successful!")
        y_true, y_pred = [], []
        with torch.no_grad():
            with tqdm(total=len(self.test_data), leave=False, dynamic_ncols=True) as pbar:
                pbar.set_description_str(desc="Testing")
                total_loss = 0
                for batch in self.test_data:
                    batch = (tup.to(self.args.device)  if isinstance(tup, torch.Tensor) else tup for tup in batch)  
                    attention_mask, labels, logits, loss = self._step(batch, mode="dev")    
                    total_loss += loss.detach().cpu().item()
                    if isinstance(logits, torch.Tensor): 
                        logits = logits.argmax(-1).detach().cpu().tolist()  
                    label_ids = labels.detach().cpu().numpy()
                    input_mask = attention_mask.detach().cpu().numpy()
                    label_map = {idx:label for label, idx in self.label_map.items()}
                    for row, mask_line in enumerate(input_mask):
                        true_label = []
                        true_predict = []
                        for column, mask in enumerate(mask_line):
                            if column == 0:
                                continue
                            if mask:
                                if label_map[label_ids[row][column]] != "X" and label_map[label_ids[row][column]] != "[SEP]":
                                    true_label.append(label_map[label_ids[row][column]])
                                    true_predict.append(label_map[logits[row][column]])
                            else:
                                break
                        y_true.append(true_label)
                        y_pred.append(true_predict)
                    pbar.update()
                pbar.close()
                results = classification_report(y_true, y_pred, digits=4) 
                self.logger.info("***** Test Eval results *****")
                self.logger.info("\n%s", results)
                f1_score = float(results.split('\n')[-4].split('      ')[-2].split('    ')[-1])
                if self.writer:
                    self.writer.add_scalar(tag='test_f1', scalar_value=f1_score)   
                    self.writer.add_scalar(tag='test_loss', scalar_value=total_loss/len(self.test_data))    
                total_loss = 0
                self.logger.info("Test f1 score: {}.".format(f1_score))
                    
        self.model.train()
        
    def _step(self, batch, mode="train"):
        if self.args.use_prompt:
            input_ids, token_type_ids, attention_mask, labels, images, aux_imgs = batch
        else:
            images, aux_imgs = None, None
            input_ids, token_type_ids, attention_mask, labels = batch
        output = self.model(input_ids=input_ids, attention_mask=attention_mask, token_type_ids=token_type_ids, labels=labels, images=images, aux_imgs=aux_imgs)
        logits, loss = output.logits, output.loss
        return attention_mask, labels, logits, loss



    def bert_before_train(self):
        self.optimizer = optim.AdamW(self.model.parameters(), lr=self.args.lr)
        self.model.to(self.args.device)
        self.scheduler = get_linear_schedule_with_warmup(optimizer=self.optimizer, 
                                                            num_warmup_steps=self.args.warmup_ratio*self.train_num_steps, 
                                                            num_training_steps=self.train_num_steps)

    def multiModal_before_train(self):
        parameters = []
        params = {'lr':self.args.lr, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'bert' in name:
                params['params'].append(param)
        parameters.append(params)
        params = {'lr':self.args.lr, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'encoder_conv' in name or 'gates' in name:
                params['params'].append(param)
        parameters.append(params)
        params = {'lr':5e-2, 'weight_decay':1e-2}
        params['params'] = []
        for name, param in self.model.named_parameters():
            if 'crf' in name or name.startswith('fc'):
                params['params'].append(param)
        parameters.append(params)
        self.optimizer = optim.AdamW(parameters)
        for name, par in self.model.named_parameters(): 
            if 'image_model' in name:   par.requires_grad = False
        self.scheduler = get_linear_schedule_with_warmup(optimizer=self.optimizer, 
                                                            num_warmup_steps=self.args.warmup_ratio*self.train_num_steps, 
                                                            num_training_steps=self.train_num_steps)
        self.model.to(self.args.device)
