import numpy as np
import torch
import torch.nn as nn
import gym
import sys

from typing import Dict, Union, Tuple
from copy import deepcopy
from offlinerlkit.policy import BasePolicy


class CQLNRewardPolicy(BasePolicy):
    """
    Ensemble-Diversified Actor Critic <Ref: https://arxiv.org/abs/2110.01548>
    """

    def __init__(
        self,
        actor: nn.Module,
        # critic_init: nn.Module, 
        critics: nn.ModuleList,
        actor_optim: torch.optim.Optimizer,
        # critic_optim: torch.optim.Optimizer,
        critics_optim: torch.optim.Optimizer,
        action_space: gym.spaces.Space,
        tau: float = 0.005,
        gamma: float  = 0.99,
        alpha: Union[float, Tuple[float, torch.Tensor, torch.optim.Optimizer]] = 0.2,
        max_q_backup: bool = False,
        deterministic_backup: bool = True,
        eta: float = 1.0,
        cql_weight: float = 1.0,
        temperature: float = 1.0,
        num_repeat_actions:int = 10,
    ) -> None:

        super().__init__()
        self.actor = actor
        self.critics = critics
        self.critics_old = deepcopy(critics)
        self.critics_old.eval()

        self.actor_optim = actor_optim
        self.critics_optim = critics_optim
        
        self.action_space = action_space

        self._tau = tau
        self._gamma = gamma

        self._is_auto_alpha = False
        if isinstance(alpha, tuple):
            self._is_auto_alpha = True
            self._target_entropy, self._log_alpha, self.alpha_optim = alpha
            self._alpha = self._log_alpha.detach().exp()
        else:
            self._alpha = alpha

        self._max_q_backup = max_q_backup
        self._deterministic_backup = deterministic_backup
        self._eta = eta
        self._cql_weight = cql_weight
        self._temperature = temperature
        self._num_repeat_actions = num_repeat_actions
        self._num_critics = self.critics._num_ensemble
        # self.critic_list = [deepcopy(critic_init)] * self._num_critics
        
        

    def train(self) -> None:
        self.actor.train()
        self.critics.train()
        

    def eval(self) -> None:
        self.actor.eval()
        self.critics.eval()

    def _sync_weight(self) -> None:
        for o, n in zip(self.critics_old.parameters(), self.critics.parameters()):
            o.data.copy_(o.data * (1.0 - self._tau) + n.data * self._tau)
            
    def calc_pi_value(
        self,
        obs_pi: torch.Tensor,
        obs_to_pred: torch.Tensor
    ) -> Tuple[torch.Tensor, torch.Tensor]:
        act, log_prob = self.actforward(obs_pi)

        q = self.critics(obs_to_pred, act)
        # q = self.critics_old(obs_to_pred, act)

        return q - log_prob.detach()

    def calc_random_value(
        self,
        obs: torch.Tensor,
        random_act: torch.Tensor
    ) -> Tuple[torch.Tensor, torch.Tensor]:
        q = self.critics(obs, random_act)
        # q = self.critics_old(obs, random_act)

        log_prob = np.log(0.5**random_act.shape[-1])

        return q - log_prob
            


    
    def actforward(
        self,
        obs: torch.Tensor,
        deterministic: bool = False
    ) -> Tuple[torch.Tensor, torch.Tensor]:
        dist = self.actor(obs)
        if deterministic:
            squashed_action, raw_action = dist.mode()
        else:
            squashed_action, raw_action = dist.rsample()
        log_prob = dist.log_prob(squashed_action, raw_action)
        return squashed_action, log_prob

    def select_action(
        self,
        obs: np.ndarray,
        deterministic: bool = False
    ) -> np.ndarray:
        with torch.no_grad():
            action, _ = self.actforward(obs, deterministic)
        return action.cpu().numpy()

    def learn_actor(self, batch: Dict) -> Dict[str, float]:
        self.obss, self.actions, self.next_obss, self.rewards, self.terminals = batch["observations"], batch["actions"], \
            batch["next_observations"], batch["rewards"], batch["terminals"]
        batch_size = self.obss.shape[0]
        
        # update actor
        a, log_probs = self.actforward(self.obss)
        qas = self.critics(self.obss, a)
        actor_loss = -torch.min(qas, 0)[0].mean() + self._alpha * log_probs.mean()
        self.actor_optim.zero_grad()
        actor_loss.backward()
        self.actor_optim.step()
        
    def learn_critic(self, batch: Dict) -> Dict[str, float]:
        self.learn(batch)

    def learn(self, batch: Dict) -> Dict:
        obss, actions, next_obss, rewards, terminals, = \
            batch["observations"], batch["actions"], batch["next_observations"], batch["rewards"], batch["terminals"]
        rewards0, rewards1, rewards2, rewards3, rewards4 = batch["rewards0"], batch["rewards1"], batch["rewards2"], batch["rewards3"], batch["rewards4"]

        batch_size = obss.shape[0]
        
        if self._eta > 0:
            actions.requires_grad_(True)

        # update actor
        a, log_probs = self.actforward(obss)
        # qas: [num_critics, batch_size, 1]
        qas = self.critics(obss, a)
        actor_loss = -torch.min(qas, 0)[0].mean() + self._alpha * log_probs.mean()
        self.actor_optim.zero_grad()
        actor_loss.backward()
        self.actor_optim.step()

        if self._is_auto_alpha:
            log_probs = log_probs.detach() + self._target_entropy
            alpha_loss = -(self._log_alpha * log_probs).mean()
            self.alpha_optim.zero_grad()
            alpha_loss.backward()
            self.alpha_optim.step()
            self._alpha = torch.clamp(self._log_alpha.detach().exp(), 0.0, 1.0)

        # update critic
        if self._max_q_backup:
            with torch.no_grad():
                batch_size = obss.shape[0]
                tmp_next_obss = next_obss.unsqueeze(1).repeat(1, 10, 1) \
                    .view(batch_size * 10, next_obss.shape[-1])
                tmp_next_actions, _ = self.actforward(tmp_next_obss)
                tmp_next_qs = self.critics_old(tmp_next_obss, tmp_next_actions) \
                    .view(self._num_critics, batch_size, 10, 1).max(2)[0] \
                    .view(self._num_critics, batch_size, 1)
                next_q = tmp_next_qs.min(0)[0]
                print("wrong test")
        else:
            with torch.no_grad():
                next_actions, next_log_probs = self.actforward(next_obss)
                Next_q = self.critics_old(next_obss, next_actions)
                # Next_q = self.critics(next_obss, next_actions)
                next_q = self.critics_old(next_obss, next_actions).min(0)[0]

                if not self._deterministic_backup:
                    next_q -= self._alpha * next_log_probs
                    # print(next_q.shape)
                    # print("no")

        # target_q: [batch_size, 1]
        # target_q = rewards + self._gamma * (1 - terminals) * next_q
        # if self._num_critics == 5:
        #     num_cat = 1
        # else:
        #     num_cat = 10
        num_cat = int(self._num_critics / 5)
        Rewards = torch.cat([rewards0.unsqueeze(0),rewards1.unsqueeze(0), rewards2.unsqueeze(0), rewards3.unsqueeze(0), rewards4.unsqueeze(0)] * num_cat, 0)
        Target_q = Rewards + self._gamma * (1 - terminals) * Next_q
        # 计算均值，结果形状为 [256, 1]
        mean_Target_q = Target_q.mean(dim=0)
        # 计算方差，结果形状为 [256, 1]
        var_Target_q = Target_q.var(dim=0, unbiased=False)
        # print(Target_q.shape)
        # print((Target_q.min(0)).shape)
        # print(next_log_probs.shape)
        # print((Target_q.min(0)[0]).shape)
        # print((mean_val).shape)
        # print((var_val).shape)
        # sys.exit()
        

        # target_q = Target_q.min(0)[0] - self._alpha * next_log_probs
        target_q = mean_Target_q - var_Target_q - self._alpha * next_log_probs
        qs = self.critics(obss, actions)
        qs_loss = ((qs - target_q.unsqueeze(0)).pow(2)).mean(dim=(1, 2)).sum()
        critics_loss = ((qs - target_q.unsqueeze(0)).pow(2)).mean(dim=(1, 2)).sum()

        
        
        # part of cql
        # compute conservative loss
        obs_pi_value_list, next_obs_pi_value_list, random_value_list = [[None] * self._num_critics for _ in range(3)]
        q_list, cat_q_list, conservative_loss_list = [[None] * self._num_critics for _ in range(3)]
        random_actions = torch.FloatTensor(
            batch_size * self._num_repeat_actions, actions.shape[-1]
        ).uniform_(self.action_space.low[0], self.action_space.high[0]).to(self.actor.device)

        tmp_obss = obss.unsqueeze(1) \
            .repeat(1, self._num_repeat_actions, 1) \
            .view(batch_size * self._num_repeat_actions, obss.shape[-1])
        tmp_next_obss = next_obss.unsqueeze(1) \
            .repeat(1, self._num_repeat_actions, 1) \
            .view(batch_size * self._num_repeat_actions, obss.shape[-1])
        # calculate 
        obs_pi_value = self.calc_pi_value(tmp_obss, tmp_obss)
        next_obs_pi_value = self.calc_pi_value(tmp_next_obss, tmp_obss)
        random_value = self.calc_random_value(tmp_obss, random_actions) 
        for value in [
                obs_pi_value, next_obs_pi_value, random_value
            ]:
                value.reshape(self._num_critics * batch_size, self._num_repeat_actions, 1)
        
        # # normal
        # con_qs = self.critics(obss, actions)
        # cat_qs = torch.cat([obs_pi_value, next_obs_pi_value, random_value], 2) 
        # conservative_loss = \
        #         torch.logsumexp(cat_qs / self._temperature, dim=2).mean() * self._cql_weight * self._temperature - \
        #     con_qs.mean() * self._cql_weight 
        
        # # con_qs        
        # next_actions, next_log_probs = self.actforward(next_obss)        
        # con_next_qs = self.critics_old(next_obss, next_actions)
        # con_qs = Rewards + self._gamma * (1 - terminals) * con_next_qs      
        # cat_qs = torch.cat([obs_pi_value, next_obs_pi_value, random_value], 2) 
        # conservative_loss = \
        #         torch.logsumexp(cat_qs / self._temperature, dim=2).mean() * self._cql_weight * self._temperature - \
        #     con_qs.mean() * self._cql_weight 
        
        # # cat_qs
        # con_qs = self.critics(obss, actions)
        # cat_next_qs = torch.cat([obs_pi_value, next_obs_pi_value, random_value], 2) 
        # tmp_Rewards = Rewards.repeat(1, 10, 3)
        # tmp_terminals = terminals.repeat(1, 10, 3)
        # cat_qs = tmp_Rewards + self._gamma * (1 - tmp_terminals) * cat_next_qs
        # conservative_loss = \
        #         torch.logsumexp(cat_qs / self._temperature, dim=2).mean() * self._cql_weight * self._temperature - \
        #     con_qs.mean() * self._cql_weight 
        
        
        # last_next
        # next_actions, next_log_probs = self.actforward(next_obss)        
        con_qs = self.critics(next_obss, next_actions)
        # con_qs = self.critics(obss, actions)
        con_last_qs = Rewards + self._gamma * (1 - terminals) * con_qs      
        tmp_Rewards = Rewards.repeat(1, self._num_repeat_actions, 3)
        tmp_terminals = terminals.repeat(1, self._num_repeat_actions, 3)
        cat_qs = torch.cat([obs_pi_value, next_obs_pi_value, random_value], 2) 
        cat_last_qs = tmp_Rewards + self._gamma * (1 - tmp_terminals) * cat_qs
        conservative_loss = \
                torch.logsumexp(cat_last_qs / self._temperature, dim=2).mean() * self._cql_weight * self._temperature - \
            con_last_qs.mean() * self._cql_weight 
          
          
        # if self._eta < 0:   # 应该是'>'  
        # critics_loss_zero = max(conservative_loss,0)  
        critics_loss += conservative_loss # sum(conservative_loss_list) / (self._num_critics - 1) # / (self._num_critics)
        
        
        
        
        
        
       
        
        grad_loss = torch.tensor(0.0, dtype=torch.float32)
        # 均匀采样
        if self._eta < 0:   # 应该是'>'
            # last
            obss_tile = obss.unsqueeze(0).repeat(self._num_critics, 1, 1)
            actions_tile = actions.unsqueeze(0).repeat(self._num_critics, 1, 1).requires_grad_(True)
            qs_preds_tile_tmp = self.critics(obss_tile, actions_tile)
            qs_preds_tile = Rewards + self._gamma * (1 - terminals) * qs_preds_tile_tmp  
            
            
            qs_pred_grads, = torch.autograd.grad(qs_preds_tile.sum(), actions_tile, retain_graph=True, create_graph=True)
            qs_pred_grads = qs_pred_grads / (torch.norm(qs_pred_grads, p=2, dim=2).unsqueeze(-1) + 1e-10)
            qs_pred_grads = qs_pred_grads.transpose(0, 1)
            
            qs_pred_grads = torch.einsum('bik,bjk->bij', qs_pred_grads, qs_pred_grads)
            masks = torch.eye(self._num_critics, device=obss.device).unsqueeze(dim=0).repeat(qs_pred_grads.size(0), 1, 1)
            qs_pred_grads = (1 - masks) * qs_pred_grads
            grad_loss = torch.mean(torch.sum(qs_pred_grads, dim=(1, 2))) / (self._num_critics - 1)

            # grad_loss_zero = max(self._eta * grad_loss, 0)
            critics_loss += grad_loss
            # print(critics_loss.shape)
            # print(critics_loss)
            critics_loss = torch.clamp(critics_loss, min=0)
            # critics_loss = max(critics_loss, 0.0)
            # print(critics_loss.shape)
            # print(critics_loss)


        self.critics_optim.zero_grad()
        critics_loss.backward()
        self.critics_optim.step()

        self._sync_weight()
        
        # sys.exit()

        # if qs_loss is None:
        #     qs_loss = critics_loss
        # if conservative_loss is None:
        #     conservative_loss = critics_loss
        # if grad_loss is None:
            # grad_loss = critics_loss
        result =  {
            "loss/actor": actor_loss.item(),
            "loss/critics": critics_loss.item(),
            "loss/qs": qs_loss.item(),
            # "loss/conservative": conservative_loss.item(),
            # "loss/grad": grad_loss.item(),
        }

        if self._is_auto_alpha:
            result["loss/alpha"] = alpha_loss.item()
            result["alpha"] = self._alpha.item()
        
        return result




