import aiofiles
import asyncio
import logging

async def generate_async(client, model, vllm_prompt, extra_body, temperature, max_tokens, index, total):
    response = await client.completions.create(
            model=model,
            prompt=vllm_prompt,
            stream=False,
            extra_body=extra_body,
            temperature=temperature,
            max_tokens=max_tokens,
    )
    completion = response.choices[0].text.strip()
    print(f"Generated question for prompt {index+1}", flush=True)
    return completion


async def async_wrapper(vllm_client, model_id, prompts, extra_body, temperature, max_tokens, batch_size=50, temp_file=None, custom_fnc=None, custom_fnc_extra_kwargs=None):
    total = len(prompts)
    logging.info(f"Starting to process {total} prompts.")
    if not custom_fnc_extra_kwargs:
        custom_fnc_extra_kwargs = {}

    outputs = []
    for start in range(0, total, batch_size):
        end = min(start + batch_size, total)
        batch_prompts = prompts[start:end]

        logging.info(f"Processing batch {start//batch_size + 1}/{(total + batch_size - 1) // batch_size} (prompts {start} to {end-1}).")

        fnc = custom_fnc if custom_fnc else generate_async
        tasks = [
            fnc(vllm_client, model_id, prompt, extra_body, temperature, max_tokens, i, total, **custom_fnc_extra_kwargs)
            for i, prompt in enumerate(batch_prompts, start=start)                                                                          ]

        results = await asyncio.gather(*tasks, return_exceptions=True)

        # Process results and log progress
        for index, (result, prompt) in enumerate(zip(results, batch_prompts), start=start):
            if isinstance(result, Exception):
                logging.error(f'Processing prompt "{prompt[:50]}..." generated an exception: {result}')
            else:
                outputs.append(result)
                logging.info(f"Processed {index+1}/{total} prompts successfully.")

                # Append results to temp_file asynchronously if provided
                if temp_file is not None:
                    async with aiofiles.open(temp_file, 'a') as f:
                        await f.write(prompt.replace('\n', '').replace(';', ',') + ";" + result.replace('\n', ' ').replace(';', ',') + '\n')

    logging.info("Completed all prompts.")
    return outputs


def get_bm25_context(
    item,
) -> list[str] :
    return [item['context']]

def get_prompt_context(
    item,
) -> str:
    return item['context']

def get_gt_answer(
    item,
) -> str:
    return item['answers']['text']
