/* This file is part of the Pangolin Project.
 * http://github.com/stevenlovegrove/Pangolin
 *
 * Copyright (c) 2014 Steven Lovegrove
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#pragma once

#include <pangolin/video/video_interface.h>

namespace pangolin
{

enum class TransformOptions
{
    None, FlipX, FlipY, FlipXY,
    Transpose, RotateCW, RotateCCW,
};

// Video class that transforms its video input using the specified method.
class PANGOLIN_EXPORT TransformVideo :
    public VideoInterface,
    public VideoFilterInterface,
    public BufferAwareVideoInterface
{
public:
    TransformVideo(std::unique_ptr<VideoInterface>& videoin, const std::vector<TransformOptions>& flips);
    ~TransformVideo();

    //! Implement VideoInput::Start()
    void Start();

    //! Implement VideoInput::Stop()
    void Stop();

    //! Implement VideoInput::SizeBytes()
    size_t SizeBytes() const;

    //! Implement VideoInput::Streams()
    const std::vector<StreamInfo>& Streams() const;

    //! Implement VideoInput::GrabNext()
    bool GrabNext( unsigned char* image, bool wait = true );

    //! Implement VideoInput::GrabNewest()
    bool GrabNewest( unsigned char* image, bool wait = true );

    //! Implement VideoFilterInterface method
    std::vector<VideoInterface*>& InputStreams();

    uint32_t AvailableFrames() const;

    bool DropNFrames(uint32_t n);

protected:
    void Process(unsigned char* image, const unsigned char* buffer);

    std::unique_ptr<VideoInterface> videoin;
    std::vector<VideoInterface*> inputs;
    std::vector<StreamInfo> streams;
    std::vector<TransformOptions> flips;
    size_t size_bytes;
    unsigned char* buffer;

    picojson::value device_properties;
    picojson::value frame_properties;
};

}
