# Model is a simple class that handles the API requests to different end services.

# import Model class
from google import genai
from google.genai import types
import json

from pydantic import BaseModel

from credentials import GEMINI_CREDENTIAL

from src.entity.models.Model import Model
class Gemini(Model):
    api_key: str
    ctx_len: int
    def __init__(self, ctx_len: int=512):
        # load from environment variable.
        self.api_key = GEMINI_CREDENTIAL
        self.ctx_len = ctx_len

    def interact(self, messages: str, **kwargs):
        # require a json_format, temperature, e.t.c

        json_format = kwargs.get("json_format")
        temperature = kwargs.get("temperature")

        messages = [
            {"role": "user", "content": messages}
        ] if isinstance(messages, str) else messages

        if json_format is None:
            raise ValueError("json_format is required")
        if temperature is None:
            raise ValueError("temperature is required")

        # Parse the messages
        system_message = [message['content'] for message in messages if message["role"] == "system"][0]
        messages = [{
            "role": "user",
            "parts": [
                {"text": message["content"]}
            ]
        } for message in messages if message["role"] != "system"]
        
        # call the API and return the response.
        client = genai.Client(api_key=self.api_key)
        prompt_input = {
            "model": "gemini-1.5-flash-002",
            "contents": messages,
            "config": types.GenerateContentConfig(
                response_mime_type='application/json',
                response_schema=json_format,
                temperature=temperature,
                system_instruction=system_message,
                max_output_tokens=self.ctx_len
            )
        }
        

        response = client.models.generate_content(
            **prompt_input
        )
        '''
        We nee to be VERY CAREFUL when handling Gemini. The responses from it
        may not be auto parsed or may not be in a valid JSON format, even if 
        we ask Gemini to geenrate a JSON response. When calling interact, we 
        should probably try some error catching mechanisms.
        '''
        parse_success = False
        trial_count = 0
        while not parse_success and trial_count < 5:
            try:
                # Extract the parsed response directly
                parsed_response = response.to_json_dict()['parsed']
                parse_success = True
            except KeyError:
                # There are chances Gemini does not give parsed response
                parsed_response = response.to_json_dict()['candidates'][0]['content']['parts'][0]['text']
                try:
                    # Try load from natural language. This happens quite a lot, the text is JSON but Gemini just does not parse it.
                    # There is still a chance that text is not a valid JSON
                    parsed_response = json.loads(parsed_response)
                    parse_success = True
                except json.JSONDecodeError:
                    # If it fails, try again
                    # Return None if it fails 3 times
                    parsed_response = None
                    trial_count += 1
                    continue
        # if parsed_response is None:
        #     print(response.to_json_dict())
        return parsed_response