import numpy as np
from nltk.corpus import wordnet as wn
from nltk.corpus import wordnet_ic

def create_row(tau, data, module):
    '''
    Data = List(List(Tuple(iou, Concept)))
    '''
    all_concepts = sum([[c for v,c in u] for u in data], [])
    all_values = sum([[v for v,c in u] for u in data], [])
    unique_concepts = set(all_concepts)
    return {
        'Module': module,
        'Units': len(data),
        'Unique concepts': len(unique_concepts),
        'Leaves': len([c for c in unique_concepts if c.is_leaf()]),
        'Non-leaves': len([c for c in unique_concepts if not c.is_leaf()]),
        'Total concepts': len(all_concepts),
        'Size': np.average([len(e) for e in data]),
        'Size if any': np.average([len(e) for e in data if len(e)]),
        'Depth': np.average([c.depth for c in unique_concepts]),
        'Sigma': np.average(all_values),
        'Threshold': tau
    }

def transpose_dict(ld, main='Module'):
    header = [d[main] for d in ld]
    new_d = {main:[k for k in ld[0]], **{k:[] for k in header}}
    
    for d in ld:
        mod = d[main]
        for k in d:
            if isinstance(d[k], float):
                if d[k] < 1:
                    new_d[mod].append("{:.2e}".format(d[k]))
                else:
                    new_d[mod].append("%.2f" % d[k])
            else:
                new_d[mod].append(d[k])
                
    return new_d


def build_id(ss):
    return 10**8 + ss.offset()

def check_relatedness(a, b):
    '''
    a, b: Concepts
    
    Checks if a and b are
    related by an IsA or a PartOf
    semantical relation.
    '''
    a_cid, b_cid = a.id, b.id
    a_synset = a.synset
    b_lineage = b.lineage
    
    a_meronyms = {build_id(e) for e in a_synset.part_meronyms()}
    a_holonyms = {build_id(e) for e in a_synset.part_holonyms()}
    
    sim = int(bool(a_cid != b_cid and (a_cid in b_lineage or
                                       a_meronyms & b_lineage or
                                       a_holonyms & b_lineage)))
    
    return sim


def ssc(a, b, f, *p):
    '''
    Synset call.
    
    Calls the function f
    on a.synset and b.synset.
    '''
    return f(a.synset,
             b.synset, *p)
    
brown_ic = wordnet_ic.ic('ic-brown.dat')

sim_functions = {
    'jcn': lambda a, b: ssc(a, b, wn.jcn_similarity, brown_ic),
    'lch': lambda a, b: ssc(a, b, wn.lch_similarity),
    'lin': lambda a, b: ssc(a, b, wn.lin_similarity, brown_ic),
    'pth': lambda a, b: ssc(a, b, wn.path_similarity),
    'res': lambda a, b: ssc(a, b, wn.res_similarity, brown_ic),
    'wup': lambda a, b: ssc(a, b, wn.wup_similarity)
}