import random
import tenacity
import json

from swegraft.seed import Seed
from swegraft.runtime.swalm import SwalmRuntime
from swegraft.strategy.migrate.agents.test import TestGenAgent
from swegraft.strategy.migrate.agents.issue import IssueImplementAgent
from swegraft.strategy.migrate.models import MigrateTaskSpec
from swegraft.strategy.migrate.constants import (
    MIGRATE_INSTRUCTION,
    ISSUE_DIRECT,
    ISSUE_ABSTRACT,
    MIGRATE_POSSIBILITY,
    TEST_GEN_INSTRUCTION,
)
from swegraft.utils.llm import make_llm_from_provider_config, LLMClient
import logging
from swalm.core.utils.log import setup_logging
import datetime

setup_logging(
    log_level=logging.WARN,
    debug_file=f"migrate_{datetime.datetime.now().strftime('%Y%m%d_%H%M%S')}.log",
)


async def abstract_bug_description(
    llm_client: LLMClient, prompt: str, model: str, max_tokens: int, temperature: float
) -> str:
    response = await llm_client.chat_completion_async(
        model=model,
        messages=[{"role": "user", "content": prompt}],
        max_tokens=max_tokens,
        temperature=temperature,
    )
    if response is None:
        return
    response = response.choices[0].message.content
    return response.split("```md")[1].split("```")[0]


@tenacity.retry(
    stop=tenacity.stop_after_attempt(10),
    wait=tenacity.wait_exponential(multiplier=1, min=5, max=60),
)
async def migrate_possibility(
    spec: MigrateTaskSpec,
    readme: str,
    tests: list[str],
    possibility_config: dict,
    provider_config: dict,
    model: str,
) -> str:
    prompt = MIGRATE_POSSIBILITY.format(
        body=spec.description,
        diff=spec.patch,
        readme=readme,
        test_suite="\n\t".join(tests),
    )
    llm_client = make_llm_from_provider_config(provider_config)
    response = await llm_client.chat_completion_async(
        messages=[{"role": "user", "content": prompt}],
        model=model,
        max_tokens=possibility_config.get("max_tokens", 1024),
        temperature=possibility_config.get("temperature", 0.0),
    )
    response = response.choices[0].message.content
    print(response)
    try:
        possibility = eval(response.split("```python")[1].split("```")[0].strip())
        possibility, reason = possibility
    except Exception as e:
        print(f"Error parse possibility: {e}")
        return False, "Error parse possibility"
    return possibility, reason


async def get_migrate_instruction(
    spec: MigrateTaskSpec, seed: Seed, rewrite_config: dict, llm_config: dict
) -> str:
    if not rewrite_config["enable"]:
        description = ISSUE_DIRECT.format(body=spec.description, diff=spec.patch)
    else:
        prompt = ISSUE_ABSTRACT.format(body=spec.description, diff=spec.patch)
        description = await abstract_bug_description(
            prompt,
            model=random.choice(rewrite_config["models"]),
            temperature=rewrite_config["temperature"],
            max_tokens=rewrite_config["max_tokens"],
            n=1,
            api_base=llm_config["base_url"],
            api_version=llm_config["api_version"],
            api_key=random.choice(llm_config["api_keys"]),
        )
    return MIGRATE_INSTRUCTION.format(
        language=seed.language,
        description=description,
        workspace_dir_name=seed.workspace,
    )


async def get_test_gen_instruction(
    seed: Seed,
    issue: str,
) -> str:
    prompt = TEST_GEN_INSTRUCTION.format(
        workspace_dir_name=seed.workspace,
        language=seed.language,
        issue=issue,
    )
    return prompt


async def run_migrate_task_agentless(
    spec: MigrateTaskSpec,
    seed: Seed,
    model_to_provider: dict,
    rewrite_config: dict,
    testgen_config: dict,
    issue_impl_config: dict,
) -> list[dict] | None:
    if not rewrite_config["enable"]:
        issue = ISSUE_DIRECT.format(body=spec.description, diff=spec.patch)
    else:
        prompt = ISSUE_ABSTRACT.format(body=spec.description, diff=spec.patch)
        model = random.choice(rewrite_config["models"])
        provider_config = random.choice(model_to_provider[model])
        llm_client = make_llm_from_provider_config(provider_config)
        issue = await abstract_bug_description(
            llm_client,
            prompt,
            model=random.choice(rewrite_config["models"]),
            max_tokens=rewrite_config["max_tokens"],
            temperature=rewrite_config["temperature"],
        )

    testgen_model = random.choice(testgen_config["models"])
    issue_impl_model = random.choice(issue_impl_config["models"])
    testgen_provider_config = random.choice(model_to_provider[testgen_model])
    issue_impl_provider_config = random.choice(model_to_provider[issue_impl_model])
    testgen_llm_client = make_llm_from_provider_config(testgen_provider_config)
    issue_impl_llm_client = make_llm_from_provider_config(issue_impl_provider_config)
    async with SwalmRuntime() as runtime:
        testgen_agent = TestGenAgent(
            model=testgen_model,
            llm_client=testgen_llm_client,
            swalm_run_time=runtime,
            config=testgen_config,
        )
        issue_implement_agent = IssueImplementAgent(
            model=issue_impl_model,
            llm_client=issue_impl_llm_client,
            config=issue_impl_config,
            swalm_run_time=runtime,
        )
        try:
            testgen_result = await testgen_agent.run(seed, issue)
            if testgen_result is None:
                return None
            imple_result = await issue_implement_agent.run(
                seed, issue, testgen_result["patch"], testgen_result["run_result"]
            )
            if imple_result is None:
                return None
            testgen_result["run_result"] = json.loads(
                testgen_result["run_result"].json()
            )
            imple_result["run_result"] = json.loads(imple_result["run_result"].json())
            return {
                "testgen": testgen_result,
                "issue_implement": imple_result,
            }
        except Exception as e:
            import traceback

            traceback.print_exc()
            print(f"Error run migrate: {e}")
            return None
