import re
from collections import OrderedDict
import subprocess
import pathlib
import shutil
import uuid
from swegraft.utils.repo import workspace
from swegraft.seed import Seed

# HOME
PLAYGROUND_DIR = pathlib.Path.home() / ".swegraft_playground"


def extract_md_blocks(text, langauge: str):
    if langauge == "js":
        langauge = "javascript"
    pattern = f"```{langauge}\n(.*?)\n```"
    matches = re.findall(pattern, text, re.DOTALL)
    return matches


def split_edit_multifile_commands(commands: list[str]) -> dict[str, list[str]]:
    """Split commands based on edited files."""
    file_to_commands = OrderedDict()
    for command in commands:
        file_name = None
        for subcommand in command.split(">>>>>>> REPLACE")[:-1]:
            subcommand = subcommand.strip()
            if "<<<<<<< SEARCH" in subcommand:
                fn = subcommand.split("<<<<<<< SEARCH")[0].lstrip("#").strip()
                if fn:
                    file_name = fn

            if len(subcommand.split("<<<<<<< SEARCH")) != 2:
                continue
            converted_command = (
                "<<<<<<< SEARCH"
                + subcommand.split("<<<<<<< SEARCH")[1]
                + "\n"
                + ">>>>>>> REPLACE"
            )
            # deduplicate
            if file_name is not None and (
                file_name not in file_to_commands
                or converted_command not in file_to_commands[file_name]
            ):
                file_to_commands.setdefault(file_name, []).append(converted_command)
    return file_to_commands


def apply_diff_edit_commands(file: pathlib.Path, commands: list[str]):
    # apply the edits from the end of file to the beginning of file
    # this is to make sure context is correct
    # since we want to replace the original context, let's first check for all edits.
    can_apply = []
    content = file.read_text()
    for subcommand in commands:
        if not subcommand.startswith("<<<<<<< SEARCH") and subcommand.endswith(
            ">>>>>>> REPLACE"
        ):
            continue

        subcommand = "\n".join(subcommand.splitlines()[1:-1])
        if len(subcommand.split("\n=======\n")) != 2:
            continue

        original, replace = subcommand.split("\n=======\n")

        # if original in content:
        # change to count the number of occurences
        if content.count(original) != 1:
            continue
        can_apply.append((original, replace))

    for original, replace in can_apply:
        content = content.replace(original, replace)
    file.write_text(content)


def _post_process_multifile_repair(
    output: str, seed: Seed, pre_patch: str | None = None
) -> str:
    edit_multifile_commands = extract_md_blocks(output, seed.language)
    file_to_commands = split_edit_multifile_commands(edit_multifile_commands)

    with workspace(seed, "convert") as ws:
        # apply the pre_patch if it exists
        if pre_patch:
            fake_git_apply(pre_patch, ws)
            subprocess.run(
                ["git", "add", "."],
                cwd=ws,
                capture_output=True,
            )
            subprocess.run(
                ["git", "commit", "-m", "pre_patch"],
                cwd=ws,
                capture_output=True,
            )
        # apply the edits
        for edited_file, edit_commands in file_to_commands.items():
            edited_file = ws / edited_file
            if not edited_file.exists():
                continue
            apply_diff_edit_commands(edited_file, edit_commands)
        # run git diff
        patch = subprocess.run(
            ["git", "diff", "."],
            cwd=ws,
            capture_output=True,
        )
        patch = patch.stdout.decode("utf-8")
        return patch


def show_project_structure(
    structure: dict,
    level: int = 0,
    indentation: int = 4,
) -> str:
    """pprint the project structure, with randomization option"""

    pp_string = ""

    items = list(structure.items())
    for key, value in items:
        pp_string += " " * (level * indentation) + str(key) + "\n"
        if (
            isinstance(value, dict) and "classes" not in value
        ):  # Check if value is a dictionary
            pp_string += show_project_structure(value, level + 1, indentation)

    return pp_string


def fake_git_diff(
    edited_files: list[str],
    old_contents: list[str],
    new_contents: list[str],
) -> str:
    """create a fake git repo to obtain git diff format"""
    ws = PLAYGROUND_DIR / str(uuid.uuid4())
    ws.mkdir(parents=True, exist_ok=True)
    # create a fake git repo
    subprocess.run(
        ["git", "init"],
        cwd=ws,
        capture_output=True,
    )

    for file_path, old_content, new_content in zip(
        edited_files, old_contents, new_contents
    ):
        # create a file
        file_path = ws / file_path
        file_path.parent.mkdir(parents=True, exist_ok=True)
        file_path.write_text(old_content)
        subprocess.run(
            ["git", "add", file_path],
            cwd=ws,
            capture_output=True,
        )
        subprocess.run(
            ["git", "commit", "-m", "initial commit"],
            cwd=ws,
            capture_output=True,
        )

    for file_path, old_content, new_content in zip(
        edited_files, old_contents, new_contents
    ):
        file_path = ws / file_path
        file_path.write_text(new_content)
    o = subprocess.run(
        ["git", "diff", "."],
        cwd=ws,
        capture_output=True,
    )
    s = o.stdout.decode("utf-8")
    shutil.rmtree(ws)
    return s


def fake_git_apply(patch: str, ws: pathlib.Path):
    temp_patch = ws / "temp.patch"
    temp_patch.write_text(patch)
    subprocess.run(
        ["git", "apply", temp_patch],
        cwd=ws,
        capture_output=True,
    )
    temp_patch.unlink()


def remove_empty_lines(code: str) -> str:
    # Split the code into lines
    lines = code.splitlines()
    # Remove empty lines
    filtered_lines = [line for line in lines if line.strip() != ""]
    return "\n".join(filtered_lines)
