import pathlib
from pathlib import Path
import uuid
import os
import json
import random
from termcolor import cprint
import asyncio
import tqdm
from swegraft.seed import Seed
from swegraft.strategy import SynStrategy
from swegraft.runtime.swalm import SwalmRuntime, get_swalm_image
from swegraft.collect.pulls import collect_pulls, PULL_DIR
from swegraft.utils.github import get_repo
from swegraft.utils.common import wait_coros_with_progress
import concurrent.futures
from swegraft.strategy.migrate.github_utils import (
    RelevanceLevel,
    find_similar_repos,
    get_repo_readme,
    SKIP_REPOS,
)
from swegraft.strategy.migrate.constants import (
    RELATIVE_REPO_FILE,
    STRATEGY,
    POSSIBILITY_FILE,
)
from swegraft.strategy.constants import TASK_FILE, PATCHES_DIR
from swegraft.strategy.migrate.agent import (
    run_migrate_task_agentless,
    migrate_possibility,
)
from multi_swe_bench.harness.test_result import TestResult
from swegraft.strategy.migrate.models import MigrateTaskSpec
from collections import defaultdict
from swegraft.strategy.migrate.agentless import (
    abstract_issue,
    testgen_localize,
    testgen_edits,
    save_repo_structure,
    convert_testgen_edits_to_patches,
    testgen_patch_run,
    issue_implement_localize,
    issue_implement_edits,
    convert_issue_implement_edits_to_patches,
    issue_implement_patch_run,
)


def save_result(result, result_file: pathlib.Path, indent=4):
    result: dict = json.loads(result.json())
    with open(result_file, "w") as f:
        f.write(json.dumps(result, indent=indent))


async def run(
    seed: Seed,
    result_dir: pathlib.Path,
    runtime: SwalmRuntime,
):
    if (result_dir / "run.log").exists():
        return
    result_dir.mkdir(parents=True, exist_ok=True)
    async with runtime.session(
        image=get_swalm_image(seed.source, seed.instance_id)
    ) as session:
        run_log = await seed.run(session)
        with open(result_dir / "run.log", "w") as f:
            f.write(run_log if run_log else "")
        save_result(seed.parse_log(run_log), result_dir / "run.json")


@SynStrategy.register(STRATEGY)
class Migrate(SynStrategy):
    def __init__(self, seed: Seed, config, result_dir: Path):
        super().__init__(name=STRATEGY, seed=seed, config=config, result_dir=result_dir)
        self.model_to_provider = defaultdict(list)
        for provider in self.config["llm"]:
            for model in provider["supported_models"]:
                self.model_to_provider[model].append(provider)
        self.strategy_config = self.config["strategy"][self.name]
        self.agent_config = self.strategy_config["agent"]
        self.collect_config = self.strategy_config["collect"]
        self.possibility_config = self.collect_config["possibility"]

        self.patches_dir = self.seed_result_dir / PATCHES_DIR

    async def _collect_repos(self) -> list[str]:
        min_stars = self.collect_config["repo"]["min_stars"]
        relative_repo_file = self.seed_result_dir / RELATIVE_REPO_FILE

        if (
            relative_repo_file.exists()
            and relative_repo_file.read_text().strip() != ""
            and not self.collect_config["recollect"]
        ):
            cprint(f"Loading relative repos from {relative_repo_file}", "light_cyan")
            repos = [
                json.loads(line) for line in relative_repo_file.read_text().splitlines()
            ]
            repos = [
                repo
                for repo in repos
                if repo["full_name"] not in SKIP_REPOS
                and repo["full_name"].lower() not in SKIP_REPOS
            ]
            repos = [
                repo
                for repo in repos
                if repo["full_name"].lower() != self.seed.repo.lower()
            ]
            return repos

        cprint(f"Collecting relative repos for {self.seed.repo}", "light_cyan")
        model = random.choice(self.collect_config["repo"]["models"])
        provider_config = random.choice(self.model_to_provider[model])

        # get seed repo
        seed_repo = await get_repo(self.seed.repo)
        relative_repos = await find_similar_repos(
            seed_repo=seed_repo,
            min_stars=min_stars,
            min_relevance_level=self.collect_config["repo"]["min_relevance_level"],
            provider_config=provider_config,
            model=model,
            concurrency=self.collect_config["repo"]["concurrency"],
        )
        # relevance statics
        relevance_stats = {}
        for repo in relative_repos:
            relevance_stats[repo["relevance_level"]] = (
                relevance_stats.get(repo["relevance_level"], 0) + 1
            )
        for relevance_level, count in relevance_stats.items():
            cprint(
                f"Relevance level {RelevanceLevel(relevance_level).name}: {count}",
                "light_cyan",
            )
        cprint(
            f"Saving {len(relative_repos)} repos to {relative_repo_file}", "light_green"
        )
        with open(relative_repo_file, "w") as f:
            for repo in relative_repos:
                f.write(json.dumps(repo) + "\n")
        return relative_repos

    def _collect_specs(
        self, repo2issues: dict[str, list[dict]]
    ) -> list[MigrateTaskSpec]:
        specs = []
        with open(self.seed_result_dir / "specs.jsonl", "w") as f:
            for repo, issues in repo2issues.items():
                for issue in issues:
                    spec = self._make_migrate_spec(repo, issue)
                    specs.append(spec)
                    f.write(spec.model_dump_json() + "\n")
        return specs

    def _make_migrate_spec(self, repo: str, pull: dict) -> MigrateTaskSpec:
        descipription = f"""\
## Pull Request 
{pull["title"]}
{pull["body"]}

## Issues
"""
        for issue in pull["issues"]:
            if issue is None:
                cprint(f"Issue is None for pull {pull['number']}", "red")
                continue
            descipription += f"""\
## Issue {issue["number"]}
{issue["title"]}
{issue["body"]}
"""
        task_id = f"migrate--{self.seed.repo.replace('/', '__')}"
        task_id += f"--{repo.replace('/', '__')}"
        task_id += f"--{pull['number']}"
        return MigrateTaskSpec(
            task_id=task_id,
            description=descipription,
            patch=pull["patch"],
        )

    async def _run_migrate(self, spec: MigrateTaskSpec, semaphore: asyncio.Semaphore):
        result_dir = self.patches_dir / spec.task_id
        result_dir.mkdir(parents=True, exist_ok=True)
        (result_dir / "spec.json").write_text(spec.model_dump_json(indent=4))
        result_file = result_dir / "migrate_result.json"
        possibility_file = result_dir / "possibility.jsonl"
        if possibility_file.exists():
            possibility = json.loads(possibility_file.read_text())
            skip = not possibility["possibility"]
            reason = possibility["reason"]
            if skip:
                cprint(
                    f"Skipping {spec.task_id} because it is not possible to migrate, reason: {reason}",
                    "yellow",
                )
                return None
        # for backward compatibility
        bug_patch_file = result_dir / "bug.patch"
        if bug_patch_file.exists():
            if bug_patch_file.read_text():
                cprint(
                    f"Skipping {spec.task_id} because it already has a patch", "yellow"
                )
        if result_file.exists():
            if result_file.read_text():
                cprint(
                    f"Skipping {spec.task_id} because it already has a result", "yellow"
                )
                return None
        async with semaphore:
            try:
                result = await run_migrate_task_agentless(
                    spec,
                    self.seed,
                    self.model_to_provider,
                    self.strategy_config["rewrite"],
                    self.strategy_config["testgen"],
                    self.strategy_config["buggen"],
                )
            except Exception as e:
                import traceback

                traceback.print_exc()
                cprint(f"Error running migrate task: {e}", "red")
                return
        if result is None:
            return
        with open(result_file, "w") as f:
            f.write(json.dumps(result, indent=4))

    async def _collect_pulls(self, repos: list[dict]) -> dict[str, list[dict]]:
        repo2pull_file = {}
        repo2pulls = {}
        coros = []
        pull_semaphore = asyncio.Semaphore(self.config["pull"]["concurrency"])
        repo_semaphore = asyncio.Semaphore(20)

        async def _collect(language: str, repo: dict):
            async with repo_semaphore:
                await collect_pulls(language, repo, pull_semaphore)

        for repo in repos:
            name = repo["full_name"]
            if repo["full_name"] is None:
                name = f"{repo['owner']['login']}/{repo['name']}"
            language = repo["language"]
            if language is None:
                language = "unknown"
            language = language.lower()
            pull_result_file = (
                PULL_DIR / language / f"{repo['full_name'].replace('/', '__')}.jsonl"
            )
            repo2pull_file[name] = pull_result_file
            coros.append(_collect(language, repo))
        await wait_coros_with_progress(coros, "Collecting pulls")
        repo2pulls = {}
        for name, pull_file in repo2pull_file.items():
            with open(pull_file, "r") as f:
                repo2pulls[name] = [json.loads(line) for line in f]
        return repo2pulls

    async def collect(self) -> list[MigrateTaskSpec]:
        repos = await self._collect_repos()
        repo2pulls = await self._collect_pulls(repos)
        specs = [
            self._make_migrate_spec(repo, pull)
            for repo, pulls in repo2pulls.items()
            for pull in pulls
        ]
        for spec in specs:
            spec_file = self.patches_dir / spec.task_id / "spec.json"
            spec_file.parent.mkdir(parents=True, exist_ok=True)
            spec_file.write_text(spec.model_dump_json())
        print(f"Collected {len(specs)} specs")


        async with SwalmRuntime() as runtime:
            await run(self.seed, self.seed_result_dir, runtime)
        concurrency = self.possibility_config["concurrency"]
        semaphore = asyncio.Semaphore(concurrency)

        if len(specs) > 5000:
            specs = random.sample(specs, 5000)
        async def _measure(spec: MigrateTaskSpec):
            async with semaphore:
                await self.measure_migrate_possibility(spec)

        coros = [_measure(spec) for spec in specs]
        await wait_coros_with_progress(coros, "Measuring migrate possibility")
        for spec in specs:
            spec_file = self.patches_dir / spec.task_id / "spec.json"
            possibility_file = spec_file.parent / "possibility.jsonl"
            if not possibility_file.exists():
                continue
            possibility = json.loads(possibility_file.read_text())
            if not possibility["possibility"]:
                cprint(
                    f"Skipping {spec.task_id} because it is not possible to migrate, reason: {possibility['reason']}",
                    "yellow",
                )
                continue

    async def measure_migrate_possibility(self, spec: MigrateTaskSpec) -> bool:
        result_dir = self.patches_dir / spec.task_id
        possibility_file = result_dir / POSSIBILITY_FILE
        if possibility_file.exists():
            return
        readme_file = self.seed_result_dir / "README"
        if not readme_file.exists():
            readme = await get_repo_readme(self.seed.repo)
            readme_file.write_text(readme)
        readme = readme_file.read_text()
        run_result = self.seed_result_dir / "run.json"
        with open(run_result, "r") as f:
            run_result = json.load(f)
        tests = list(
            set(
                run_result["passed_tests"]
                + run_result["failed_tests"]
                + run_result["skipped_tests"]
            )
        )
        model = random.choice(self.possibility_config["models"])
        provider_config = random.choice(self.model_to_provider[model])
        possibility, reason = await migrate_possibility(
            spec, readme, tests, self.possibility_config, provider_config, model
        )
        with open(possibility_file, "w") as f:
            f.write(
                json.dumps({"possibility": possibility, "reason": reason}, indent=4)
            )

    async def _run_issue_abstract(
        self, spec: MigrateTaskSpec, semaphore: asyncio.Semaphore
    ):
        result_dir = self.patches_dir / spec.task_id
        model = random.choice(self.strategy_config["rewrite"]["models"])
        provider_config = random.choice(self.model_to_provider[model])
        async with semaphore:
            await abstract_issue(
                spec,
                provider_config,
                model,
                self.strategy_config["rewrite"],
                result_dir,
            )

    async def _run_testgen_localize(
        self, spec: MigrateTaskSpec, semaphore: asyncio.Semaphore
    ):
        result_dir = self.patches_dir / spec.task_id
        async with semaphore:
            await testgen_localize(
                model_to_provider=self.model_to_provider,
                rewrite_issue=self.strategy_config["rewrite"]["enable"],
                testgen_config=self.strategy_config["testgen"],
                result_dir=result_dir,
            )

    async def _run_testgen_edits(
        self, spec: MigrateTaskSpec, semaphore: asyncio.Semaphore
    ):
        result_dir = self.patches_dir / spec.task_id
        model = random.choice(self.strategy_config["testgen"]["models"])
        provider_config = random.choice(self.model_to_provider[model])
        async with semaphore:
            await testgen_edits(
                seed=self.seed,
                provider_config=provider_config,
                model=model,
                rewrite_issue=self.strategy_config["rewrite"]["enable"],
                testgen_config=self.strategy_config["testgen"],
                result_dir=result_dir,
            )

    async def _run_testgen_patch_run(
        self,
        runtime: SwalmRuntime,
        spec: MigrateTaskSpec,
        patch_result: dict,
        semaphore: asyncio.Semaphore,
    ):
        result_dir = self.patches_dir / spec.task_id
        run_result_file = self.seed_result_dir / "run.json"
        run_result = TestResult.from_dict(json.loads(run_result_file.read_text()))
        async with semaphore:
            await testgen_patch_run(
                seed=self.seed,
                runtime=runtime,
                patch_result=patch_result,
                result_dir=result_dir,
                run_result=run_result,
            )

    def _convert_testgen_response_to_patches(self, spec: MigrateTaskSpec):
        result_dir = self.patches_dir / spec.task_id
        localize_result_file = result_dir / "testgen_localize.json"
        if not localize_result_file.exists():
            return
        localize_results = json.loads(localize_result_file.read_text())
        convert_testgen_edits_to_patches(self.seed, localize_results, result_dir)

    def _find_valid_testgen_patches(self, spec: MigrateTaskSpec) -> list[dict]:
        result_dir = self.patches_dir / spec.task_id
        patches_file = result_dir / "testgen_patches.json"
        if not patches_file.exists():
            return []
        patches = json.loads(patches_file.read_text())
        ret = []
        for patch in patches:
            testgen_run_result_file = (
                result_dir
                / f"testgen_run_{patch['localize_index']}_{patch['edit_index']}.json"
            )
            if (
                not testgen_run_result_file.exists()
                or testgen_run_result_file.read_text().strip() == ""
            ):
                continue
            run_result = json.loads(testgen_run_result_file.read_text())
            if not run_result["new_tests"]:
                continue
            patch["new_tests"] = run_result["new_tests"]
            patch["spec"] = spec
            ret.append(patch)
        return ret

    async def _run_issue_implement_localize(
        self,
        testgen_patch: dict,
        semaphore: asyncio.Semaphore,
    ):
        spec: MigrateTaskSpec = testgen_patch["spec"]
        result_dir = self.patches_dir / spec.task_id
        async with semaphore:
            await issue_implement_localize(
                self.seed,
                self.model_to_provider,
                testgen_patch,
                self.strategy_config["rewrite"]["enable"],
                result_dir,
                self.strategy_config["buggen"],
            )

    async def _run_issue_implement_edit(
        self,
        testgen_patch: dict,
        semaphore: asyncio.Semaphore,
    ):
        spec: MigrateTaskSpec = testgen_patch["spec"]
        result_dir = self.patches_dir / spec.task_id
        model = random.choice(self.strategy_config["buggen"]["models"])
        provider_config = random.choice(self.model_to_provider[model])
        async with semaphore:
            await issue_implement_edits(
                self.seed,
                provider_config,
                model,
                self.strategy_config["rewrite"]["enable"],
                self.strategy_config["buggen"],
                testgen_patch,
                result_dir,
            )

    def _convert_issue_implement_response_to_patches(self, testgen_patch: dict):
        spec: MigrateTaskSpec = testgen_patch["spec"]
        result_dir = self.patches_dir / spec.task_id
        localize_result_file = (
            result_dir
            / f"issue_implement_localize_{testgen_patch['localize_index']}_{testgen_patch['edit_index']}.json"
        )
        if not localize_result_file.exists():
            return
        localize_results = json.loads(localize_result_file.read_text())
        convert_issue_implement_edits_to_patches(
            self.seed, testgen_patch, localize_results, result_dir
        )

    async def _run_issue_implement_patch_run(
        self,
        runtime: SwalmRuntime,
        testgen_patch: dict,
        issue_implement_patch: dict,
        run_result: TestResult,
        semaphore: asyncio.Semaphore,
    ):
        result_dir = self.patches_dir / testgen_patch["spec"].task_id
        result_file = (
            result_dir
            / f"issue_implement_run_{testgen_patch['localize_index']}_{testgen_patch['edit_index']}_{issue_implement_patch['localize_index']}_{issue_implement_patch['edit_index']}.json"
        )
        if result_file.exists():
            return
        async with semaphore:
            await issue_implement_patch_run(
                self.seed,
                runtime,
                testgen_patch,
                issue_implement_patch,
                result_dir,
                run_result,
            )

    def _gather_results(self, testgen_patch: dict):
        result_dir = self.patches_dir / testgen_patch["spec"].task_id
        patches_file = (
            result_dir
            / f"issue_implement_patches_{testgen_patch['localize_index']}_{testgen_patch['edit_index']}.json"
        )
        patches = json.loads(patches_file.read_text())
        ret = []
        for patch in patches:
            if patch["patch"].strip() == "":
                continue
            run_result_file = (
                result_dir
                / f"issue_implement_run_{testgen_patch['localize_index']}_{testgen_patch['edit_index']}_{patch['localize_index']}_{patch['edit_index']}.json"
            )
            if not run_result_file.exists():
                continue
            with open(run_result_file, "r") as f:
                run_result = json.load(f)
            if not run_result["issue_implement_run_log"]:
                continue
            ret.append(
                dict(
                    testgen_patch=testgen_patch["patch"],
                    testgen_run_result=testgen_patch["run_result"],
                    testgen_run_log=testgen_patch["run_log"],
                    issue_implement_patch=patch,
                    issue_implement_run_result=testgen_patch[
                        "issue_implement_run_result"
                    ],
                    issue_implement_run_log=run_result["issue_implement_run_log"],
                )
            )
            ret.append(patch)
        return ret

    def _gather_testgen_results(self, spec: MigrateTaskSpec):
        result_dir = self.patches_dir / spec.task_id
        patches_file = result_dir / "testgen_patches.json"
        if not patches_file.exists():
            return []
        patches = json.loads(patches_file.read_text())
        ret = []
        for patch in patches:
            testgen_run_result_file = (
                result_dir
                / f"testgen_run_{patch['localize_index']}_{patch['edit_index']}.json"
            )
            if (
                not testgen_run_result_file.exists()
                or testgen_run_result_file.read_text().strip() == ""
            ):
                continue
            run_result = json.loads(testgen_run_result_file.read_text())
            ret.append(
                dict(
                    localize_index=patch["localize_index"],
                    edit_index=patch["edit_index"],
                    patch=patch,
                    run_result=run_result["test_run_result"],
                    run_log=run_result["test_run_log"],
                    new_tests=run_result["new_tests"],
                )
            )
        return ret

    def _gather_issue_implement_results(
        self, spec: MigrateTaskSpec, testgen_result: dict
    ):
        result_dir = self.patches_dir / spec.task_id
        patches_file = (
            result_dir
            / f"issue_implement_patches_{testgen_result['localize_index']}_{testgen_result['edit_index']}.json"
        )
        if not patches_file.exists():
            return []
        patches = json.loads(patches_file.read_text())
        ret = []
        for patch in patches:
            run_result_file = (
                result_dir
                / f"issue_implement_run_{testgen_result['localize_index']}_{testgen_result['edit_index']}_{patch['localize_index']}_{patch['edit_index']}.json"
            )
            if (
                not run_result_file.exists()
                or run_result_file.read_text().strip() == ""
            ):
                continue
            try:
                run_result = json.loads(run_result_file.read_text())
            except Exception as _:
                # due to process exit or dumps error, the file is not complete
                continue
            if not run_result["issue_implement_broken_tests"]:
                continue
            ret.append(
                dict(
                    patch=patch,
                    run_result=run_result["issue_implement_run_result"],
                    run_log=run_result["issue_implement_run_log"],
                    broken_tests=run_result["issue_implement_broken_tests"],
                )
            )
        # drop index
        testgen_result.pop("localize_index")
        testgen_result.pop("edit_index")
        return [
            dict(
                testgen=testgen_result,
                issue_implement=result,
            )
            for result in ret
        ]

    def gather_results(self, spec: MigrateTaskSpec):
        result_dir = self.patches_dir / spec.task_id
        # gather
        testgen_patches = self._gather_testgen_results(spec)
        ret = []
        for testgen_patch in testgen_patches:
            ret.extend(self._gather_issue_implement_results(spec, testgen_patch))
        for result in ret:
            result["instance_id"] = spec.task_id + "_" + str(uuid.uuid4())[:4]
        with open(result_dir / "success.jsonl", "w") as f:
            for result in ret:
                f.write(json.dumps(result) + "\n")
        return ret

    async def synthesize(self):
        specs: list[MigrateTaskSpec] = []
        patches_dir = self.patches_dir
        impossible_specs = 0
        unmeasured_specs = 0
        for subdir in tqdm.tqdm(patches_dir.iterdir(), desc="Loading specs"):
            if not subdir.is_dir():
                continue
            if not (subdir / TASK_FILE).exists():
                continue
            if not (subdir / POSSIBILITY_FILE).exists():
                unmeasured_specs += 1
                continue
            possibility = json.loads((subdir / POSSIBILITY_FILE).read_text())
            if not possibility["possibility"]:
                impossible_specs += 1
                continue
            spec = MigrateTaskSpec.model_validate_json((subdir / TASK_FILE).read_text())
            specs.append(spec)
        cprint(f"Possible specs: {len(specs)}", "light_cyan")
        cprint(f"Unmeasured specs: {unmeasured_specs}", "yellow")
        cprint(f"Impossible specs: {impossible_specs}", "red")
        save_repo_structure(self.seed, self.seed_result_dir)
        # Test Generation
        concurrency = self.strategy_config["rewrite"]["concurrency"]
        semaphore = asyncio.Semaphore(concurrency)
        specs = specs[:1000]
        coros = [self._run_issue_abstract(spec, semaphore) for spec in specs]
        await wait_coros_with_progress(coros, "Running Issue Abstract")
        concurrency = self.strategy_config["testgen"]["concurrency"]
        semaphore = asyncio.Semaphore(concurrency)
        coros = [self._run_testgen_localize(spec, semaphore) for spec in specs]
        await wait_coros_with_progress(coros, "Running TestGen Localize")
        coros = [self._run_testgen_edits(spec, semaphore) for spec in specs]
        await wait_coros_with_progress(coros, "Running TestGen Edit")
        with concurrent.futures.ProcessPoolExecutor(max_workers=12) as executor:
            futures = [
                executor.submit(self._convert_testgen_response_to_patches, spec)
                for spec in specs
            ]
            for future in tqdm.tqdm(
                concurrent.futures.as_completed(futures),
                total=len(specs),
                desc="Converting TestGen Response to Patches",
            ):
                future.result()
        cprint(f"Synthesized {len(specs)} specs", "light_cyan")
        concurrency = self.config["env"]["concurrency"]
        semaphore = asyncio.Semaphore(concurrency)
        tasks = []
        for spec in specs:
            result_dir = self.patches_dir / spec.task_id
            result_file = result_dir / "testgen_patches.json"
            if not result_file.exists():
                continue
            patch_results = json.loads(result_file.read_text())
            for patch_result in patch_results:
                tasks.append(
                    dict(
                        spec=spec,
                        patch_result=patch_result,
                    )
                )
        tasks = [task for task in tasks if task["patch_result"]["patch"].strip()]
        async with SwalmRuntime() as runtime:
            coros = [
                self._run_testgen_patch_run(
                    runtime, task["spec"], task["patch_result"], semaphore
                )
                for task in tasks
            ]
            await wait_coros_with_progress(coros, "Running TestGen Patch Run")
        valid_testgen_patches = sum(
            [self._find_valid_testgen_patches(spec) for spec in specs], []
        )
        # Dedup using new tests
        seen_new_tests = []
        testgen_patches = []
        for patch in valid_testgen_patches:
            # if have overlap than 50 %, skip
            if any(
                set(patch["new_tests"]) == set(new_test) for new_test in seen_new_tests
            ):
                continue
            seen_new_tests.append(patch["new_tests"])
            testgen_patches.append(patch)
        cprint(
            f"Found {len(testgen_patches)} testgen patches out of {len(specs)} issues",
            "light_cyan",
        )

        # Issue Implement for each valid testgen patch
        concurrency = self.strategy_config["buggen"]["concurrency"]
        semaphore = asyncio.Semaphore(concurrency)
        coros = [
            self._run_issue_implement_localize(patch, semaphore)
            for patch in testgen_patches
        ]
        await wait_coros_with_progress(coros, "Running Issue Implement Localize")
        coros = [
            self._run_issue_implement_edit(patch, semaphore)
            for patch in testgen_patches
        ]
        await wait_coros_with_progress(coros, "Running Issue Implement Edit")
        cpu_count = os.cpu_count()
        with concurrent.futures.ProcessPoolExecutor(
            max_workers=cpu_count // 2
        ) as executor:
            futures = [
                executor.submit(
                    self._convert_issue_implement_response_to_patches, patch
                )
                for patch in testgen_patches
            ]
            for future in tqdm.tqdm(
                concurrent.futures.as_completed(futures),
                total=len(testgen_patches),
                desc="Converting Issue Implement Response to Patches",
            ):
                future.result()
        concurrency = self.config["env"]["concurrency"]
        semaphore = asyncio.Semaphore(concurrency)
        tasks = []
        for testgen_patch in testgen_patches:
            result_dir = self.patches_dir / testgen_patch["spec"].task_id
            result_file = (
                result_dir
                / f"issue_implement_patches_{testgen_patch['localize_index']}_{testgen_patch['edit_index']}.json"
            )
            with open(result_file, "r") as f:
                patch_results = json.load(f)
            if not patch_results:
                continue
            seen_patches = set()
            for patch_result in patch_results:
                if patch_result["patch"].strip() == "":
                    continue
                if patch_result["patch"] in seen_patches:
                    continue
                seen_patches.add(patch_result["patch"])
                tasks.append(
                    dict(
                        testgen_patch=testgen_patch,
                        issue_implement_patch=patch_result,
                    )
                )

        run_result_file = self.seed_result_dir / "run.json"
        with open(run_result_file, "r") as f:
            run_result = TestResult.from_dict(json.load(f))
        async with SwalmRuntime() as runtime:
            coros = [
                self._run_issue_implement_patch_run(
                    runtime,
                    task["testgen_patch"],
                    task["issue_implement_patch"],
                    run_result,
                    semaphore,
                )
                for task in tasks
            ]
        await wait_coros_with_progress(coros, "Running Issue Implement Patch Run")
        count = 0
        success_specs = 0
        all_success_results = []
        for spec in specs:
            success_results = self.gather_results(spec)
            spec_count = len(success_results)
            if spec_count > 0:
                success_specs += 1
            count += spec_count
            all_success_results.extend(success_results)
        cprint(f"Successfully synthesized {count} patches", "light_cyan")
        cprint(f"Successfully synthesized {success_specs} specs", "light_cyan")
        cprint(
            f"Saving {len(all_success_results)} results to {self.seed_result_dir / 'success.jsonl'}",
            "light_cyan",
        )
        with open(self.seed_result_dir / "success.jsonl", "w") as f:
            for result in all_success_results:
                f.write(json.dumps(result) + "\n")
