# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Preprocess the GSM8k dataset to parquet format
"""

import os
import datasets
import json
import random

from verl.utils.hdfs_io import copy, makedirs
import argparse

from verl.utils.reward_score.math import remove_boxed, last_boxed_only_string

random.seed(42)

def extract_solution(solution_str):
    return remove_boxed(last_boxed_only_string(solution_str))

def instruction_following_gen():
        num_tokens = random.randint(100, 4000)
        return "\n\nLet's think step by step and output the final answer within \\boxed{}." + f" Think for {num_tokens} tokens.", num_tokens    
    

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--local_dir', default='~/STILL-3-Preview-RL-Data-HF')
    parser.add_argument('--hdfs_dir', default=None)
    output_dir = "~/Still3-l1"

    args = parser.parse_args()

    data_source = 'STILL-3-l1'
    print(f"Loading the {data_source} dataset from huggingface...", flush=True)
    
    train_dataset = datasets.load_dataset(args.local_dir, "default")['train']
    # add a row to each data item that represents a unique id
    def make_map_fn(split):

        def process_fn(example, idx):
            question_raw = example.pop('question')
            instruction_following, num_tokens = instruction_following_gen()
            question = question_raw + ' ' + instruction_following

            answer = example.pop('answer')
            data = {
                "data_source": data_source,
                "prompt": [{
                    "role": "user",
                    "content": question
                }],
                "ability": "math",
                "reward_model": {
                    "style": "rule",
                    "ground_truth": answer,
                    "ori_budget": num_tokens
                },
                "extra_info": {
                    'split': split,
                    'index': idx,
                    'answer': answer,
                    "question": question_raw,
                }
            }
            return data

        return process_fn

    train_dataset = train_dataset.map(function=make_map_fn('train'), with_indices=True)

    local_dir = args.local_dir
    hdfs_dir = args.hdfs_dir

    train_dataset.to_parquet(os.path.join(output_dir, 'train_l1.parquet'))
    if hdfs_dir is not None:
        makedirs(hdfs_dir)

        copy(src=local_dir, dst=hdfs_dir)
