import numpy as np
import matplotlib.pyplot as plt

def plot_data(data, 
              dimensions=3, 
              show_origin=False, 
              special_points=None,
              orthogonal_vector=None,
              show_unit_circle_sphere=True,
              title_3d='3D Plot of Data'):
    n_samples, n_features = data.shape

    if dimensions <= 2:
        # 2D Plot
        plt.figure(figsize=(8, 6))
        plt.scatter(data[:, 0], data[:, 1], c='b', marker='.', label='Data')
        if show_origin:
            plt.scatter(0, 0, c='r', marker='x', label='Origin')
        if show_unit_circle_sphere:
            circle = plt.Circle((0, 0), 1, color='orange', fill=False, linestyle='--', label='Unit Circle')
            plt.gca().add_artist(circle)
        plt.xlabel('Dimension 1')
        plt.ylabel('Dimension 2')
        plt.title('2D Plot of Data')
        plt.grid(True)
        plt.legend()
        if special_points is not None:
            plt.scatter(special_points[:, 0], special_points[:, 1], c='g', marker='x', s=100, label='Special Point')
            for point in special_points:
                plt.arrow(0, 0, point[0], point[1], head_width=0.02, head_length=0.05, fc='r', ec='r', label='Vector from Origin')
        if orthogonal_vector is not None:
            # Define the line corresponding to the orthogonal vector
            a, b = orthogonal_vector
            x_values = np.array([-1, 1])
            y_values = -(a * x_values) / b
            plt.plot(x_values, y_values, 'b--', label='Orthogonal Line')
    elif dimensions == 3:
        # 3D Plot
        fig = plt.figure(figsize=(10, 7))
        ax = fig.add_subplot(111, projection='3d')
        ax.scatter(data[:, 0], data[:, 1], data[:, 2], c='b', marker='.', label='Data')
        if show_origin:
            ax.scatter(0, 0, 0, c='r', marker='x', s=100, label='Origin')  # Larger size for visibility
        if show_unit_circle_sphere:
            # Create a mesh grid for the unit sphere
            u = np.linspace(0, 2 * np.pi, 100)
            v = np.linspace(0, np.pi, 100)
            x = np.outer(np.cos(u), np.sin(v))
            y = np.outer(np.sin(u), np.sin(v))
            z = np.outer(np.ones(np.size(u)), np.cos(v))
            ax.plot_surface(x, y, z, color='orange', alpha=0.3, rstride=5, cstride=5, linewidth=0, label='Unit Sphere')
        if special_points is not None:
            ax.scatter(special_points[:, 0], special_points[:, 1], special_points[:, 2], c='g', marker='x', s=100, label='Special Point')
            ax.quiver(0, 0, 0, special_points[:, 0], special_points[:, 1], special_points[:, 2], color='r', length=1, normalize=True, label='Vector from Origin')
        if orthogonal_vector is not None:
            a, b, c = orthogonal_vector
            xx, yy = np.meshgrid(np.linspace(-1, 1, 10), np.linspace(-1, 1, 10))
            zz = (-a * xx - b * yy) / c
            ax.plot_surface(xx, yy, zz, alpha=0.3, color='b', label='Orthogonal Plane')
            
        ax.set_xlabel('Dimension 1')
        ax.set_ylabel('Dimension 2')
        ax.set_zlabel('Dimension 3')
        ax.set_title(title_3d)
        ax.set_box_aspect([1, 1, 1])  # Equal aspect ratio
        # ax.legend()
    else:
        # 3D Subspace Plots for more than 3 dimensions
        for i in range(n_features - 2):
            fig = plt.figure(figsize=(12, 8))
            ax = fig.add_subplot(111, projection='3d')
            ax.scatter(data[:, i], data[:, i+1], data[:, i+2], c='b', marker='o', label='Data')
            if show_origin:
                ax.scatter(0, 0, 0, c='r', marker='x', s=100, label='Origin')  # Larger size for visibility
            if special_points is not None:
                ax.scatter(special_points[:, i], special_points[:, i+1], special_points[:, i+2], c='g', marker='x', s=100, label='Special Point')
            ax.set_xlabel(f'Dimension {i+1}')
            ax.set_ylabel(f'Dimension {i+2}')
            ax.set_zlabel(f'Dimension {i+3}')
            ax.set_title(f'3D Plot of Dimensions {i+1}, {i+2}, {i+3}')
            ax.set_box_aspect([1, 1, 1])  # Equal aspect ratio
            ax.legend()
    plt.show()
    return True
