import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.autograd import Variable
import torch.optim as optim


def squared_l2_norm(x):
    flattened = x.view(x.unsqueeze(0).shape[0], -1)
    return (flattened ** 2).sum(1)


def l2_norm(x):
    return squared_l2_norm(x).sqrt()


def trades_loss(model,
                x_natural,
                y,
                acc_ref,
                optimizer,
                step_size=0.003,
                step_size_prime = 0.0105,
                epsilon=0.031,
                perturb_steps=10,
                perturb_steps_prime=2,
                beta=1.0,
                distance='l_inf'):
    # define KL-loss
    criterion_kl = nn.KLDivLoss(size_average=False)

    x_adv = x_natural.clone().detach()
    model.eval()
    batch_size = len(x_natural)
    
    with torch.enable_grad():
        natural_output = model(x_natural)
        natural_loss = F.softmax(natural_output, dim=1)

    sign = 2 * ((natural_output.max(1)[1] == y).float()) - 1
    pred_var = (torch.std(natural_loss, dim=1) * sign).detach()
    gamma = 0.75

    p = torch.clamp(1.0-acc_ref, 0, 1) # in [0, 1]
    th = F.relu(torch.quantile(pred_var, p))

    pred_robust = (pred_var > th).int()
    pred_boundary = torch.logical_and(pred_var <= th, pred_var >= 0).int()

    r_size = pred_robust.sum()
    b_size = pred_boundary.sum()

    robust_index = torch.nonzero(pred_robust)[:, 0]
    boundary_index = torch.nonzero(pred_boundary)[:, 0]

    index = torch.cat((boundary_index, robust_index), 0)
    
    x1, y1 = x_natural[index, :, :, :], y[index]
    natural_loss1 = natural_loss[index]
    
    a1 = torch.cat((torch.ones_like(boundary_index) * step_size, torch.ones_like(robust_index) * step_size_prime), 0)
    a1 = a1.reshape((a1.shape[0], 1, 1, 1))
    # generate adversarial example
    x_adv1 = x1.detach() + 0.001 * torch.randn(x1.shape).cuda().detach()

    # rescale the loss as the actual batch size shrinks
    v = 20
    s1 = torch.sqrt(batch_size / torch.clamp(r_size + b_size + v, v, batch_size))
    s2 = torch.sqrt(batch_size / torch.clamp(b_size + v, v, batch_size))
    
    for _ in range(perturb_steps_prime):
        x_adv1.requires_grad_()
        with torch.enable_grad():
            output = model(x_adv1)
            loss_kl = s1 * criterion_kl(F.log_softmax(output, dim=1), natural_loss1)
        grad = torch.autograd.grad(loss_kl, [x_adv1])[0]
        x_adv1 = x_adv1.detach() + a1 * torch.sign(grad.detach())
        x_adv1 = torch.min(torch.max(x_adv1, x1 - epsilon), x1 + epsilon)
        x_adv1 = torch.clamp(x_adv1, 0.0, 1.0)

    if b_size > 0:
        x_adv2, x_adv1 = x_adv1[0:b_size].detach(), x_adv1[b_size:].detach()
        natural_loss2 = natural_loss1[0:b_size]
        x2, y2 = x1[0:b_size], y1[0:b_size]
        a2 = torch.ones_like(y2) * step_size
        a2 = a2.reshape(a2.shape[0], 1, 1, 1)
        for _ in range(perturb_steps-perturb_steps_prime):
            x_adv2.requires_grad_()
            with torch.enable_grad():
                output = model(x_adv2)
                loss_kl =s2 * criterion_kl(F.log_softmax(output, dim=1), natural_loss2)
            grad = torch.autograd.grad(loss_kl, [x_adv2])[0]
            x_adv2 = x_adv2.detach() + a2 * torch.sign(grad.detach())
            x_adv2 = torch.min(torch.max(x_adv2, x2 - epsilon), x2 + epsilon)
            x_adv2 = torch.clamp(x_adv2, 0.0, 1.0)

        x_adv[robust_index] = x_adv1
        x_adv[boundary_index] = x_adv2
    else:
        x_adv[robust_index] = x_adv1

    model.train()
    x_adv.requires_grad_(False)
    x_natural.requires_grad_(False)
    # zero gradient
    optimizer.zero_grad()
    output_adv = model(x_adv)
    batch_acc = gamma * (output_adv.max(1)[1] == y).detach().float().mean()
    # calculate robust loss
    logits = model(x_natural)
    loss_natural = F.cross_entropy(logits, y)
    loss_robust = (1.0 / batch_size) * criterion_kl(F.log_softmax(output_adv, dim=1), F.softmax(model(x_natural), dim=1))
    loss = loss_natural + beta * loss_robust
    return loss, batch_acc
