from datasets import load_dataset
from typing import List
import tyro
from tqdm import tqdm
import torch
import h5py
import codecs

# Assume the utility functions are defined elsewhere or imported correctly
from utils.data_utils import render_text, shuffle_text
from utils.model_init import init_subject_model
from utils.embeddings import get_image_embedding
from utils.convert_raw_to_embeddings import create_output_filename


def rot_k(text, k):
    result = ""
    for char in text:
        if char.isalpha():
            ascii_offset = 65 if char.isupper() else 97
            rotated = chr((ord(char) - ascii_offset + k) % 26 + ascii_offset)
            result += rotated
        else:
            result += char
    return result


def main(
    model_name: str = "openai/clip-vit-large-patch14",
    embedding_size: int = 768,
    font_path: str = "fonts/Roboto-Regular.ttf",
    jumble: bool = False,
    rot13: bool = False,
    rot9: bool = False,
):
    dataset = load_dataset("sst2")
    model_dict = init_subject_model(model_name, "clip", device="cuda")

    # file_name = create_output_filename('sst2', '.', model_name)
    font_name = font_path.split("/")[-1].split(".")[0]
    file_name = f"{model_name}_image_embeddings_{font_name}_{'jumbled' if jumble else ''}_sst2_{'rot13' if rot13 else ''}_{'rot9' if rot9 else ''}.h5".replace(
        "/", "_"
    )
    print(file_name)
    with h5py.File(file_name, "a") as h5f:
        for split in ["train", "validation", "test"]:
            embeddings_key = f"{split}/embeddings"

            if embeddings_key not in h5f:
                embeddings_dset = h5f.create_dataset(
                    embeddings_key,
                    shape=(0, embedding_size),
                    maxshape=(None, embedding_size),
                    dtype="float32",
                )
            else:
                embeddings_dset = h5f[embeddings_key]

            for idx, example in tqdm(
                enumerate(dataset[split]), total=len(dataset[split])
            ):
                text = example["sentence"]

                # print(text, flush=True)

                if jumble:
                    text = shuffle_text(text)
                elif rot13:
                    text = codecs.encode(text, "rot_13")
                elif rot9:
                    text = rot_k(text, k=9)

                # print(text, flush=True)

                image, ok = render_text(text, font_path=font_path)
                if not ok:
                    print("OVERFLOWS", text, flush=True)
                    # continue  # Now correctly skips to the next iteration

                image_embedding = get_image_embedding(model_dict, image)
                # print(image_embedding.shape)

                embeddings_dset.resize(
                    (embeddings_dset.shape[0] + 1, embeddings_dset.shape[1])
                )
                embeddings_dset[-1] = image_embedding.cpu().numpy()


if __name__ == "__main__":
    tyro.cli(main)
