from datasets import load_dataset, DatasetDict
from typing import List
import tyro
from tqdm import tqdm
import torch
import h5py
import codecs

# Import open_clip instead of Huggingface's transformers
import open_clip

# Assume the utility functions are defined elsewhere or imported correctly
from utils.data_utils import render_text, shuffle_text
from utils.embeddings import get_image_embedding


def init_subject_model_open_clip(model_name: str, pretrained: str = "openai"):
    # model, _, preprocess = open_clip.create_model_and_transforms('xlm-roberta-base-ViT-B-32', pretrained='laion5b_s13b_b90k')
    # tokenizer = open_clip.get_tokenizer('xlm-roberta-base-ViT-B-32')
    model, _, preprocess = open_clip.create_model_and_transforms(
        "ViT-bigG-14", pretrained="laion2b_s39b_b160k"
    )
    tokenizer = open_clip.get_tokenizer("ViT-bigG-14")
    return {"model": model, "preprocess": preprocess, "tokenizer": tokenizer}


def create_output_filename(dataset_name: str, directory: str, model_name: str) -> str:
    # Your logic to create a filename
    return f"{directory}/{model_name.replace('/', '_')}_embeddings.h5"


def rot_k(text, k):
    result = ""
    for char in text:
        if char.isalpha():
            ascii_offset = 65 if char.isupper() else 97
            rotated = chr((ord(char) - ascii_offset + k) % 26 + ascii_offset)
            result += rotated
        else:
            result += char
    return result


def main(
    model_name: str = "laion2b_s39b_b160k",
    embedding_size: int = 1280,  # Adjust embedding size according to the new model
    font_path: str = "fonts/Roboto-Regular.ttf",
    jumble: bool = False,
    rot13: bool = False,
    rot9: bool = False,
):
    # dataset = load_dataset("sst2")
    dataset = load_dataset("mattymchen/mr")
    print(dataset)

    model_dict = init_subject_model_open_clip(model_name)
    model_dict["model"] = model_dict["model"].to("cuda")

    font_name = font_path.split("/")[-1].split(".")[0]
    file_name = f"mr_{model_name}_image_embeddings_{font_name}_{'jumbled' if jumble else ''}_imdb_{'rot13' if rot13 else ''}_{'rot9' if rot9 else ''}.h5".replace(
        "/", "_"
    )
    print(file_name)

    with h5py.File(file_name, "a") as h5f:
        for split in ["test"]:
            embeddings_key = f"{split}/embeddings"

            if embeddings_key not in h5f:
                embeddings_dset = h5f.create_dataset(
                    embeddings_key,
                    shape=(0, embedding_size),
                    maxshape=(None, embedding_size),
                    dtype="float32",
                )
            else:
                embeddings_dset = h5f[embeddings_key]

            for idx, example in tqdm(
                enumerate(dataset[split]), total=len(dataset[split])
            ):
                text = example["text"]

                if jumble:
                    text = shuffle_text(text)
                elif rot13:
                    text = codecs.encode(text, "rot_13")
                elif rot9:
                    text = rot_k(text, k=9)

                image, ok = render_text(text, font_path=font_path)
                if not ok:
                    print("OVERFLOWS", flush=True)
                    continue  # Now correctly skips to the next iteration

                image_embedding = get_image_embedding(model_dict, image)

                embeddings_dset.resize(
                    (embeddings_dset.shape[0] + 1, embeddings_dset.shape[1])
                )
                embeddings_dset[-1] = image_embedding.cpu().numpy()


if __name__ == "__main__":
    tyro.cli(main)
