from modules.agents import REGISTRY as agent_REGISTRY
from components.action_selectors import REGISTRY as action_REGISTRY
import torch as th
import torch.nn.functional as F
from modules.trajectory_encoder.transformer_encoder import Transformer

# This multi-agent controller shares parameters between agents
class BasicMAC:
    def __init__(self, scheme, groups, args):
        self.n_agents = args.n_agents
        self.args = args
        input_shape = self._get_input_shape(scheme)
        self._build_agents(input_shape)
        self.agent_output_type = args.agent_output_type

        self.action_selector = action_REGISTRY[args.action_selector](args)

        if vars(self.args).get("defense_loaded", False):
            self.defense_classifier = Net(scheme["obs"]["vshape"], args.defense_module_hidden_size, 2).to(args.device)
            self.defense_reconstructor = Transformer(scheme["obs"]["vshape"] + 3, args.enc_emb, args.enc_heads, args.enc_depth, scheme["obs"]["vshape"]).to(args.device)

        self.hidden_states = None

    def select_actions(self, ep_batch, t_ep, t_env, bs=slice(None), test_mode=False):
        # Only select actions for the selected batch elements in bs
        discrete_action = None
        if vars(self.args).get("defense_loaded", False):
            mask_allies = th.Tensor.float(ep_batch['obs_origin'][bs, t_ep, :][:, :, 34:54:5] == 1)
        if vars(self.args).get("random_perturbation", False):
            discrete_action = (th.randint(
                0, self.args.n_agents + 1, (ep_batch.batch_size, 1)).to(ep_batch.device))[bs]
            # parameter_action = ((th.rand((ep_batch.batch_size, ep_batch['obs'].shape[-1])).to(
            #     ep_batch.device) - 0.5) * 2 * self.args.perturbation_range)[bs]
            parameter_action = (th.rand((ep_batch.batch_size, ep_batch['obs'].shape[-1])).to(
                ep_batch.device) - 0.5) * 2
            parameter_action = (parameter_action / th.linalg.norm(parameter_action, ord=1, dim=1, keepdim=True).repeat(1, parameter_action.size()[1]) * self.args.perturbation_range)[bs]
            # NOTE: broadcast
            parameter_action[discrete_action.view(-1) >= self.args.n_agents] = 0
            # NOTE: perturba_agent_id > self.groups["agents"] means no attack
            perturb_agent_id_clip = discrete_action.clamp(0, self.args.n_agents - 1)
            # parameter_action = parameter_action.clamp(
            #     -self.args.perturbation_range, self.args.perturbation_range)
            idx = perturb_agent_id_clip.repeat(
                1, parameter_action.size()[-1]).view(parameter_action.size()[0], 1, -1)
            val = parameter_action.view(parameter_action.size()[0], 1, -1)
            ep_batch['obs_perturbed'][bs, t_ep, :] = ep_batch['obs'][bs,
                                                        t_ep, :].scatter_(1, idx, val, reduce="add")
        if vars(self.args).get("defense_loaded", False):
            with th.no_grad():
                disturbed = th.argmax(self.defense_classifier(ep_batch['obs_perturbed'][bs, t_ep, :]), dim=-1).unsqueeze(2).repeat(1, 1, ep_batch['obs_perturbed'].shape[-1])
                reconstruction_input = []
                for num_agent in range(self.args.n_agents):
                    x = th.arange(self.args.n_agents)
                    reconstruction_input.append(th.cat([ep_batch['obs_perturbed'][bs, t_ep, :].clone().unsqueeze(1)[:, :, x != num_agent, :], ep_batch['obs_perturbed'][bs, t_ep, :][:, num_agent, [34, 35, 36, 39, 40, 41, 44, 45, 46, 49, 50, 51]].view(-1, 1, self.args.n_agents - 1, 3)], dim=-1))
                reconstruction_input = th.cat(reconstruction_input, dim=1).view(len(bs) * self.args.n_agents, self.args.n_agents - 1, -1)
                reconstructed_obs = th.mean(self.defense_reconstructor(reconstruction_input, mask_allies.view(len(bs) * self.args.n_agents, -1))[0], dim=1).view(len(bs), self.args.n_agents, -1)
                ep_batch['obs'][bs, t_ep, :] = th.where(disturbed == 0, ep_batch['obs_perturbed'][bs, t_ep, :], reconstructed_obs)
        elif vars(self.args).get("random_perturbation", False):
            ep_batch['obs'][bs, t_ep, :] = ep_batch['obs_perturbed'][bs, t_ep, :]
        avail_actions = ep_batch["avail_actions"][:, t_ep]
        agent_outputs = self.forward(ep_batch, t_ep, test_mode=test_mode)
        chosen_actions = self.action_selector.select_action(agent_outputs[bs], avail_actions[bs], t_env, test_mode=test_mode)
        if vars(self.args).get("defense_loaded", False):
            return chosen_actions, discrete_action
        else:
            return chosen_actions

    def forward(self, ep_batch, t, test_mode=False):
        agent_inputs = self._build_inputs(ep_batch, t)
        avail_actions = ep_batch["avail_actions"][:, t]
        self.hidden_states = self.hidden_states.to(ep_batch.device)
        self.agent = self.agent.to(ep_batch.device)
        agent_outs, self.hidden_states = self.agent(agent_inputs, self.hidden_states)

        # Softmax the agent outputs if they're policy logits
        if self.agent_output_type == "pi_logits":

            if getattr(self.args, "mask_before_softmax", True):
                # Make the logits for unavailable actions very negative to minimise their affect on the softmax
                reshaped_avail_actions = avail_actions.reshape(ep_batch.batch_size * self.n_agents, -1)
                agent_outs[reshaped_avail_actions == 0] = -1e10

            agent_outs = th.nn.functional.softmax(agent_outs, dim=-1)
            if not test_mode:
                # Epsilon floor
                epsilon_action_num = agent_outs.size(-1)
                if getattr(self.args, "mask_before_softmax", True):
                    # With probability epsilon, we will pick an available action uniformly
                    epsilon_action_num = reshaped_avail_actions.sum(dim=1, keepdim=True).float()

                agent_outs = ((1 - self.action_selector.epsilon) * agent_outs
                               + th.ones_like(agent_outs) * self.action_selector.epsilon/epsilon_action_num)

                if getattr(self.args, "mask_before_softmax", True):
                    # Zero out the unavailable actions
                    agent_outs[reshaped_avail_actions == 0] = 0.0

        return agent_outs.view(ep_batch.batch_size, self.n_agents, -1)

    def init_hidden(self, batch_size):
        self.hidden_states = self.agent.init_hidden().unsqueeze(0).expand(batch_size, self.n_agents, -1)  # bav

    def parameters(self):
        return self.agent.parameters()

    def load_state(self, other_mac):
        self.agent.load_state_dict(other_mac.agent.state_dict())

    def cuda(self):
        self.agent.cuda()

    def save_models(self, path):
        print("Saving qmix models to {}".format(path))
        th.save(self.agent.state_dict(), "{}/agent.th".format(path))
        if vars(self.args).get("defense_loaded", False):
            th.save(self.defense_classifier.state_dict(), "{}/defense_classifier.th".format(path))
            th.save(self.defense_reconstructor.state_dict(), "{}/defense_reconstructor.th".format(path))

    def load_models(self, path):
        self.agent.load_state_dict(th.load("{}/agent.th".format(path), map_location=lambda storage, loc: storage))

    def load_defense_models(self, path):
        self.defense_classifier.load_state_dict(th.load("{}/defense_classifier.th".format(path), map_location=lambda storage, loc: storage))
        self.defense_reconstructor.load_state_dict(th.load("{}/defense_reconstructor.th".format(path), map_location=lambda storage, loc: storage))

    def _build_agents(self, input_shape):
        self.agent = agent_REGISTRY[self.args.agent](input_shape, self.args)

    def _build_inputs(self, batch, t):
        # Assumes homogenous agents with flat observations.
        # Other MACs might want to e.g. delegate building inputs to each agent
        bs = batch.batch_size
        inputs = []
        inputs.append(batch["obs"][:, t])  # b1av
        if self.args.obs_last_action:
            if t == 0:
                inputs.append(th.zeros_like(batch["actions_onehot"][:, t]))
            else:
                inputs.append(batch["actions_onehot"][:, t-1])
        if self.args.obs_agent_id:
            inputs.append(th.eye(self.n_agents, device=batch.device).unsqueeze(0).expand(bs, -1, -1))

        inputs = th.cat([x.reshape(bs*self.n_agents, -1) for x in inputs], dim=1)
        return inputs

    def _get_input_shape(self, scheme):
        input_shape = scheme["obs"]["vshape"]
        if self.args.obs_last_action:
            input_shape += scheme["actions_onehot"]["vshape"][0]
        if self.args.obs_agent_id:
            input_shape += self.n_agents

        return input_shape

class Net(th.nn.Module):
    def __init__(self,input_size, hidden_size, output_size):
        super(Net,self).__init__()
        self.fc1 = th.nn.Linear(input_size, hidden_size)
        self.fc2 = th.nn.Linear(hidden_size, output_size)

    def forward(self, input):
        x = F.relu(self.fc1(input))
        x = self.fc2(x)
        return x