from collections import defaultdict
import os
import json
import logging
import numpy as np
from tensorboardX.writer import SummaryWriter
import wandb
import time

class Logger:
    def __init__(self, console_logger, result_dir):
        self.console_logger = console_logger

        self.use_tb = False
        self.use_sacred = False
        self.use_hdf = False
        self.use_wandb = False

        self.result_dir = result_dir

        self.stats = defaultdict(lambda: [])

    def setup_tb(self, directory_name):
        # Import here so it doesn't have to be installed if you don't use it
        from tensorboardX import SummaryWriter
        self.writer = SummaryWriter(logdir=directory_name)
        self.use_tb = True

    def setup_wandb(self, directory_name, args):
        os.makedirs(os.path.join(directory_name, 'wandb'), exist_ok=True)
        if args.run_file == "run_online":
            project_name = args.name
        else:
            project_name = args.name + '_' + args.attacker + '_' + args.defender
        name = time.strftime("%m-%d-%H:%M:%S") + '_seed' + str(args.seed)
        self.use_wandb = True
        config = wandb.config
        args_dict = vars(args)
        wandb.init(name=name, 
                   config=args_dict, 
                   dir=directory_name, 
                   project=project_name, 
                   group=args.env + '_' + args.env_args['map_name'], 
                   job_type=args.remark, 
                   entity="admaster")

    def setup_sacred(self, sacred_run_dict):
        self.sacred_info = sacred_run_dict.info
        self.use_sacred = True

    def log_stat(self, key, value, t, to_sacred=True):
        value = float(value)
        self.stats[key].append((t, value))

        if self.use_tb:
            self.writer.add_scalar(key, value, t)
        if self.use_wandb:
            wandb.log({key: value}, step=t)

        if self.use_sacred and to_sacred:
            if key in self.sacred_info:
                self.sacred_info["{}_T".format(key)].append(t)
                self.sacred_info[key].append(value)
            else:
                self.sacred_info["{}_T".format(key)] = [t]
                self.sacred_info[key] = [value]

    def log_histogram(self, key, value, t):
        self.writer.add_histogram(key, value, t)

    def log_embedding(self, key, value):
        self.writer.add_embedding(value, tag=key)

    def print_recent_stats(self):
        json.dump(self.stats, open(os.path.join(self.result_dir, 'metrics.json'), 'w'))
        log_str = "Recent Stats | t_env: {:>10}\t Episode: {:>10}\n".format(self.stats["episode"][-1][0], self.stats["episode"][-1][1])
        i = 0
        for (k, v) in sorted(self.stats.items()):
            if k == "episode":
                continue
            i += 1
            item = "{:.4f}".format(float(self.stats[k][-1][1]))
            log_str += "{:<25}{:>8}".format(k + ":", item)
            log_str += "\n" if i % 4 == 0 else "\t"

        self.console_logger.info(log_str)


# set up a custom logger
def get_logger():
    logger = logging.getLogger()
    logger.handlers = []
    ch = logging.StreamHandler()
    formatter = logging.Formatter('[%(levelname)s %(asctime)s] %(name)s %(message)s', '%H:%M:%S')
    ch.setFormatter(formatter)
    logger.addHandler(ch)
    logger.setLevel('DEBUG')

    return logger

