import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.vdn import VDNMixer
from modules.mixers.qmix import QMixer
import torch as th
from torch.optim import RMSprop, Adam
import torch.nn.functional as F


class QLearner:
    def __init__(self, mac, scheme, logger, args):
        self.args = args
        self.mac = mac
        self.logger = logger

        self.params = list(mac.parameters())

        self.last_target_update_episode = 0

        self.mixer = None
        if args.mixer is not None:
            if args.mixer == "vdn":
                self.mixer = VDNMixer()
            elif args.mixer == "qmix":
                self.mixer = QMixer(args)
            else:
                raise ValueError("Mixer {} not recognised.".format(args.mixer))
            self.params += list(self.mixer.parameters())
            self.target_mixer = copy.deepcopy(self.mixer)
        if args.runner == "parallel":
            self.optimiser = Adam(params=self.params, lr=args.lr, eps=args.optim_eps)
        else:
            self.optimiser = RMSprop(params=self.params, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)

        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)

        self.log_stats_t = -self.args.learner_log_interval - 1
        self.train_defense_t = -self.args.learner_log_interval - 1
        if vars(self.args).get("defense_loaded", False):
            self.defense_classifier_optimizer = Adam(params=self.mac.defense_classifier.parameters(), lr=args.lr, eps=args.optim_eps)
            self.defense_classifier_criterion = th.nn.CrossEntropyLoss()
            self.defense_reconstructor_optimizer = Adam(params=self.mac.defense_reconstructor.parameters(), lr=args.lr, eps=args.optim_eps)
            self.defense_reconstructor_criterion = th.nn.MSELoss()

    def train(self, batch: EpisodeBatch, t_env: int, episode_num: int):
        # Get the relevant quantities
        rewards = batch["reward"][:, :-1]
        actions = batch["actions"][:, :-1]
        terminated = batch["terminated"][:, :-1].float()
        mask = batch["filled"][:, :-1].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]

        # Calculate estimated Q-Values
        mac_out = []
        self.mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            agent_outs = self.mac.forward(batch, t=t)
            mac_out.append(agent_outs)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time

        # Pick the Q-Values for the actions taken by each agent
        chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim

        # Calculate the Q-Values necessary for the target
        target_mac_out = []
        self.target_mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            target_agent_outs = self.target_mac.forward(batch, t=t)
            target_mac_out.append(target_agent_outs)

        # We don't need the first timesteps Q-Value estimate for calculating targets
        target_mac_out = th.stack(target_mac_out[1:], dim=1)  # Concat across time

        # Mask out unavailable actions
        target_mac_out[avail_actions[:, 1:] == 0] = -9999999

        # Max over target Q-Values
        if self.args.double_q:
            # Get actions that maximise live Q (for double q-learning)
            mac_out_detach = mac_out.clone().detach()
            mac_out_detach[avail_actions == 0] = -9999999
            cur_max_actions = mac_out_detach[:, 1:].max(dim=3, keepdim=True)[1]
            target_max_qvals = th.gather(target_mac_out, 3, cur_max_actions).squeeze(3)
        else:
            target_max_qvals = target_mac_out.max(dim=3)[0]

        # Mix
        if self.mixer is not None:
            chosen_action_qvals = self.mixer(chosen_action_qvals, batch["state"][:, :-1])
            target_max_qvals = self.target_mixer(target_max_qvals, batch["state"][:, 1:])

        # Calculate 1-step Q-Learning targets
        targets = rewards + self.args.gamma * (1 - terminated) * target_max_qvals

        # Td-error
        td_error = (chosen_action_qvals - targets.detach())

        mask = mask.expand_as(td_error)

        # 0-out the targets that came from padded data
        masked_td_error = td_error * mask

        # Normal L2 loss, take mean over actual data
        loss = (masked_td_error ** 2).sum() / mask.sum()

        # Optimise
        self.optimiser.zero_grad()
        loss.backward()
        grad_norm = th.nn.utils.clip_grad_norm_(self.params, self.args.grad_norm_clip)
        self.optimiser.step()

        if vars(self.args).get("defense_loaded", False):
            if t_env - self.train_defense_t > self.args.log_interval:
                # Train classfier
                eplens = th.argmax(batch['terminated'], dim=1)
                clip_slices = [slice(0, eplen.item() + 1) for eplen in eplens]
                obs_x = []
                obs_y = []
                for idx_slice, clip_slice in enumerate(clip_slices):
                    for num_agent in range(self.args.n_agents):
                        obs_x.append(batch['obs_perturbed'][idx_slice, clip_slice, num_agent, :])
                        obs_y.append(batch['ad_discrete_actions'][idx_slice, clip_slice, 0] == num_agent)
                obs_x = th.cat(obs_x, dim=0)
                obs_y = th.cat(obs_y, dim=0)
                for start_index in range(0, obs_x.shape[0], self.args.batch_size * 8):
                    end_index = min(start_index + self.args.batch_size, obs_x.shape[0])
                    batch_x = obs_x[start_index:end_index]
                    batch_y = obs_y[start_index:end_index]
                    predict_y = self.mac.defense_classifier(batch_x)
                    classifier_loss = self.defense_classifier_criterion(predict_y, batch_y.to(th.int64))
                    self.defense_classifier_optimizer.zero_grad()
                    classifier_loss.backward()
                    classifier_grad_nrom = th.nn.utils.clip_grad_norm_(self.mac.defense_classifier.parameters(), self.args.grad_norm_clip)
                    self.defense_classifier_optimizer.step()

                # Train reconstructor
                obs_x = []
                mask_x = []
                obs_y = []
                for idx_slice, clip_slice in enumerate(clip_slices):
                    reconstruction_input = []
                    for num_agent in range(self.args.n_agents):
                        x = th.arange(self.args.n_agents)
                        reconstruction_input.append(th.cat([batch['obs_origin'][idx_slice, clip_slice].clone().unsqueeze(1)[:, :, x != num_agent, :], batch['obs_origin'][idx_slice, clip_slice][:, num_agent, [34, 35, 36, 39, 40, 41, 44, 45, 46, 49, 50, 51]].view(-1, 1, self.args.n_agents - 1, 3)], dim=-1))
                    reconstruction_input = th.cat(reconstruction_input, dim=1)
                    obs_x.append(reconstruction_input.view(-1, self.args.n_agents - 1, reconstruction_input.size()[-1]))
                    mask_x.append(th.Tensor.float(batch['obs_origin'][idx_slice, clip_slice, :][:, :, 34:54:5] == 1).view(-1, self.args.n_agents - 1))
                    obs_y.append(batch['obs_origin'][idx_slice, clip_slice].clone().view(-1, batch['obs_origin'][idx_slice, clip_slice].size()[-1]))
                obs_x = th.cat(obs_x, dim=0)
                mask_x = th.cat(mask_x, dim=0)
                obs_y = th.cat(obs_y, dim=0)
                for start_index in range(0, obs_x.shape[0], self.args.batch_size * 8):
                    end_index = min(start_index + self.args.batch_size, obs_x.shape[0])
                    batch_x = obs_x[start_index:end_index]
                    batch_mask_x = mask_x[start_index:end_index]
                    batch_y = obs_y[start_index:end_index]
                    predict_y = th.mean(self.mac.defense_reconstructor(batch_x, batch_mask_x)[0], dim=1)
                    reconstructor_loss = self.defense_reconstructor_criterion(predict_y, batch_y)
                    self.defense_reconstructor_optimizer.zero_grad()
                    reconstructor_loss.backward()
                    reconstructor_grad_nrom = th.nn.utils.clip_grad_norm_(self.mac.defense_reconstructor.parameters(), self.args.grad_norm_clip)
                    self.defense_reconstructor_optimizer.step()
                self.train_defense_t = t_env
                self.logger.log_stat("classifier_loss", classifier_loss.item(), t_env)
                self.logger.log_stat("classifier_grad_norm", classifier_grad_nrom, t_env)
                self.logger.log_stat("reconstructor_loss", reconstructor_loss.item(), t_env)
                self.logger.log_stat("reconstructor_grad_norm", reconstructor_grad_nrom, t_env)

        if (episode_num - self.last_target_update_episode) / self.args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = episode_num

        if t_env - self.log_stats_t >= self.args.learner_log_interval:
            self.logger.log_stat("loss", loss.item(), t_env)
            self.logger.log_stat("grad_norm", grad_norm, t_env)
                
            mask_elems = mask.sum().item()
            self.logger.log_stat("td_error_abs", (masked_td_error.abs().sum().item()/mask_elems), t_env)
            self.logger.log_stat("q_taken_mean", (chosen_action_qvals * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.logger.log_stat("target_mean", (targets * mask).sum().item()/(mask_elems * self.args.n_agents), t_env)
            self.log_stats_t = t_env

    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())
        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()

    def save_models(self, path):
        self.mac.save_models(path)
        if self.mixer is not None:
            th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
        th.save(self.optimiser.state_dict(), "{}/opt.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        # Not quite right but I don't want to save target networks
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
        if not vars(self.args).get("concurrent_train", False):
            self.optimiser.load_state_dict(th.load("{}/opt.th".format(path), map_location=lambda storage, loc: storage))
